/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.context;

import com.google.common.annotations.VisibleForTesting;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.util.ArrayList;
import org.apache.cassandra.db.ClockAndCount;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.CounterId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CounterContext {
    private static final int HEADER_SIZE_LENGTH = TypeSizes.sizeof((short)Short.MAX_VALUE);
    private static final int HEADER_ELT_LENGTH = TypeSizes.sizeof((short)Short.MAX_VALUE);
    private static final int CLOCK_LENGTH = TypeSizes.sizeof(Long.MAX_VALUE);
    private static final int COUNT_LENGTH = TypeSizes.sizeof(Long.MAX_VALUE);
    private static final int STEP_LENGTH = 16 + CLOCK_LENGTH + COUNT_LENGTH;
    private static final Logger logger = LoggerFactory.getLogger(CounterContext.class);

    public static CounterContext instance() {
        return LazyHolder.counterContext;
    }

    public ByteBuffer createGlobal(CounterId id, long clock, long count) {
        ContextState state = ContextState.allocate(1, 0, 0);
        state.writeGlobal(id, clock, count);
        return state.context;
    }

    public ByteBuffer createLocal(long count) {
        ContextState state = ContextState.allocate(0, 1, 0);
        state.writeLocal(CounterId.getLocalId(), 1L, count);
        return state.context;
    }

    public ByteBuffer createRemote(CounterId id, long clock, long count) {
        ContextState state = ContextState.allocate(0, 0, 1);
        state.writeRemote(id, clock, count);
        return state.context;
    }

    private static int headerLength(ByteBuffer context) {
        return HEADER_SIZE_LENGTH + Math.abs(context.getShort(context.position())) * HEADER_ELT_LENGTH;
    }

    private static int compareId(ByteBuffer bb1, int pos1, ByteBuffer bb2, int pos2) {
        return ByteBufferUtil.compareSubArrays(bb1, pos1, bb2, pos2, 16);
    }

    public Relationship diff(ByteBuffer left, ByteBuffer right) {
        Relationship relationship = Relationship.EQUAL;
        ContextState leftState = ContextState.wrap(left);
        ContextState rightState = ContextState.wrap(right);
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int compareId = leftState.compareIdTo(rightState);
            if (compareId == 0) {
                long leftClock = leftState.getClock();
                long rightClock = rightState.getClock();
                long leftCount = leftState.getCount();
                long rightCount = rightState.getCount();
                leftState.moveToNext();
                rightState.moveToNext();
                if (leftClock == rightClock) {
                    if (leftCount == rightCount) continue;
                    return Relationship.DISJOINT;
                }
                if (leftClock >= 0L && rightClock > 0L && leftClock > rightClock || leftClock < 0L && (rightClock > 0L || leftClock < rightClock)) {
                    if (relationship == Relationship.EQUAL) {
                        relationship = Relationship.GREATER_THAN;
                        continue;
                    }
                    if (relationship != Relationship.LESS_THAN) continue;
                    return Relationship.DISJOINT;
                }
                if (relationship == Relationship.EQUAL) {
                    relationship = Relationship.LESS_THAN;
                    continue;
                }
                if (relationship != Relationship.GREATER_THAN) continue;
                return Relationship.DISJOINT;
            }
            if (compareId > 0) {
                rightState.moveToNext();
                if (relationship == Relationship.EQUAL) {
                    relationship = Relationship.LESS_THAN;
                    continue;
                }
                if (relationship != Relationship.GREATER_THAN) continue;
                return Relationship.DISJOINT;
            }
            leftState.moveToNext();
            if (relationship == Relationship.EQUAL) {
                relationship = Relationship.GREATER_THAN;
                continue;
            }
            if (relationship != Relationship.LESS_THAN) continue;
            return Relationship.DISJOINT;
        }
        if (leftState.hasRemaining()) {
            if (relationship == Relationship.EQUAL) {
                return Relationship.GREATER_THAN;
            }
            if (relationship == Relationship.LESS_THAN) {
                return Relationship.DISJOINT;
            }
        }
        if (rightState.hasRemaining()) {
            if (relationship == Relationship.EQUAL) {
                return Relationship.LESS_THAN;
            }
            if (relationship == Relationship.GREATER_THAN) {
                return Relationship.DISJOINT;
            }
        }
        return relationship;
    }

    public ByteBuffer merge(ByteBuffer left, ByteBuffer right) {
        boolean leftIsSuperSet = true;
        boolean rightIsSuperSet = true;
        int globalCount = 0;
        int localCount = 0;
        int remoteCount = 0;
        ContextState leftState = ContextState.wrap(left);
        ContextState rightState = ContextState.wrap(right);
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int cmp = leftState.compareIdTo(rightState);
            if (cmp == 0) {
                Relationship rel = this.compare(leftState, rightState);
                if (rel == Relationship.GREATER_THAN) {
                    rightIsSuperSet = false;
                } else if (rel == Relationship.LESS_THAN) {
                    leftIsSuperSet = false;
                } else if (rel == Relationship.DISJOINT) {
                    rightIsSuperSet = false;
                    leftIsSuperSet = false;
                }
                if (leftState.isGlobal() || rightState.isGlobal()) {
                    ++globalCount;
                } else if (leftState.isLocal() || rightState.isLocal()) {
                    ++localCount;
                } else {
                    ++remoteCount;
                }
                leftState.moveToNext();
                rightState.moveToNext();
                continue;
            }
            if (cmp > 0) {
                leftIsSuperSet = false;
                if (rightState.isGlobal()) {
                    ++globalCount;
                } else if (rightState.isLocal()) {
                    ++localCount;
                } else {
                    ++remoteCount;
                }
                rightState.moveToNext();
                continue;
            }
            rightIsSuperSet = false;
            if (leftState.isGlobal()) {
                ++globalCount;
            } else if (leftState.isLocal()) {
                ++localCount;
            } else {
                ++remoteCount;
            }
            leftState.moveToNext();
        }
        if (leftState.hasRemaining()) {
            rightIsSuperSet = false;
        } else if (rightState.hasRemaining()) {
            leftIsSuperSet = false;
        }
        if (leftIsSuperSet) {
            return left;
        }
        if (rightIsSuperSet) {
            return right;
        }
        while (leftState.hasRemaining()) {
            if (leftState.isGlobal()) {
                ++globalCount;
            } else if (leftState.isLocal()) {
                ++localCount;
            } else {
                ++remoteCount;
            }
            leftState.moveToNext();
        }
        while (rightState.hasRemaining()) {
            if (rightState.isGlobal()) {
                ++globalCount;
            } else if (rightState.isLocal()) {
                ++localCount;
            } else {
                ++remoteCount;
            }
            rightState.moveToNext();
        }
        leftState.reset();
        rightState.reset();
        return this.merge(ContextState.allocate(globalCount, localCount, remoteCount), leftState, rightState);
    }

    private ByteBuffer merge(ContextState mergedState, ContextState leftState, ContextState rightState) {
        while (leftState.hasRemaining() && rightState.hasRemaining()) {
            int cmp = leftState.compareIdTo(rightState);
            if (cmp == 0) {
                Relationship rel = this.compare(leftState, rightState);
                if (rel == Relationship.DISJOINT) {
                    mergedState.writeLocal(leftState.getCounterId(), leftState.getClock() + rightState.getClock(), leftState.getCount() + rightState.getCount());
                } else if (rel == Relationship.GREATER_THAN) {
                    leftState.copyTo(mergedState);
                } else {
                    rightState.copyTo(mergedState);
                }
                rightState.moveToNext();
                leftState.moveToNext();
                continue;
            }
            if (cmp > 0) {
                rightState.copyTo(mergedState);
                rightState.moveToNext();
                continue;
            }
            leftState.copyTo(mergedState);
            leftState.moveToNext();
        }
        while (leftState.hasRemaining()) {
            leftState.copyTo(mergedState);
            leftState.moveToNext();
        }
        while (rightState.hasRemaining()) {
            rightState.copyTo(mergedState);
            rightState.moveToNext();
        }
        return mergedState.context;
    }

    private Relationship compare(ContextState leftState, ContextState rightState) {
        long leftClock = leftState.getClock();
        long leftCount = leftState.getCount();
        long rightClock = rightState.getClock();
        long rightCount = rightState.getCount();
        if (leftState.isGlobal() || rightState.isGlobal()) {
            if (leftState.isGlobal() && rightState.isGlobal()) {
                if (leftClock == rightClock) {
                    if (leftCount != rightCount && CompactionManager.isCompactionManager.get().booleanValue()) {
                        logger.warn("invalid global counter shard detected; ({}, {}, {}) and ({}, {}, {}) differ only in count; will pick highest to self-heal on compaction", new Object[]{leftState.getCounterId(), leftClock, leftCount, rightState.getCounterId(), rightClock, rightCount});
                    }
                    if (leftCount > rightCount) {
                        return Relationship.GREATER_THAN;
                    }
                    if (leftCount == rightCount) {
                        return Relationship.EQUAL;
                    }
                    return Relationship.LESS_THAN;
                }
                return leftClock > rightClock ? Relationship.GREATER_THAN : Relationship.LESS_THAN;
            }
            return leftState.isGlobal() ? Relationship.GREATER_THAN : Relationship.LESS_THAN;
        }
        if (leftState.isLocal() || rightState.isLocal()) {
            if (leftState.isLocal() && rightState.isLocal()) {
                return Relationship.DISJOINT;
            }
            return leftState.isLocal() ? Relationship.GREATER_THAN : Relationship.LESS_THAN;
        }
        if (leftClock == rightClock) {
            if (leftCount != rightCount && CompactionManager.isCompactionManager.get().booleanValue()) {
                logger.warn("invalid remote counter shard detected; ({}, {}, {}) and ({}, {}, {}) differ only in count; will pick highest to self-heal on compaction", new Object[]{leftState.getCounterId(), leftClock, leftCount, rightState.getCounterId(), rightClock, rightCount});
            }
            if (leftCount > rightCount) {
                return Relationship.GREATER_THAN;
            }
            if (leftCount == rightCount) {
                return Relationship.EQUAL;
            }
            return Relationship.LESS_THAN;
        }
        if (leftClock >= 0L && rightClock > 0L && leftClock >= rightClock || leftClock < 0L && (rightClock > 0L || leftClock < rightClock)) {
            return Relationship.GREATER_THAN;
        }
        return Relationship.LESS_THAN;
    }

    public String toString(ByteBuffer context) {
        ContextState state = ContextState.wrap(context);
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        while (state.hasRemaining()) {
            if (state.getElementIndex() > 0) {
                sb.append(",");
            }
            sb.append("{");
            sb.append(state.getCounterId()).append(", ");
            sb.append(state.getClock()).append(", ");
            sb.append(state.getCount());
            sb.append("}");
            if (state.isGlobal()) {
                sb.append("$");
            } else if (state.isLocal()) {
                sb.append("*");
            }
            state.moveToNext();
        }
        sb.append("]");
        return sb.toString();
    }

    public long total(ByteBuffer context) {
        long total = 0L;
        for (int offset = context.position() + CounterContext.headerLength(context); offset < context.limit(); offset += STEP_LENGTH) {
            total += context.getLong(offset + 16 + CLOCK_LENGTH);
        }
        return total;
    }

    public boolean shouldClearLocal(ByteBuffer context) {
        return context.getShort(context.position()) < 0;
    }

    public boolean hasLegacyShards(ByteBuffer context) {
        int totalCount = (context.remaining() - CounterContext.headerLength(context)) / STEP_LENGTH;
        int localAndGlobalCount = Math.abs(context.getShort(context.position()));
        if (localAndGlobalCount < totalCount) {
            return true;
        }
        for (int i = 0; i < localAndGlobalCount; ++i) {
            if (context.getShort(context.position() + HEADER_SIZE_LENGTH + i * HEADER_ELT_LENGTH) < 0) continue;
            return true;
        }
        return false;
    }

    public ByteBuffer markLocalToBeCleared(ByteBuffer context) {
        int count = context.getShort(context.position());
        if (count <= 0) {
            return context;
        }
        boolean hasLocalShards = false;
        for (int i = 0; i < count; ++i) {
            if (context.getShort(context.position() + HEADER_SIZE_LENGTH + i * HEADER_ELT_LENGTH) < 0) continue;
            hasLocalShards = true;
            break;
        }
        if (!hasLocalShards) {
            return context;
        }
        ByteBuffer marked = ByteBuffer.allocate(context.remaining());
        marked.putShort(marked.position(), (short)(count * -1));
        ByteBufferUtil.arrayCopy(context, context.position() + HEADER_SIZE_LENGTH, marked, marked.position() + HEADER_SIZE_LENGTH, context.remaining() - HEADER_SIZE_LENGTH);
        return marked;
    }

    public ByteBuffer clearAllLocal(ByteBuffer context) {
        int count = Math.abs(context.getShort(context.position()));
        if (count == 0) {
            return context;
        }
        ArrayList<Short> globalShardIndexes = new ArrayList<Short>(count);
        for (int i = 0; i < count; ++i) {
            short elt = context.getShort(context.position() + HEADER_SIZE_LENGTH + i * HEADER_ELT_LENGTH);
            if (elt >= 0) continue;
            globalShardIndexes.add(elt);
        }
        if (count == globalShardIndexes.size()) {
            return context;
        }
        ByteBuffer cleared = ByteBuffer.allocate(context.remaining() - (count - globalShardIndexes.size()) * HEADER_ELT_LENGTH);
        cleared.putShort(cleared.position(), (short)globalShardIndexes.size());
        for (int i = 0; i < globalShardIndexes.size(); ++i) {
            cleared.putShort(cleared.position() + HEADER_SIZE_LENGTH + i * HEADER_ELT_LENGTH, (Short)globalShardIndexes.get(i));
        }
        int origHeaderLength = CounterContext.headerLength(context);
        ByteBufferUtil.arrayCopy(context, context.position() + origHeaderLength, cleared, cleared.position() + CounterContext.headerLength(cleared), context.remaining() - origHeaderLength);
        return cleared;
    }

    public void validateContext(ByteBuffer context) throws MarshalException {
        if ((context.remaining() - CounterContext.headerLength(context)) % STEP_LENGTH != 0) {
            throw new MarshalException("Invalid size for a counter context");
        }
    }

    public void updateDigest(MessageDigest message, ByteBuffer context) {
        ByteBuffer dup = context.duplicate();
        dup.position(context.position() + CounterContext.headerLength(context));
        message.update(dup);
    }

    public ClockAndCount getLocalClockAndCount(ByteBuffer context) {
        return this.getClockAndCountOf(context, CounterId.getLocalId());
    }

    @VisibleForTesting
    public ClockAndCount getClockAndCountOf(ByteBuffer context, CounterId id) {
        int position = this.findPositionOf(context, id);
        if (position == -1) {
            return ClockAndCount.BLANK;
        }
        long clock = context.getLong(position + 16);
        long count = context.getLong(position + 16 + CLOCK_LENGTH);
        return ClockAndCount.create(clock, count);
    }

    @VisibleForTesting
    public int findPositionOf(ByteBuffer context, CounterId id) {
        int headerLength = CounterContext.headerLength(context);
        int offset = context.position() + headerLength;
        int left = 0;
        int right = (context.remaining() - headerLength) / STEP_LENGTH - 1;
        while (right >= left) {
            int middle = (left + right) / 2;
            int cmp = CounterContext.compareId(context, offset + middle * STEP_LENGTH, id.bytes(), id.bytes().position());
            if (cmp == -1) {
                left = middle + 1;
                continue;
            }
            if (cmp == 0) {
                return offset + middle * STEP_LENGTH;
            }
            right = middle - 1;
        }
        return -1;
    }

    public static class ContextState {
        public final ByteBuffer context;
        public final int headerLength;
        private int headerOffset;
        private int bodyOffset;
        private boolean currentIsGlobal;
        private boolean currentIsLocal;

        private ContextState(ByteBuffer context) {
            this.context = context;
            this.headerLength = this.bodyOffset = CounterContext.headerLength(context);
            this.headerOffset = HEADER_SIZE_LENGTH;
            this.updateIsGlobalOrLocal();
        }

        public static ContextState wrap(ByteBuffer context) {
            return new ContextState(context);
        }

        public static ContextState allocate(int globalCount, int localCount, int remoteCount) {
            int headerLength = HEADER_SIZE_LENGTH + (globalCount + localCount) * HEADER_ELT_LENGTH;
            int bodyLength = (globalCount + localCount + remoteCount) * STEP_LENGTH;
            ByteBuffer buffer = ByteBuffer.allocate(headerLength + bodyLength);
            buffer.putShort(buffer.position(), (short)(globalCount + localCount));
            return ContextState.wrap(buffer);
        }

        public boolean isGlobal() {
            return this.currentIsGlobal;
        }

        public boolean isLocal() {
            return this.currentIsLocal;
        }

        public boolean isRemote() {
            return !this.currentIsGlobal && !this.currentIsLocal;
        }

        private void updateIsGlobalOrLocal() {
            if (this.headerOffset >= this.headerLength) {
                this.currentIsLocal = false;
                this.currentIsGlobal = false;
            } else {
                short headerElt = this.context.getShort(this.context.position() + this.headerOffset);
                this.currentIsGlobal = headerElt == this.getElementIndex() + Short.MIN_VALUE;
                this.currentIsLocal = headerElt == this.getElementIndex();
            }
        }

        public boolean hasRemaining() {
            return this.bodyOffset < this.context.remaining();
        }

        public void moveToNext() {
            this.bodyOffset += STEP_LENGTH;
            if (this.currentIsGlobal || this.currentIsLocal) {
                this.headerOffset += HEADER_ELT_LENGTH;
            }
            this.updateIsGlobalOrLocal();
        }

        public void copyTo(ContextState other) {
            other.writeElement(this.getCounterId(), this.getClock(), this.getCount(), this.currentIsGlobal, this.currentIsLocal);
        }

        public int compareIdTo(ContextState other) {
            return CounterContext.compareId(this.context, this.context.position() + this.bodyOffset, other.context, other.context.position() + other.bodyOffset);
        }

        public void reset() {
            this.headerOffset = HEADER_SIZE_LENGTH;
            this.bodyOffset = this.headerLength;
            this.updateIsGlobalOrLocal();
        }

        public int getElementIndex() {
            return (this.bodyOffset - this.headerLength) / STEP_LENGTH;
        }

        public CounterId getCounterId() {
            return CounterId.wrap(this.context, this.context.position() + this.bodyOffset);
        }

        public long getClock() {
            return this.context.getLong(this.context.position() + this.bodyOffset + 16);
        }

        public long getCount() {
            return this.context.getLong(this.context.position() + this.bodyOffset + 16 + CLOCK_LENGTH);
        }

        public void writeGlobal(CounterId id, long clock, long count) {
            this.writeElement(id, clock, count, true, false);
        }

        public void writeLocal(CounterId id, long clock, long count) {
            this.writeElement(id, clock, count, false, true);
        }

        public void writeRemote(CounterId id, long clock, long count) {
            this.writeElement(id, clock, count, false, false);
        }

        private void writeElement(CounterId id, long clock, long count, boolean isGlobal, boolean isLocal) {
            this.writeElementAtOffset(this.context, this.context.position() + this.bodyOffset, id, clock, count);
            if (isGlobal) {
                this.context.putShort(this.context.position() + this.headerOffset, (short)(this.getElementIndex() + Short.MIN_VALUE));
            } else if (isLocal) {
                this.context.putShort(this.context.position() + this.headerOffset, (short)this.getElementIndex());
            }
            this.currentIsGlobal = isGlobal;
            this.currentIsLocal = isLocal;
            this.moveToNext();
        }

        private void writeElementAtOffset(ByteBuffer ctx, int offset, CounterId id, long clock, long count) {
            ctx = ctx.duplicate();
            ctx.position(offset);
            ctx.put(id.bytes().duplicate());
            ctx.putLong(clock);
            ctx.putLong(count);
        }
    }

    private static class LazyHolder {
        private static final CounterContext counterContext = new CounterContext();

        private LazyHolder() {
        }
    }

    public static enum Relationship {
        EQUAL,
        GREATER_THAN,
        LESS_THAN,
        DISJOINT;

    }
}

