/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.util.concurrent.RateLimiter;
import java.io.DataInputStream;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.BatchlogManagerMBean;
import org.apache.cassandra.db.CFRowAdder;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.HintedHandOffManager;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.WriteFailureException;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessageIn;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    private static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final long REPLAY_INTERVAL = 60000L;
    private static final int PAGE_SIZE = 128;
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    private final AtomicLong totalBatchesReplayed = new AtomicLong();
    private static final ScheduledExecutorService batchlogTasks = new DebuggableScheduledThreadPoolExecutor("BatchlogTasks");

    public void start() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        batchlogTasks.scheduleWithFixedDelay(runnable, StorageService.RING_DELAY, 60000L, TimeUnit.MILLISECONDS);
    }

    public static void shutdown() throws InterruptedException {
        batchlogTasks.shutdown();
        batchlogTasks.awaitTermination(60L, TimeUnit.SECONDS);
    }

    @Override
    public int countAllBatches() {
        String query = String.format("SELECT count(*) FROM %s.%s", "system", "batchlog");
        return (int)QueryProcessor.executeInternal(query, new Object[0]).one().getLong("count");
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed.longValue();
    }

    @Override
    public void forceBatchlogReplay() {
        this.startBatchlogReplay();
    }

    public Future<?> startBatchlogReplay() {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        return batchlogTasks.submit(runnable);
    }

    public static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid, int version) {
        return BatchlogManager.getBatchlogMutationFor(mutations, uuid, version, FBUtilities.timestampMicros());
    }

    @VisibleForTesting
    static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid, int version, long now) {
        ArrayBackedSortedColumns cf = ArrayBackedSortedColumns.factory.create(SystemKeyspace.Batchlog);
        CFRowAdder adder = new CFRowAdder(cf, SystemKeyspace.Batchlog.comparator.builder().build(), now);
        adder.add("data", BatchlogManager.serializeMutations(mutations, version)).add("written_at", new Date(now / 1000L)).add("version", version);
        return new Mutation("system", UUIDType.instance.decompose(uuid), cf);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static ByteBuffer serializeMutations(Collection<Mutation> mutations, int version) {
        try (DataOutputBuffer buf = new DataOutputBuffer();){
            buf.writeInt(mutations.size());
            for (Mutation mutation : mutations) {
                Mutation.serializer.serialize(mutation, (DataOutputPlus)buf, version);
            }
            ByteBuffer byteBuffer = buf.buffer();
            return byteBuffer;
        }
        catch (IOException e) {
            throw new AssertionError();
        }
    }

    private void replayAllFailedBatches() throws ExecutionException, InterruptedException {
        logger.debug("Started replayAllFailedBatches");
        int throttleInKB = DatabaseDescriptor.getBatchlogReplayThrottleInKB() / StorageService.instance.getTokenMetadata().getAllEndpoints().size();
        RateLimiter rateLimiter = RateLimiter.create((double)(throttleInKB == 0 ? Double.MAX_VALUE : (double)(throttleInKB * 1024)));
        UntypedResultSet page = QueryProcessor.executeInternal(String.format("SELECT id, data, written_at, version FROM %s.%s LIMIT %d", "system", "batchlog", 128), new Object[0]);
        while (!page.isEmpty()) {
            UUID id = this.processBatchlogPage(page, rateLimiter);
            if (page.size() < 128) break;
            page = QueryProcessor.executeInternal(String.format("SELECT id, data, written_at, version FROM %s.%s WHERE token(id) > token(?) LIMIT %d", "system", "batchlog", 128), id);
        }
        this.cleanup();
        logger.debug("Finished replayAllFailedBatches");
    }

    private void deleteBatch(UUID id) {
        Mutation mutation = new Mutation("system", UUIDType.instance.decompose(id));
        mutation.delete("batchlog", FBUtilities.timestampMicros());
        mutation.apply();
    }

    private UUID processBatchlogPage(UntypedResultSet page, RateLimiter rateLimiter) {
        UUID id = null;
        ArrayList<Batch> batches = new ArrayList<Batch>(page.size());
        for (UntypedResultSet.Row row : page) {
            id = row.getUUID("id");
            long writtenAt = row.getLong("written_at");
            long timeout = this.getBatchlogTimeout();
            if (System.currentTimeMillis() < writtenAt + timeout) continue;
            int version = row.has("version") ? row.getInt("version") : 6;
            Batch batch = new Batch(id, writtenAt, row.getBytes("data"), version);
            try {
                if (batch.replay(rateLimiter) > 0) {
                    batches.add(batch);
                    continue;
                }
                this.deleteBatch(id);
                this.totalBatchesReplayed.incrementAndGet();
            }
            catch (IOException e) {
                logger.warn("Skipped batch replay of {} due to {}", (Object)id, (Object)e);
                this.deleteBatch(id);
            }
        }
        for (Batch batch : batches) {
            batch.finish();
            this.deleteBatch(batch.id);
        }
        this.totalBatchesReplayed.addAndGet(batches.size());
        return id;
    }

    public long getBatchlogTimeout() {
        return DatabaseDescriptor.getWriteRpcTimeout() * 2L;
    }

    private void cleanup() throws ExecutionException, InterruptedException {
        ColumnFamilyStore cfs = Keyspace.open("system").getColumnFamilyStore("batchlog");
        cfs.forceBlockingFlush();
        ArrayList<Descriptor> descriptors = new ArrayList<Descriptor>();
        for (SSTableReader sstr : cfs.getSSTables()) {
            descriptors.add(sstr.descriptor);
        }
        if (!descriptors.isEmpty()) {
            CompactionManager.instance.submitUserDefined(cfs, descriptors, Integer.MAX_VALUE).get();
        }
    }

    public static class EndpointFilter {
        private final String localRack;
        private final Multimap<String, InetAddress> endpoints;

        public EndpointFilter(String localRack, Multimap<String, InetAddress> endpoints) {
            this.localRack = localRack;
            this.endpoints = endpoints;
        }

        public Collection<InetAddress> filter() {
            Collection racks;
            if (this.endpoints.values().size() == 1) {
                return this.endpoints.values();
            }
            ArrayListMultimap validated = ArrayListMultimap.create();
            for (Map.Entry entry : this.endpoints.entries()) {
                if (!this.isValid((InetAddress)entry.getValue())) continue;
                validated.put(entry.getKey(), entry.getValue());
            }
            if (validated.size() <= 2) {
                return validated.values();
            }
            if (validated.size() - validated.get((Object)this.localRack).size() >= 2) {
                validated.removeAll((Object)this.localRack);
            }
            if (validated.keySet().size() == 1) {
                List otherRack = validated.get(validated.keySet().iterator().next());
                Collections.shuffle(otherRack);
                return Lists.newArrayList((Iterable)Iterables.limit((Iterable)otherRack, (int)2));
            }
            if (validated.keySet().size() == 2) {
                racks = validated.keySet();
            } else {
                racks = Lists.newArrayList((Iterable)validated.keySet());
                Collections.shuffle((List)racks);
            }
            ArrayList<InetAddress> result = new ArrayList<InetAddress>(2);
            for (String rack : Iterables.limit((Iterable)racks, (int)2)) {
                List rackMembers = validated.get((Object)rack);
                result.add((InetAddress)rackMembers.get(this.getRandomInt(rackMembers.size())));
            }
            return result;
        }

        @VisibleForTesting
        protected boolean isValid(InetAddress input) {
            return !input.equals(FBUtilities.getBroadcastAddress()) && FailureDetector.instance.isAlive(input);
        }

        @VisibleForTesting
        protected int getRandomInt(int bound) {
            return ThreadLocalRandom.current().nextInt(bound);
        }
    }

    private static class Batch {
        private final UUID id;
        private final long writtenAt;
        private final ByteBuffer data;
        private final int version;
        private List<ReplayWriteResponseHandler<Mutation>> replayHandlers;

        public Batch(UUID id, long writtenAt, ByteBuffer data, int version) {
            this.id = id;
            this.writtenAt = writtenAt;
            this.data = data;
            this.version = version;
        }

        public int replay(RateLimiter rateLimiter) throws IOException {
            logger.debug("Replaying batch {}", (Object)this.id);
            List<Mutation> mutations = this.replayingMutations();
            if (mutations.isEmpty()) {
                return 0;
            }
            int ttl = this.calculateHintTTL(mutations);
            if (ttl <= 0) {
                return 0;
            }
            this.replayHandlers = this.sendReplays(mutations, this.writtenAt, ttl);
            rateLimiter.acquire(this.data.remaining());
            return this.replayHandlers.size();
        }

        public void finish() {
            for (int i = 0; i < this.replayHandlers.size(); ++i) {
                ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                try {
                    handler.get();
                    continue;
                }
                catch (WriteFailureException | WriteTimeoutException e) {
                    logger.debug("Failed replaying a batched mutation to a node, will write a hint");
                    logger.debug("Failure was : {}", (Object)e.getMessage());
                    this.writeHintsForUndeliveredEndpoints(i);
                    return;
                }
            }
        }

        private List<Mutation> replayingMutations() throws IOException {
            DataInputStream in = new DataInputStream(ByteBufferUtil.inputStream(this.data));
            int size = in.readInt();
            ArrayList<Mutation> mutations = new ArrayList<Mutation>(size);
            for (int i = 0; i < size; ++i) {
                Mutation mutation = Mutation.serializer.deserialize(in, this.version);
                for (UUID cfId : mutation.getColumnFamilyIds()) {
                    if (this.writtenAt > SystemKeyspace.getTruncatedAt(cfId)) continue;
                    mutation = mutation.without(cfId);
                }
                if (mutation.isEmpty()) continue;
                mutations.add(mutation);
            }
            return mutations;
        }

        private void writeHintsForUndeliveredEndpoints(int startFrom) {
            try {
                List<Mutation> replayingMutations = this.replayingMutations();
                for (int i = startFrom; i < this.replayHandlers.size(); ++i) {
                    Mutation undeliveredMutation = replayingMutations.get(i);
                    int ttl = this.calculateHintTTL(replayingMutations);
                    ReplayWriteResponseHandler<Mutation> handler = this.replayHandlers.get(i);
                    if (ttl <= 0 || handler == null) continue;
                    for (InetAddress endpoint : ((ReplayWriteResponseHandler)handler).undelivered) {
                        StorageProxy.writeHintForMutation(undeliveredMutation, this.writtenAt, ttl, endpoint);
                    }
                }
            }
            catch (IOException e) {
                logger.error("Cannot schedule hints for undelivered batch", (Throwable)e);
            }
        }

        private List<ReplayWriteResponseHandler<Mutation>> sendReplays(List<Mutation> mutations, long writtenAt, int ttl) {
            ArrayList<ReplayWriteResponseHandler<Mutation>> handlers = new ArrayList<ReplayWriteResponseHandler<Mutation>>(mutations.size());
            for (Mutation mutation : mutations) {
                ReplayWriteResponseHandler<Mutation> handler = this.sendSingleReplayMutation(mutation, writtenAt, ttl);
                if (handler == null) continue;
                handlers.add(handler);
            }
            return handlers;
        }

        private ReplayWriteResponseHandler<Mutation> sendSingleReplayMutation(Mutation mutation, long writtenAt, int ttl) {
            HashSet<InetAddress> liveEndpoints = new HashSet<InetAddress>();
            String ks = mutation.getKeyspaceName();
            Token tk = StorageService.getPartitioner().getToken(mutation.key());
            for (InetAddress endpoint : Iterables.concat(StorageService.instance.getNaturalEndpoints(ks, tk), StorageService.instance.getTokenMetadata().pendingEndpointsFor(tk, ks))) {
                if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                    mutation.apply();
                    continue;
                }
                if (FailureDetector.instance.isAlive(endpoint)) {
                    liveEndpoints.add(endpoint);
                    continue;
                }
                StorageProxy.writeHintForMutation(mutation, writtenAt, ttl, endpoint);
            }
            if (liveEndpoints.isEmpty()) {
                return null;
            }
            ReplayWriteResponseHandler<Mutation> handler = new ReplayWriteResponseHandler<Mutation>(liveEndpoints);
            MessageOut<Mutation> message = mutation.createMessage();
            for (InetAddress endpoint : liveEndpoints) {
                MessagingService.instance().sendRR(message, endpoint, handler, false);
            }
            return handler;
        }

        private int calculateHintTTL(Collection<Mutation> mutations) {
            int unadjustedTTL = Integer.MAX_VALUE;
            for (Mutation mutation : mutations) {
                unadjustedTTL = Math.min(unadjustedTTL, HintedHandOffManager.calculateHintTTL(mutation));
            }
            return unadjustedTTL - (int)TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - this.writtenAt);
        }

        private static class ReplayWriteResponseHandler<T>
        extends WriteResponseHandler<T> {
            private final Set<InetAddress> undelivered = Collections.newSetFromMap(new ConcurrentHashMap());

            public ReplayWriteResponseHandler(Collection<InetAddress> writeEndpoints) {
                super(writeEndpoints, Collections.emptySet(), null, null, null, WriteType.UNLOGGED_BATCH);
                this.undelivered.addAll(writeEndpoints);
            }

            @Override
            protected int totalBlockFor() {
                return this.naturalEndpoints.size();
            }

            @Override
            public void response(MessageIn<T> m) {
                boolean removed = this.undelivered.remove(m == null ? FBUtilities.getBroadcastAddress() : m.from);
                assert (removed);
                super.response(m);
            }
        }
    }
}

