/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.util.AbstractMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.AtomicBTreeColumns;
import org.apache.cassandra.db.BufferDecoratedKey;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.db.composites.CellNameType;
import org.apache.cassandra.db.index.SecondaryIndexManager;
import org.apache.cassandra.dht.Murmur3Partitioner;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.SSTableWriter;
import org.apache.cassandra.io.sstable.metadata.MetadataCollector;
import org.apache.cassandra.io.util.DiskAwareRunnable;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.ObjectSizes;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.OpOrder;
import org.apache.cassandra.utils.memory.HeapAllocator;
import org.apache.cassandra.utils.memory.MemtableAllocator;
import org.apache.cassandra.utils.memory.MemtablePool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Memtable
implements Comparable<Memtable> {
    private static final Logger logger = LoggerFactory.getLogger(Memtable.class);
    static final MemtablePool MEMORY_POOL = DatabaseDescriptor.getMemtableAllocatorPool();
    private static final int ROW_OVERHEAD_HEAP_SIZE = Memtable.estimateRowOverhead(Integer.parseInt(System.getProperty("cassandra.memtable_row_overhead_computation_step", "100000")));
    private final MemtableAllocator allocator;
    private final AtomicLong liveDataSize = new AtomicLong(0L);
    private final AtomicLong currentOperations = new AtomicLong(0L);
    private volatile OpOrder.Barrier writeBarrier;
    private volatile AtomicReference<ReplayPosition> lastReplayPosition;
    private final ReplayPosition minReplayPosition = CommitLog.instance.getContext();
    private final ConcurrentNavigableMap<RowPosition, AtomicBTreeColumns> rows = new ConcurrentSkipListMap<RowPosition, AtomicBTreeColumns>();
    public final ColumnFamilyStore cfs;
    private final long creationTime = System.currentTimeMillis();
    private final long creationNano = System.nanoTime();
    public final CellNameType initialComparator;

    @Override
    public int compareTo(Memtable that) {
        return this.minReplayPosition.compareTo(that.minReplayPosition);
    }

    public Memtable(ColumnFamilyStore cfs) {
        this.cfs = cfs;
        this.allocator = MEMORY_POOL.newAllocator();
        this.initialComparator = cfs.metadata.comparator;
        this.cfs.scheduleFlush();
    }

    @VisibleForTesting
    public Memtable(CFMetaData metadata) {
        this.initialComparator = metadata.comparator;
        this.cfs = null;
        this.allocator = null;
    }

    public MemtableAllocator getAllocator() {
        return this.allocator;
    }

    public long getLiveDataSize() {
        return this.liveDataSize.get();
    }

    public long getOperations() {
        return this.currentOperations.get();
    }

    @VisibleForTesting
    public void setDiscarding(OpOrder.Barrier writeBarrier, AtomicReference<ReplayPosition> lastReplayPosition) {
        assert (this.writeBarrier == null);
        this.lastReplayPosition = lastReplayPosition;
        this.writeBarrier = writeBarrier;
        this.allocator.setDiscarding();
    }

    void setDiscarded() {
        this.allocator.setDiscarded();
    }

    public boolean accepts(OpOrder.Group opGroup, ReplayPosition replayPosition) {
        ReplayPosition currentLast;
        OpOrder.Barrier barrier = this.writeBarrier;
        if (barrier == null) {
            return true;
        }
        if (!barrier.isAfter(opGroup)) {
            return false;
        }
        if (replayPosition == null) {
            return true;
        }
        do {
            if ((currentLast = this.lastReplayPosition.get()) instanceof LastReplayPosition) {
                return currentLast.compareTo(replayPosition) >= 0;
            }
            if (currentLast == null || currentLast.compareTo(replayPosition) < 0) continue;
            return true;
        } while (!this.lastReplayPosition.compareAndSet(currentLast, replayPosition));
        return true;
    }

    public boolean isLive() {
        return this.allocator.isLive();
    }

    public boolean isClean() {
        return this.rows.isEmpty();
    }

    public boolean isCleanAfter(ReplayPosition position) {
        return this.isClean() || position != null && this.minReplayPosition.compareTo(position) >= 0;
    }

    public boolean isExpired() {
        int period = this.cfs.metadata.getMemtableFlushPeriod();
        return period > 0 && System.nanoTime() - this.creationNano >= TimeUnit.MILLISECONDS.toNanos(period);
    }

    long put(DecoratedKey key, ColumnFamily cf, SecondaryIndexManager.Updater indexer, OpOrder.Group opGroup) {
        AtomicBTreeColumns previous = (AtomicBTreeColumns)this.rows.get(key);
        long initialSize = 0L;
        if (previous == null) {
            AtomicBTreeColumns empty = cf.cloneMeShallow(AtomicBTreeColumns.factory, false);
            DecoratedKey cloneKey = this.allocator.clone(key, opGroup);
            previous = this.rows.putIfAbsent(cloneKey, empty);
            if (previous == null) {
                previous = empty;
                int overhead = (int)(key.getToken().getHeapSize() + (long)ROW_OVERHEAD_HEAP_SIZE);
                this.allocator.onHeap().allocate(overhead, opGroup);
                initialSize = 8L;
            } else {
                this.allocator.reclaimer().reclaimImmediately(cloneKey);
            }
        }
        Pair<Long, Long> pair = previous.addAllWithSizeDelta(cf, this.allocator, opGroup, indexer);
        this.liveDataSize.addAndGet(initialSize + (Long)pair.left);
        this.currentOperations.addAndGet(cf.getColumnCount() + (cf.isMarkedForDelete() ? 1 : 0) + cf.deletionInfo().rangeCount());
        return (Long)pair.right;
    }

    public String contents() {
        StringBuilder builder = new StringBuilder();
        builder.append("{");
        for (Map.Entry entry : this.rows.entrySet()) {
            builder.append(entry.getKey()).append(": ").append(entry.getValue()).append(", ");
        }
        builder.append("}");
        return builder.toString();
    }

    public int partitionCount() {
        return this.rows.size();
    }

    public FlushRunnable flushRunnable() {
        return new FlushRunnable(this.lastReplayPosition.get());
    }

    public String toString() {
        return String.format("Memtable-%s@%s(%s serialized bytes, %s ops, %.0f%%/%.0f%% of on/off-heap limit)", this.cfs.name, this.hashCode(), this.liveDataSize, this.currentOperations, Float.valueOf(100.0f * this.allocator.onHeap().ownershipRatio()), Float.valueOf(100.0f * this.allocator.offHeap().ownershipRatio()));
    }

    public Iterator<Map.Entry<DecoratedKey, ColumnFamily>> getEntryIterator(final RowPosition startWith, final RowPosition stopAt) {
        return new Iterator<Map.Entry<DecoratedKey, ColumnFamily>>(){
            private Iterator<? extends Map.Entry<? extends RowPosition, AtomicBTreeColumns>> iter;
            private Map.Entry<? extends RowPosition, ? extends ColumnFamily> currentEntry;
            {
                this.iter = stopAt.isMinimum() ? Memtable.this.rows.tailMap(startWith).entrySet().iterator() : Memtable.this.rows.subMap(startWith, true, stopAt, true).entrySet().iterator();
            }

            @Override
            public boolean hasNext() {
                return this.iter.hasNext();
            }

            @Override
            public Map.Entry<DecoratedKey, ColumnFamily> next() {
                Map.Entry<? extends RowPosition, AtomicBTreeColumns> entry = this.iter.next();
                assert (entry.getKey() instanceof DecoratedKey);
                if (MEMORY_POOL.needToCopyOnHeap()) {
                    DecoratedKey key = (DecoratedKey)entry.getKey();
                    key = new BufferDecoratedKey(key.getToken(), HeapAllocator.instance.clone(key.getKey()));
                    ArrayBackedSortedColumns cells = ArrayBackedSortedColumns.localCopy(entry.getValue(), HeapAllocator.instance);
                    entry = new AbstractMap.SimpleImmutableEntry<DecoratedKey, ArrayBackedSortedColumns>(key, cells);
                }
                this.currentEntry = entry;
                return entry;
            }

            @Override
            public void remove() {
                this.iter.remove();
                Memtable.this.liveDataSize.addAndGet(-this.currentEntry.getValue().dataSize());
                this.currentEntry = null;
            }
        };
    }

    public ColumnFamily getColumnFamily(DecoratedKey key) {
        return (ColumnFamily)this.rows.get(key);
    }

    public long creationTime() {
        return this.creationTime;
    }

    private static int estimateRowOverhead(int count) {
        try (OpOrder.Group group = new OpOrder().start();){
            MemtableAllocator allocator = MEMORY_POOL.newAllocator();
            ConcurrentSkipListMap<DecoratedKey, Object> rows = new ConcurrentSkipListMap<DecoratedKey, Object>();
            Object val = new Object();
            for (int i = 0; i < count; ++i) {
                rows.put(allocator.clone(new BufferDecoratedKey(new Murmur3Partitioner.LongToken(i), ByteBufferUtil.EMPTY_BYTE_BUFFER), group), val);
            }
            double avgSize = (double)ObjectSizes.measureDeep(rows) / (double)count;
            int rowOverhead = (int)(avgSize - Math.floor(avgSize) < 0.05 ? Math.floor(avgSize) : Math.ceil(avgSize));
            rowOverhead = (int)((long)rowOverhead - ObjectSizes.measureDeep(new Murmur3Partitioner.LongToken(0L)));
            rowOverhead = (int)((long)rowOverhead + AtomicBTreeColumns.EMPTY_SIZE);
            allocator.setDiscarding();
            allocator.setDiscarded();
            int n = rowOverhead;
            return n;
        }
    }

    class FlushRunnable
    extends DiskAwareRunnable {
        private final ReplayPosition context;
        private final long estimatedSize;

        FlushRunnable(ReplayPosition context) {
            this.context = context;
            long keySize = 0L;
            for (RowPosition key : Memtable.this.rows.keySet()) {
                assert (key instanceof DecoratedKey);
                keySize += (long)((DecoratedKey)key).getKey().remaining();
            }
            this.estimatedSize = (long)((double)(keySize + keySize + Memtable.this.liveDataSize.get()) * 1.2);
        }

        public long getExpectedWriteSize() {
            return this.estimatedSize;
        }

        @Override
        protected void runMayThrow() throws Exception {
            long writeSize = this.getExpectedWriteSize();
            Directories.DataDirectory dataDirectory = this.getWriteDirectory(writeSize);
            File sstableDirectory = Memtable.this.cfs.directories.getLocationForDisk(dataDirectory);
            assert (sstableDirectory != null) : "Flush task is not bound to any disk";
            SSTableReader sstable = this.writeSortedContents(this.context, sstableDirectory);
            Memtable.this.cfs.replaceFlushed(Memtable.this, sstable);
        }

        @Override
        protected Directories getDirectories() {
            return Memtable.this.cfs.directories;
        }

        private SSTableReader writeSortedContents(ReplayPosition context, File sstableDirectory) {
            logger.info("Writing {}", (Object)Memtable.this.toString());
            try (SSTableWriter writer = this.createFlushWriter(Memtable.this.cfs.getTempSSTablePath(sstableDirectory));){
                SSTableReader ssTable;
                boolean trackContention = logger.isDebugEnabled();
                int heavilyContendedRowCount = 0;
                for (Map.Entry entry : Memtable.this.rows.entrySet()) {
                    AtomicBTreeColumns cf = (AtomicBTreeColumns)entry.getValue();
                    if (cf.isMarkedForDelete() && cf.hasColumns() && Memtable.this.cfs.name.equals("batchlog") && Memtable.this.cfs.keyspace.getName().equals("system")) continue;
                    if (trackContention && cf.usePessimisticLocking()) {
                        ++heavilyContendedRowCount;
                    }
                    if (cf.isEmpty()) continue;
                    writer.append((DecoratedKey)entry.getKey(), cf);
                }
                if (writer.getFilePointer() > 0L) {
                    ssTable = writer.finish(true);
                    logger.info(String.format("Completed flushing %s (%d bytes) for commitlog position %s", ssTable.getFilename(), new File(ssTable.getFilename()).length(), context));
                } else {
                    ssTable = null;
                    logger.info("Completed flushing; nothing needed to be retained.  Commitlog position was {}", (Object)context);
                }
                if (heavilyContendedRowCount > 0) {
                    logger.debug(String.format("High update contention in %d/%d partitions of %s ", heavilyContendedRowCount, Memtable.this.rows.size(), Memtable.this.toString()));
                }
                SSTableReader sSTableReader = ssTable;
                return sSTableReader;
            }
        }

        public SSTableWriter createFlushWriter(String filename) {
            MetadataCollector sstableMetadataCollector = new MetadataCollector(Memtable.this.cfs.metadata.comparator).replayPosition(this.context);
            return SSTableWriter.create(Descriptor.fromFilename(filename), Long.valueOf(Memtable.this.rows.size()), (Long)0L, Memtable.this.cfs.metadata, Memtable.this.cfs.partitioner, sstableMetadataCollector);
        }
    }

    public static final class LastReplayPosition
    extends ReplayPosition {
        public LastReplayPosition(ReplayPosition copy) {
            super(copy.segment, copy.position);
        }
    }
}

