/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.Striped;
import java.io.DataInput;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BufferCounterCell;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ClockAndCount;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.CounterCell;
import org.apache.cassandra.db.CounterUpdateCell;
import org.apache.cassandra.db.IMutation;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.SliceByNamesReadCommand;
import org.apache.cassandra.db.TypeSizes;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.composites.Composite;
import org.apache.cassandra.db.context.CounterContext;
import org.apache.cassandra.db.filter.NamesQueryFilter;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.io.IVersionedSerializer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.CacheService;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.CounterId;

public class CounterMutation
implements IMutation {
    public static final CounterMutationSerializer serializer = new CounterMutationSerializer();
    private static final Striped<Lock> LOCKS = Striped.lazyWeakLock((int)(DatabaseDescriptor.getConcurrentCounterWriters() * 1024));
    private final Mutation mutation;
    private final ConsistencyLevel consistency;

    public CounterMutation(Mutation mutation, ConsistencyLevel consistency) {
        this.mutation = mutation;
        this.consistency = consistency;
    }

    @Override
    public String getKeyspaceName() {
        return this.mutation.getKeyspaceName();
    }

    @Override
    public Collection<UUID> getColumnFamilyIds() {
        return this.mutation.getColumnFamilyIds();
    }

    @Override
    public Collection<ColumnFamily> getColumnFamilies() {
        return this.mutation.getColumnFamilies();
    }

    public Mutation getMutation() {
        return this.mutation;
    }

    @Override
    public ByteBuffer key() {
        return this.mutation.key();
    }

    public ConsistencyLevel consistency() {
        return this.consistency;
    }

    public MessageOut<CounterMutation> makeMutationMessage() {
        return new MessageOut<CounterMutation>(MessagingService.Verb.COUNTER_MUTATION, this, serializer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Mutation apply() throws WriteTimeoutException {
        Mutation result = new Mutation(this.getKeyspaceName(), this.key());
        Keyspace keyspace = Keyspace.open(this.getKeyspaceName());
        int count = 0;
        for (ColumnFamily cf : this.getColumnFamilies()) {
            count += cf.getColumnCount();
        }
        ArrayList<Lock> locks = new ArrayList<Lock>(count);
        Tracing.trace("Acquiring {} counter locks", (Object)count);
        try {
            this.grabCounterLocks(keyspace, locks);
            for (ColumnFamily cf : this.getColumnFamilies()) {
                result.add(this.processModifications(cf));
            }
            result.apply();
            this.updateCounterCache(result, keyspace);
            Mutation mutation = result;
            return mutation;
        }
        finally {
            for (Lock lock : locks) {
                lock.unlock();
            }
        }
    }

    private void grabCounterLocks(Keyspace keyspace, List<Lock> locks) throws WriteTimeoutException {
        long startTime = System.nanoTime();
        for (Lock lock : LOCKS.bulkGet(this.getCounterLockKeys())) {
            long timeout = TimeUnit.MILLISECONDS.toNanos(this.getTimeout()) - (System.nanoTime() - startTime);
            try {
                if (!lock.tryLock(timeout, TimeUnit.NANOSECONDS)) {
                    throw new WriteTimeoutException(WriteType.COUNTER, this.consistency(), 0, this.consistency().blockFor(keyspace));
                }
                locks.add(lock);
            }
            catch (InterruptedException e) {
                throw new WriteTimeoutException(WriteType.COUNTER, this.consistency(), 0, this.consistency().blockFor(keyspace));
            }
        }
    }

    private Iterable<Object> getCounterLockKeys() {
        return Iterables.concat((Iterable)Iterables.transform(this.getColumnFamilies(), (Function)new Function<ColumnFamily, Iterable<Object>>(){

            public Iterable<Object> apply(final ColumnFamily cf) {
                return Iterables.transform((Iterable)cf, (Function)new Function<Cell, Object>(){

                    public Object apply(Cell cell) {
                        return Objects.hashCode((Object[])new Object[]{cf.id(), CounterMutation.this.key(), cell.name()});
                    }
                });
            }
        }));
    }

    private ColumnFamily processModifications(ColumnFamily changesCF) {
        ColumnFamilyStore cfs = Keyspace.open(this.getKeyspaceName()).getColumnFamilyStore(changesCF.id());
        ColumnFamily resultCF = changesCF.cloneMeShallow();
        ArrayList<CounterUpdateCell> counterUpdateCells = new ArrayList<CounterUpdateCell>(changesCF.getColumnCount());
        for (Cell cell : changesCF) {
            if (cell instanceof CounterUpdateCell) {
                counterUpdateCells.add((CounterUpdateCell)cell);
                continue;
            }
            resultCF.addColumn(cell);
        }
        if (counterUpdateCells.isEmpty()) {
            return resultCF;
        }
        ClockAndCount[] currentValues = this.getCurrentValues(counterUpdateCells, cfs);
        for (int i = 0; i < counterUpdateCells.size(); ++i) {
            ClockAndCount currentValue = currentValues[i];
            CounterUpdateCell update = (CounterUpdateCell)counterUpdateCells.get(i);
            long clock = currentValue.clock + 1L;
            long count = currentValue.count + update.delta();
            resultCF.addColumn(new BufferCounterCell(update.name(), CounterContext.instance().createGlobal(CounterId.getLocalId(), clock, count), update.timestamp()));
        }
        return resultCF;
    }

    private ClockAndCount[] getCurrentValues(List<CounterUpdateCell> counterUpdateCells, ColumnFamilyStore cfs) {
        ClockAndCount[] currentValues = new ClockAndCount[counterUpdateCells.size()];
        int remaining = counterUpdateCells.size();
        if (CacheService.instance.counterCache.getCapacity() != 0L) {
            Tracing.trace("Fetching {} counter values from cache", (Object)counterUpdateCells.size());
            remaining = this.getCurrentValuesFromCache(counterUpdateCells, cfs, currentValues);
            if (remaining == 0) {
                return currentValues;
            }
        }
        Tracing.trace("Reading {} counter values from the CF", (Object)remaining);
        this.getCurrentValuesFromCFS(counterUpdateCells, cfs, currentValues);
        return currentValues;
    }

    private int getCurrentValuesFromCache(List<CounterUpdateCell> counterUpdateCells, ColumnFamilyStore cfs, ClockAndCount[] currentValues) {
        int cacheMisses = 0;
        for (int i = 0; i < counterUpdateCells.size(); ++i) {
            ClockAndCount cached = cfs.getCachedCounter(this.key(), counterUpdateCells.get(i).name());
            if (cached != null) {
                currentValues[i] = cached;
                continue;
            }
            ++cacheMisses;
        }
        return cacheMisses;
    }

    private void getCurrentValuesFromCFS(List<CounterUpdateCell> counterUpdateCells, ColumnFamilyStore cfs, ClockAndCount[] currentValues) {
        TreeSet<Composite> names = new TreeSet<Composite>(cfs.metadata.comparator);
        for (int i = 0; i < currentValues.length; ++i) {
            if (currentValues[i] != null) continue;
            names.add(counterUpdateCells.get(i).name());
        }
        SliceByNamesReadCommand cmd = new SliceByNamesReadCommand(this.getKeyspaceName(), this.key(), cfs.metadata.cfName, Long.MIN_VALUE, new NamesQueryFilter(names));
        Row row = ((ReadCommand)cmd).getRow(cfs.keyspace);
        ColumnFamily cf = row == null ? null : row.cf;
        for (int i = 0; i < currentValues.length; ++i) {
            if (currentValues[i] != null) continue;
            Cell cell = cf == null ? null : cf.getColumn(counterUpdateCells.get(i).name());
            currentValues[i] = cell == null || !cell.isLive() ? ClockAndCount.BLANK : CounterContext.instance().getLocalClockAndCount(cell.value());
        }
    }

    private void updateCounterCache(Mutation applied, Keyspace keyspace) {
        if (CacheService.instance.counterCache.getCapacity() == 0L) {
            return;
        }
        for (ColumnFamily cf : applied.getColumnFamilies()) {
            ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(cf.id());
            for (Cell cell : cf) {
                if (!(cell instanceof CounterCell)) continue;
                cfs.putCachedCounter(this.key(), cell.name(), CounterContext.instance().getLocalClockAndCount(cell.value()));
            }
        }
    }

    @Override
    public void addAll(IMutation m) {
        if (!(m instanceof CounterMutation)) {
            throw new IllegalArgumentException();
        }
        CounterMutation cm = (CounterMutation)m;
        this.mutation.addAll(cm.mutation);
    }

    @Override
    public long getTimeout() {
        return DatabaseDescriptor.getCounterWriteRpcTimeout();
    }

    public String toString() {
        return this.toString(false);
    }

    @Override
    public String toString(boolean shallow) {
        return String.format("CounterMutation(%s, %s)", new Object[]{this.mutation.toString(shallow), this.consistency});
    }

    public static class CounterMutationSerializer
    implements IVersionedSerializer<CounterMutation> {
        @Override
        public void serialize(CounterMutation cm, DataOutputPlus out, int version) throws IOException {
            Mutation.serializer.serialize(cm.mutation, out, version);
            out.writeUTF(cm.consistency.name());
        }

        @Override
        public CounterMutation deserialize(DataInput in, int version) throws IOException {
            Mutation m = Mutation.serializer.deserialize(in, version);
            ConsistencyLevel consistency = Enum.valueOf(ConsistencyLevel.class, in.readUTF());
            return new CounterMutation(m, consistency);
        }

        @Override
        public long serializedSize(CounterMutation cm, int version) {
            return Mutation.serializer.serializedSize(cm.mutation, version) + (long)TypeSizes.NATIVE.sizeof(cm.consistency.name());
        }
    }
}

