/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DataTracker;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.AlwaysPresentFilter;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionController
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CompactionController.class);
    static final boolean NEVER_PURGE_TOMBSTONES = Boolean.getBoolean("cassandra.never_purge_tombstones");
    public final ColumnFamilyStore cfs;
    private DataTracker.SSTableIntervalTree overlappingTree;
    private Refs<SSTableReader> overlappingSSTables;
    private final Iterable<SSTableReader> compacting;
    public final int gcBefore;

    protected CompactionController(ColumnFamilyStore cfs, int maxValue) {
        this(cfs, null, maxValue);
    }

    public CompactionController(ColumnFamilyStore cfs, Set<SSTableReader> compacting, int gcBefore) {
        assert (cfs != null);
        this.cfs = cfs;
        this.gcBefore = gcBefore;
        this.compacting = compacting;
        this.refreshOverlaps();
        if (NEVER_PURGE_TOMBSTONES) {
            logger.warn("You are running with -Dcassandra.never_purge_tombstones=true, this is dangerous!");
        }
    }

    void maybeRefreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            logger.debug("not refreshing overlaps - running with -Dcassandra.never_purge_tombstones=true");
            return;
        }
        for (SSTableReader reader : this.overlappingSSTables) {
            if (!reader.isMarkedCompacted()) continue;
            this.refreshOverlaps();
            return;
        }
    }

    private void refreshOverlaps() {
        if (NEVER_PURGE_TOMBSTONES) {
            return;
        }
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
        this.overlappingSSTables = this.compacting == null ? Refs.tryRef(Collections.emptyList()) : this.cfs.getAndReferenceOverlappingSSTables(this.compacting);
        this.overlappingTree = DataTracker.buildIntervalTree(this.overlappingSSTables);
    }

    public Set<SSTableReader> getFullyExpiredSSTables() {
        return CompactionController.getFullyExpiredSSTables(this.cfs, this.compacting, this.overlappingSSTables, this.gcBefore);
    }

    public static Set<SSTableReader> getFullyExpiredSSTables(ColumnFamilyStore cfStore, Iterable<SSTableReader> compacting, Iterable<SSTableReader> overlapping, int gcBefore) {
        logger.debug("Checking droppable sstables in {}", (Object)cfStore);
        if (compacting == null || NEVER_PURGE_TOMBSTONES) {
            return Collections.emptySet();
        }
        ArrayList<SSTableReader> candidates = new ArrayList<SSTableReader>();
        long minTimestamp = Long.MAX_VALUE;
        for (SSTableReader sstable : overlapping) {
            if (sstable.getSSTableMetadata().maxLocalDeletionTime < gcBefore) continue;
            minTimestamp = Math.min(minTimestamp, sstable.getMinTimestamp());
        }
        for (SSTableReader candidate : compacting) {
            if (candidate.getSSTableMetadata().maxLocalDeletionTime < gcBefore) {
                candidates.add(candidate);
                continue;
            }
            minTimestamp = Math.min(minTimestamp, candidate.getMinTimestamp());
        }
        Iterator iterator = candidates.iterator();
        while (iterator.hasNext()) {
            SSTableReader candidate;
            candidate = (SSTableReader)iterator.next();
            if (candidate.getMaxTimestamp() >= minTimestamp) {
                iterator.remove();
                continue;
            }
            logger.debug("Dropping expired SSTable {} (maxLocalDeletionTime={}, gcBefore={})", new Object[]{candidate, candidate.getSSTableMetadata().maxLocalDeletionTime, gcBefore});
        }
        return new HashSet<SSTableReader>(candidates);
    }

    public String getKeyspace() {
        return this.cfs.keyspace.getName();
    }

    public String getColumnFamily() {
        return this.cfs.name;
    }

    public long maxPurgeableTimestamp(DecoratedKey key) {
        if (NEVER_PURGE_TOMBSTONES) {
            return Long.MIN_VALUE;
        }
        List filteredSSTables = this.overlappingTree.search(key);
        long min = Long.MAX_VALUE;
        for (SSTableReader sstable : filteredSSTables) {
            if (sstable.getBloomFilter() instanceof AlwaysPresentFilter && sstable.getPosition(key, SSTableReader.Operator.EQ, false) != null) {
                min = Math.min(min, sstable.getMinTimestamp());
                continue;
            }
            if (!sstable.getBloomFilter().isPresent(key.getKey())) continue;
            min = Math.min(min, sstable.getMinTimestamp());
        }
        return min;
    }

    @Override
    public void close() {
        if (this.overlappingSSTables != null) {
            this.overlappingSSTables.release();
        }
    }
}

