/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.RateLimiter;
import java.io.DataInputStream;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.BatchlogManagerMBean;
import org.apache.cassandra.db.CFRowAdder;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.HintedHandOffManager;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.dht.RingPosition;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    private static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final long REPLAY_INTERVAL = 60000L;
    private static final int PAGE_SIZE = 128;
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    private final AtomicLong totalBatchesReplayed = new AtomicLong();
    private final AtomicBoolean isReplaying = new AtomicBoolean();
    public static final ScheduledExecutorService batchlogTasks = new DebuggableScheduledThreadPoolExecutor("BatchlogTasks");

    public void start() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        batchlogTasks.scheduleWithFixedDelay(runnable, StorageService.RING_DELAY, 60000L, TimeUnit.MILLISECONDS);
    }

    @Override
    public int countAllBatches() {
        return (int)BatchlogManager.process("SELECT count(*) FROM %s.%s", "system", "batchlog").one().getLong("count");
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed.longValue();
    }

    @Override
    public void forceBatchlogReplay() {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        batchlogTasks.execute(runnable);
    }

    public static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid, int version) {
        return BatchlogManager.getBatchlogMutationFor(mutations, uuid, version, FBUtilities.timestampMicros());
    }

    @VisibleForTesting
    static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid, int version, long now) {
        ArrayBackedSortedColumns cf = ArrayBackedSortedColumns.factory.create(CFMetaData.BatchlogCf);
        CFRowAdder adder = new CFRowAdder(cf, CFMetaData.BatchlogCf.comparator.builder().build(), now);
        adder.add("data", BatchlogManager.serializeMutations(mutations, version)).add("written_at", new Date(now / 1000L)).add("version", version);
        return new Mutation("system", UUIDType.instance.decompose(uuid), cf);
    }

    private static ByteBuffer serializeMutations(Collection<Mutation> mutations, int version) {
        DataOutputBuffer buf = new DataOutputBuffer();
        try {
            buf.writeInt(mutations.size());
            for (Mutation mutation : mutations) {
                Mutation.serializer.serialize(mutation, (DataOutputPlus)buf, version);
            }
        }
        catch (IOException e) {
            throw new AssertionError();
        }
        return buf.asByteBuffer();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void replayAllFailedBatches() throws ExecutionException, InterruptedException {
        if (!this.isReplaying.compareAndSet(false, true)) {
            return;
        }
        logger.debug("Started replayAllFailedBatches");
        int throttleInKB = DatabaseDescriptor.getBatchlogReplayThrottleInKB() / StorageService.instance.getTokenMetadata().getAllEndpoints().size();
        RateLimiter rateLimiter = RateLimiter.create((double)(throttleInKB == 0 ? Double.MAX_VALUE : (double)(throttleInKB * 1024)));
        try {
            UntypedResultSet page = BatchlogManager.process("SELECT id, data, written_at, version FROM %s.%s LIMIT %d", "system", "batchlog", 128);
            while (!page.isEmpty()) {
                UUID id = this.processBatchlogPage(page, rateLimiter);
                if (page.size() < 128) break;
                page = BatchlogManager.process("SELECT id, data, written_at, version FROM %s.%s WHERE token(id) > token(%s) LIMIT %d", "system", "batchlog", id, 128);
            }
            this.cleanup();
        }
        finally {
            this.isReplaying.set(false);
        }
        logger.debug("Finished replayAllFailedBatches");
    }

    private UUID processBatchlogPage(UntypedResultSet page, RateLimiter rateLimiter) {
        UUID id = null;
        for (UntypedResultSet.Row row : page) {
            id = row.getUUID("id");
            long writtenAt = row.getLong("written_at");
            int version = row.has("version") ? row.getInt("version") : 6;
            long timeout = DatabaseDescriptor.getWriteRpcTimeout() * 2L;
            if (System.currentTimeMillis() < writtenAt + timeout) continue;
            this.replayBatch(id, row.getBytes("data"), writtenAt, version, rateLimiter);
        }
        return id;
    }

    private void replayBatch(UUID id, ByteBuffer data, long writtenAt, int version, RateLimiter rateLimiter) {
        logger.debug("Replaying batch {}", (Object)id);
        try {
            this.replaySerializedMutations(data, writtenAt, version, rateLimiter);
        }
        catch (IOException e) {
            logger.warn("Skipped batch replay of {} due to {}", (Object)id, (Object)e);
        }
        this.deleteBatch(id);
        this.totalBatchesReplayed.incrementAndGet();
    }

    private void deleteBatch(UUID id) {
        Mutation mutation = new Mutation("system", UUIDType.instance.decompose(id));
        mutation.delete("batchlog", FBUtilities.timestampMicros());
        mutation.apply();
    }

    private void replaySerializedMutations(ByteBuffer data, long writtenAt, int version, RateLimiter rateLimiter) throws IOException {
        DataInputStream in = new DataInputStream(ByteBufferUtil.inputStream(data));
        int size = in.readInt();
        ArrayList<Mutation> mutations = new ArrayList<Mutation>(size);
        for (int i = 0; i < size; ++i) {
            Mutation mutation = Mutation.serializer.deserialize(in, version);
            for (UUID cfId : mutation.getColumnFamilyIds()) {
                if (writtenAt > SystemKeyspace.getTruncatedAt(cfId)) continue;
                mutation = mutation.without(cfId);
            }
            if (mutation.isEmpty()) continue;
            mutations.add(mutation);
        }
        if (!mutations.isEmpty()) {
            this.replayMutations(mutations, writtenAt, version, rateLimiter);
        }
    }

    private void replayMutations(List<Mutation> mutations, long writtenAt, int version, RateLimiter rateLimiter) throws IOException {
        int ttl = this.calculateHintTTL(mutations, writtenAt);
        if (ttl <= 0) {
            return;
        }
        for (Mutation mutation : mutations) {
            ArrayList<InetAddress> liveEndpoints = new ArrayList<InetAddress>();
            ArrayList<InetAddress> hintEndpoints = new ArrayList<InetAddress>();
            String ks = mutation.getKeyspaceName();
            Object tk = StorageService.getPartitioner().getToken(mutation.key());
            int mutationSize = (int)Mutation.serializer.serializedSize(mutation, version);
            for (InetAddress endpoint : Iterables.concat(StorageService.instance.getNaturalEndpoints(ks, (RingPosition)tk), StorageService.instance.getTokenMetadata().pendingEndpointsFor((Token)tk, ks))) {
                rateLimiter.acquire(mutationSize);
                if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                    mutation.apply();
                    continue;
                }
                if (FailureDetector.instance.isAlive(endpoint)) {
                    liveEndpoints.add(endpoint);
                    continue;
                }
                hintEndpoints.add(endpoint);
            }
            if (!liveEndpoints.isEmpty()) {
                hintEndpoints.addAll(this.attemptDirectDelivery(mutation, liveEndpoints));
            }
            for (InetAddress endpoint : hintEndpoints) {
                StorageProxy.writeHintForMutation(mutation, writtenAt, ttl, endpoint);
            }
        }
    }

    private Set<InetAddress> attemptDirectDelivery(Mutation mutation, List<InetAddress> endpoints) throws IOException {
        ArrayList<WriteResponseHandler> handlers = new ArrayList<WriteResponseHandler>();
        final Set<InetAddress> undelivered = Collections.synchronizedSet(new HashSet());
        for (final InetAddress ep : endpoints) {
            Runnable callback = new Runnable(){

                @Override
                public void run() {
                    undelivered.remove(ep);
                }
            };
            WriteResponseHandler handler = new WriteResponseHandler(ep, WriteType.UNLOGGED_BATCH, callback);
            MessagingService.instance().sendRR(mutation.createMessage(), ep, handler, false);
            handlers.add(handler);
        }
        for (WriteResponseHandler handler : handlers) {
            try {
                handler.get();
            }
            catch (WriteTimeoutException e) {
                logger.debug("Timed out replaying a batched mutation to a node, will write a hint");
            }
        }
        return undelivered;
    }

    private int calculateHintTTL(List<Mutation> mutations, long writtenAt) {
        int unadjustedTTL = Integer.MAX_VALUE;
        for (Mutation mutation : mutations) {
            unadjustedTTL = Math.min(unadjustedTTL, HintedHandOffManager.calculateHintTTL(mutation));
        }
        return unadjustedTTL - (int)TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - writtenAt);
    }

    private void cleanup() throws ExecutionException, InterruptedException {
        ColumnFamilyStore cfs = Keyspace.open("system").getColumnFamilyStore("batchlog");
        cfs.forceBlockingFlush();
        ArrayList<Descriptor> descriptors = new ArrayList<Descriptor>();
        for (SSTableReader sstr : cfs.getSSTables()) {
            descriptors.add(sstr.descriptor);
        }
        if (!descriptors.isEmpty()) {
            CompactionManager.instance.submitUserDefined(cfs, descriptors, Integer.MAX_VALUE).get();
        }
    }

    private static UntypedResultSet process(String format, Object ... args) {
        return QueryProcessor.processInternal(String.format(format, args));
    }
}

