/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.Iterables;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.locator.NetworkTopologyStrategy;
import org.apache.cassandra.transport.ProtocolException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public enum ConsistencyLevel {
    ANY(0),
    ONE(1),
    TWO(2),
    THREE(3),
    QUORUM(4),
    ALL(5),
    LOCAL_QUORUM(6, true),
    EACH_QUORUM(7),
    SERIAL(8),
    LOCAL_SERIAL(9),
    LOCAL_ONE(10, true);

    private static final Logger logger;
    public final int code;
    private final boolean isDCLocal;
    private static final ConsistencyLevel[] codeIdx;

    private ConsistencyLevel(int code) {
        this(code, false);
    }

    private ConsistencyLevel(int code, boolean isDCLocal) {
        this.code = code;
        this.isDCLocal = isDCLocal;
    }

    public static ConsistencyLevel fromCode(int code) {
        if (code < 0 || code >= codeIdx.length) {
            throw new ProtocolException(String.format("Unknown code %d for a consistency level", code));
        }
        return codeIdx[code];
    }

    private int quorumFor(Keyspace keyspace) {
        return keyspace.getReplicationStrategy().getReplicationFactor() / 2 + 1;
    }

    private int localQuorumFor(Keyspace keyspace, String dc) {
        return keyspace.getReplicationStrategy() instanceof NetworkTopologyStrategy ? ((NetworkTopologyStrategy)keyspace.getReplicationStrategy()).getReplicationFactor(dc) / 2 + 1 : this.quorumFor(keyspace);
    }

    public int blockFor(Keyspace keyspace) {
        switch (this) {
            case ONE: 
            case LOCAL_ONE: {
                return 1;
            }
            case ANY: {
                return 1;
            }
            case TWO: {
                return 2;
            }
            case THREE: {
                return 3;
            }
            case QUORUM: {
                return this.quorumFor(keyspace);
            }
            case ALL: {
                return keyspace.getReplicationStrategy().getReplicationFactor();
            }
            case LOCAL_QUORUM: {
                return this.localQuorumFor(keyspace, DatabaseDescriptor.getLocalDataCenter());
            }
            case EACH_QUORUM: {
                if (keyspace.getReplicationStrategy() instanceof NetworkTopologyStrategy) {
                    NetworkTopologyStrategy strategy = (NetworkTopologyStrategy)keyspace.getReplicationStrategy();
                    int n = 0;
                    for (String dc : strategy.getDatacenters()) {
                        n += this.localQuorumFor(keyspace, dc);
                    }
                    return n;
                }
                return this.quorumFor(keyspace);
            }
        }
        throw new UnsupportedOperationException("Invalid consistency level: " + this.toString());
    }

    public boolean isDatacenterLocal() {
        return this.isDCLocal;
    }

    private boolean isLocal(InetAddress endpoint) {
        return DatabaseDescriptor.getLocalDataCenter().equals(DatabaseDescriptor.getEndpointSnitch().getDatacenter(endpoint));
    }

    private int countLocalEndpoints(Iterable<InetAddress> liveEndpoints) {
        int count = 0;
        for (InetAddress endpoint : liveEndpoints) {
            if (!this.isLocal(endpoint)) continue;
            ++count;
        }
        return count;
    }

    private Map<String, Integer> countPerDCEndpoints(Keyspace keyspace, Iterable<InetAddress> liveEndpoints) {
        NetworkTopologyStrategy strategy = (NetworkTopologyStrategy)keyspace.getReplicationStrategy();
        HashMap<String, Integer> dcEndpoints = new HashMap<String, Integer>();
        for (String dc : strategy.getDatacenters()) {
            dcEndpoints.put(dc, 0);
        }
        for (InetAddress endpoint : liveEndpoints) {
            String dc = DatabaseDescriptor.getEndpointSnitch().getDatacenter(endpoint);
            dcEndpoints.put(dc, (Integer)dcEndpoints.get(dc) + 1);
        }
        return dcEndpoints;
    }

    public List<InetAddress> filterForQuery(Keyspace keyspace, List<InetAddress> liveEndpoints) {
        return this.filterForQuery(keyspace, liveEndpoints, ReadRepairDecision.NONE);
    }

    public List<InetAddress> filterForQuery(Keyspace keyspace, List<InetAddress> liveEndpoints, ReadRepairDecision readRepair) {
        if (this.isDCLocal) {
            Collections.sort(liveEndpoints, DatabaseDescriptor.getLocalComparator());
        }
        switch (readRepair) {
            case NONE: {
                return liveEndpoints.subList(0, Math.min(liveEndpoints.size(), this.blockFor(keyspace)));
            }
            case GLOBAL: {
                return liveEndpoints;
            }
            case DC_LOCAL: {
                ArrayList<InetAddress> local = new ArrayList<InetAddress>();
                ArrayList<InetAddress> other = new ArrayList<InetAddress>();
                for (InetAddress add : liveEndpoints) {
                    if (this.isLocal(add)) {
                        local.add(add);
                        continue;
                    }
                    other.add(add);
                }
                int blockFor = this.blockFor(keyspace);
                if (local.size() < blockFor) {
                    local.addAll(other.subList(0, Math.min(blockFor - local.size(), other.size())));
                }
                return local;
            }
        }
        throw new AssertionError();
    }

    public boolean isSufficientLiveNodes(Keyspace keyspace, Iterable<InetAddress> liveEndpoints) {
        switch (this) {
            case ANY: {
                return true;
            }
            case LOCAL_ONE: {
                return this.countLocalEndpoints(liveEndpoints) >= 1;
            }
            case LOCAL_QUORUM: {
                return this.countLocalEndpoints(liveEndpoints) >= this.blockFor(keyspace);
            }
            case EACH_QUORUM: {
                if (!(keyspace.getReplicationStrategy() instanceof NetworkTopologyStrategy)) break;
                for (Map.Entry<String, Integer> entry : this.countPerDCEndpoints(keyspace, liveEndpoints).entrySet()) {
                    if (entry.getValue() >= this.localQuorumFor(keyspace, entry.getKey())) continue;
                    return false;
                }
                return true;
            }
        }
        return Iterables.size(liveEndpoints) >= this.blockFor(keyspace);
    }

    public void assureSufficientLiveNodes(Keyspace keyspace, Iterable<InetAddress> liveEndpoints) throws UnavailableException {
        int blockFor = this.blockFor(keyspace);
        switch (this) {
            case ANY: {
                break;
            }
            case LOCAL_QUORUM: {
                int localLive = this.countLocalEndpoints(liveEndpoints);
                if (localLive >= blockFor) break;
                if (logger.isDebugEnabled()) {
                    StringBuilder builder = new StringBuilder("Local replicas [");
                    for (InetAddress endpoint : liveEndpoints) {
                        if (!this.isLocal(endpoint)) continue;
                        builder.append(endpoint).append(",");
                    }
                    builder.append("] are insufficient to satisfy LOCAL_QUORUM requirement of ").append(blockFor).append(" live nodes in '").append(DatabaseDescriptor.getLocalDataCenter()).append("'");
                    logger.debug(builder.toString());
                }
                throw new UnavailableException(this, blockFor, localLive);
            }
            case EACH_QUORUM: {
                if (keyspace.getReplicationStrategy() instanceof NetworkTopologyStrategy) {
                    for (Map.Entry<String, Integer> entry : this.countPerDCEndpoints(keyspace, liveEndpoints).entrySet()) {
                        int dcBlockFor = this.localQuorumFor(keyspace, entry.getKey());
                        int dcLive = entry.getValue();
                        if (dcLive >= dcBlockFor) continue;
                        throw new UnavailableException(this, dcBlockFor, dcLive);
                    }
                    break;
                }
            }
            default: {
                int live = Iterables.size(liveEndpoints);
                if (live >= blockFor) break;
                logger.debug("Live nodes {} do not satisfy ConsistencyLevel ({} required)", (Object)Iterables.toString(liveEndpoints), (Object)blockFor);
                throw new UnavailableException(this, blockFor, live);
            }
        }
    }

    public void validateForRead(String keyspaceName) throws InvalidRequestException {
        switch (this) {
            case ANY: {
                throw new InvalidRequestException("ANY ConsistencyLevel is only supported for writes");
            }
            case EACH_QUORUM: {
                throw new InvalidRequestException("EACH_QUORUM ConsistencyLevel is only supported for writes");
            }
        }
    }

    public void validateForWrite(String keyspaceName) throws InvalidRequestException {
        switch (this) {
            case SERIAL: 
            case LOCAL_SERIAL: {
                throw new InvalidRequestException("You must use conditional updates for serializable writes");
            }
        }
    }

    public void validateForCasCommit(String keyspaceName) throws InvalidRequestException {
        switch (this) {
            case EACH_QUORUM: {
                this.requireNetworkTopologyStrategy(keyspaceName);
                break;
            }
            case SERIAL: 
            case LOCAL_SERIAL: {
                throw new InvalidRequestException((Object)((Object)this) + " is not supported as conditional update commit consistency. Use ANY if you mean \"make sure it is accepted but I don't care how many replicas commit it for non-SERIAL reads\"");
            }
        }
    }

    public void validateForCas() throws InvalidRequestException {
        if (!this.isSerialConsistency()) {
            throw new InvalidRequestException("Invalid consistency for conditional update. Must be one of SERIAL or LOCAL_SERIAL");
        }
    }

    public boolean isSerialConsistency() {
        return this == SERIAL || this == LOCAL_SERIAL;
    }

    public void validateCounterForWrite(CFMetaData metadata) throws InvalidRequestException {
        if (this == ANY) {
            throw new InvalidRequestException("Consistency level ANY is not yet supported for counter columnfamily " + metadata.cfName);
        }
        if (this.isSerialConsistency()) {
            throw new InvalidRequestException("Counter operations are inherently non-serializable");
        }
    }

    private void requireNetworkTopologyStrategy(String keyspaceName) throws InvalidRequestException {
        AbstractReplicationStrategy strategy = Keyspace.open(keyspaceName).getReplicationStrategy();
        if (!(strategy instanceof NetworkTopologyStrategy)) {
            throw new InvalidRequestException(String.format("consistency level %s not compatible with replication strategy (%s)", new Object[]{this, strategy.getClass().getName()}));
        }
    }

    static {
        logger = LoggerFactory.getLogger(ConsistencyLevel.class);
        int maxCode = -1;
        for (ConsistencyLevel cl : ConsistencyLevel.values()) {
            maxCode = Math.max(maxCode, cl.code);
        }
        codeIdx = new ConsistencyLevel[maxCode + 1];
        for (ConsistencyLevel cl : ConsistencyLevel.values()) {
            if (codeIdx[cl.code] != null) {
                throw new IllegalStateException("Duplicate code");
            }
            ConsistencyLevel.codeIdx[cl.code] = cl;
        }
    }
}

