/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ArrayBackedSortedColumns;
import org.apache.cassandra.db.BatchlogManagerMBean;
import org.apache.cassandra.db.Cell;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.HintedHandOffManager;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.composites.CellName;
import org.apache.cassandra.db.marshal.LongType;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.dht.RingPosition;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.FastByteArrayOutputStream;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BatchlogManager
implements BatchlogManagerMBean {
    private static final String MBEAN_NAME = "org.apache.cassandra.db:type=BatchlogManager";
    private static final int VERSION = 6;
    private static final long REPLAY_INTERVAL = 60000L;
    private static final int PAGE_SIZE = 128;
    private static final Logger logger = LoggerFactory.getLogger(BatchlogManager.class);
    public static final BatchlogManager instance = new BatchlogManager();
    private final AtomicLong totalBatchesReplayed = new AtomicLong();
    private final AtomicBoolean isReplaying = new AtomicBoolean();
    private static final ScheduledExecutorService batchlogTasks = new DebuggableScheduledThreadPoolExecutor("BatchlogTasks");

    public void start() {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        batchlogTasks.scheduleWithFixedDelay(runnable, StorageService.RING_DELAY, 60000L, TimeUnit.MILLISECONDS);
    }

    @Override
    public int countAllBatches() {
        return (int)BatchlogManager.process("SELECT count(*) FROM %s.%s", "system", "batchlog").one().getLong("count");
    }

    @Override
    public long getTotalBatchesReplayed() {
        return this.totalBatchesReplayed.longValue();
    }

    @Override
    public void forceBatchlogReplay() {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws ExecutionException, InterruptedException {
                BatchlogManager.this.replayAllFailedBatches();
            }
        };
        batchlogTasks.execute(runnable);
    }

    public static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid) {
        return BatchlogManager.getBatchlogMutationFor(mutations, uuid, FBUtilities.timestampMicros());
    }

    @VisibleForTesting
    static Mutation getBatchlogMutationFor(Collection<Mutation> mutations, UUID uuid, long now) {
        ByteBuffer writtenAt = LongType.instance.decompose(now / 1000L);
        ByteBuffer data = BatchlogManager.serializeMutations(mutations);
        ArrayBackedSortedColumns cf = ArrayBackedSortedColumns.factory.create(CFMetaData.BatchlogCf);
        ((ColumnFamily)cf).addColumn(new Cell(BatchlogManager.cellName(""), ByteBufferUtil.EMPTY_BYTE_BUFFER, now));
        ((ColumnFamily)cf).addColumn(new Cell(BatchlogManager.cellName("data"), data, now));
        ((ColumnFamily)cf).addColumn(new Cell(BatchlogManager.cellName("written_at"), writtenAt, now));
        return new Mutation("system", UUIDType.instance.decompose(uuid), cf);
    }

    private static ByteBuffer serializeMutations(Collection<Mutation> mutations) {
        FastByteArrayOutputStream bos = new FastByteArrayOutputStream();
        DataOutputStream out = new DataOutputStream(bos);
        try {
            out.writeInt(mutations.size());
            for (Mutation mutation : mutations) {
                Mutation.serializer.serialize(mutation, (DataOutput)out, 6);
            }
        }
        catch (IOException e) {
            throw new AssertionError();
        }
        return ByteBuffer.wrap(bos.toByteArray());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void replayAllFailedBatches() throws ExecutionException, InterruptedException {
        if (!this.isReplaying.compareAndSet(false, true)) {
            return;
        }
        logger.debug("Started replayAllFailedBatches");
        int throttleInKB = DatabaseDescriptor.getBatchlogReplayThrottleInKB() / StorageService.instance.getTokenMetadata().getAllEndpoints().size();
        RateLimiter rateLimiter = RateLimiter.create((double)(throttleInKB == 0 ? Double.MAX_VALUE : (double)(throttleInKB * 1024)));
        try {
            UntypedResultSet page = BatchlogManager.process("SELECT id, data, written_at FROM %s.%s LIMIT %d", "system", "batchlog", 128);
            while (!page.isEmpty()) {
                UUID id = this.processBatchlogPage(page, rateLimiter);
                if (page.size() < 128) break;
                page = BatchlogManager.process("SELECT id, data, written_at FROM %s.%s WHERE token(id) > token(%s) LIMIT %d", "system", "batchlog", id, 128);
            }
            this.cleanup();
        }
        finally {
            this.isReplaying.set(false);
        }
        logger.debug("Finished replayAllFailedBatches");
    }

    private UUID processBatchlogPage(UntypedResultSet page, RateLimiter rateLimiter) {
        UUID id = null;
        for (UntypedResultSet.Row row : page) {
            id = row.getUUID("id");
            long writtenAt = row.getLong("written_at");
            long timeout = DatabaseDescriptor.getWriteRpcTimeout() * 2L;
            if (System.currentTimeMillis() < writtenAt + timeout) continue;
            this.replayBatch(id, row.getBytes("data"), writtenAt, rateLimiter);
        }
        return id;
    }

    private void replayBatch(UUID id, ByteBuffer data, long writtenAt, RateLimiter rateLimiter) {
        logger.debug("Replaying batch {}", (Object)id);
        try {
            this.replaySerializedMutations(data, writtenAt, rateLimiter);
        }
        catch (IOException e) {
            logger.warn("Skipped batch replay of {} due to {}", (Object)id, (Object)e);
        }
        this.deleteBatch(id);
        this.totalBatchesReplayed.incrementAndGet();
    }

    private void deleteBatch(UUID id) {
        Mutation mutation = new Mutation("system", UUIDType.instance.decompose(id));
        mutation.delete("batchlog", System.currentTimeMillis());
        mutation.apply();
    }

    private void replaySerializedMutations(ByteBuffer data, long writtenAt, RateLimiter rateLimiter) throws IOException {
        DataInputStream in = new DataInputStream(ByteBufferUtil.inputStream(data));
        int size = in.readInt();
        for (int i = 0; i < size; ++i) {
            this.replaySerializedMutation(Mutation.serializer.deserialize(in, 6), writtenAt, rateLimiter);
        }
    }

    private void replaySerializedMutation(Mutation mutation, long writtenAt, RateLimiter rateLimiter) {
        int ttl = this.calculateHintTTL(mutation, writtenAt);
        if (ttl <= 0) {
            return;
        }
        HashSet<InetAddress> liveEndpoints = new HashSet<InetAddress>();
        String ks = mutation.getKeyspaceName();
        Object tk = StorageService.getPartitioner().getToken(mutation.key());
        int mutationSize = (int)Mutation.serializer.serializedSize(mutation, 6);
        for (InetAddress endpoint : Iterables.concat(StorageService.instance.getNaturalEndpoints(ks, (RingPosition)tk), StorageService.instance.getTokenMetadata().pendingEndpointsFor((Token)tk, ks))) {
            rateLimiter.acquire(mutationSize);
            if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                mutation.apply();
                continue;
            }
            if (FailureDetector.instance.isAlive(endpoint)) {
                liveEndpoints.add(endpoint);
                continue;
            }
            StorageProxy.writeHintForMutation(mutation, ttl, endpoint);
        }
        if (!liveEndpoints.isEmpty()) {
            this.attemptDirectDelivery(mutation, writtenAt, liveEndpoints);
        }
    }

    private void attemptDirectDelivery(Mutation mutation, long writtenAt, Set<InetAddress> endpoints) {
        int ttl;
        ArrayList handlers = Lists.newArrayList();
        final CopyOnWriteArraySet<InetAddress> undelivered = new CopyOnWriteArraySet<InetAddress>(endpoints);
        for (final InetAddress ep : endpoints) {
            Runnable callback = new Runnable(){

                @Override
                public void run() {
                    undelivered.remove(ep);
                }
            };
            WriteResponseHandler handler = new WriteResponseHandler(ep, WriteType.UNLOGGED_BATCH, callback);
            MessagingService.instance().sendRR(mutation.createMessage(), ep, handler);
            handlers.add(handler);
        }
        for (WriteResponseHandler handler : handlers) {
            try {
                handler.get();
            }
            catch (WriteTimeoutException e) {
                logger.debug("Timed out replaying a batched mutation to a node, will write a hint");
            }
        }
        if (!undelivered.isEmpty() && (ttl = this.calculateHintTTL(mutation, writtenAt)) > 0) {
            for (InetAddress endpoint : undelivered) {
                StorageProxy.writeHintForMutation(mutation, ttl, endpoint);
            }
        }
    }

    private int calculateHintTTL(Mutation mutation, long writtenAt) {
        return (int)(((long)(HintedHandOffManager.calculateHintTTL(mutation) * 1000) - (System.currentTimeMillis() - writtenAt)) / 1000L);
    }

    private static CellName cellName(String name) {
        return CFMetaData.BatchlogCf.comparator.makeCellName(name);
    }

    private void cleanup() throws ExecutionException, InterruptedException {
        ColumnFamilyStore cfs = Keyspace.open("system").getColumnFamilyStore("batchlog");
        cfs.forceBlockingFlush();
        ArrayList<Descriptor> descriptors = new ArrayList<Descriptor>();
        for (SSTableReader sstr : cfs.getSSTables()) {
            descriptors.add(sstr.descriptor);
        }
        if (!descriptors.isEmpty()) {
            CompactionManager.instance.submitUserDefined(cfs, descriptors, Integer.MAX_VALUE).get();
        }
    }

    private static UntypedResultSet process(String format, Object ... args) {
        return QueryProcessor.processInternal(String.format(format, args));
    }
}

