/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.collect.Iterables;
import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.UUID;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogAllocator {
    static final Logger logger = LoggerFactory.getLogger(CommitLogAllocator.class);
    public static final int TICK_CYCLE_TIME = 100;
    private final BlockingQueue<CommitLogSegment> availableSegments = new LinkedBlockingQueue<CommitLogSegment>();
    private final BlockingQueue<Runnable> queue = new LinkedBlockingQueue<Runnable>();
    private final ConcurrentLinkedQueue<CommitLogSegment> activeSegments = new ConcurrentLinkedQueue();
    private final AtomicLong size = new AtomicLong();
    private volatile boolean createReserveSegments = false;
    private final Thread allocationThread;
    private volatile boolean run = true;

    public CommitLogAllocator() {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            public void runMayThrow() throws Exception {
                while (CommitLogAllocator.this.run) {
                    try {
                        Runnable r = (Runnable)CommitLogAllocator.this.queue.poll(100L, TimeUnit.MILLISECONDS);
                        if (r != null) {
                            r.run();
                            continue;
                        }
                        if (!CommitLogAllocator.this.availableSegments.isEmpty() || !CommitLogAllocator.this.activeSegments.isEmpty() && !CommitLogAllocator.this.createReserveSegments) continue;
                        logger.debug("No segments in reserve; creating a fresh one");
                        CommitLogAllocator.this.createFreshSegment();
                    }
                    catch (Throwable t) {
                        if (CommitLog.handleCommitError("Failed to allocate new commit log segments", t)) continue;
                        return;
                    }
                }
            }
        };
        this.allocationThread = new Thread((Runnable)runnable, "COMMIT-LOG-ALLOCATOR");
        this.allocationThread.start();
    }

    public CommitLogSegment fetchSegment() {
        CommitLogSegment next;
        try {
            next = this.availableSegments.take();
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
        assert (!this.activeSegments.contains(next));
        this.activeSegments.add(next);
        if (this.isCapExceeded()) {
            this.flushOldestKeyspaces();
        }
        return next;
    }

    public void recycleSegment(final CommitLogSegment segment) {
        this.activeSegments.remove(segment);
        if (!CommitLog.instance.archiver.maybeWaitForArchiving(segment.getName())) {
            this.discardSegment(segment, false);
            return;
        }
        if (this.isCapExceeded()) {
            this.discardSegment(segment, true);
            return;
        }
        logger.debug("Recycling {}", (Object)segment);
        this.queue.add(new Runnable(){

            @Override
            public void run() {
                CommitLogSegment recycled = segment.recycle();
                CommitLogAllocator.this.internalAddReadySegment(recycled);
            }
        });
    }

    public void recycleSegment(final File file) {
        if (this.isCapExceeded() || file.length() != (long)DatabaseDescriptor.getCommitLogSegmentSize() || CommitLogDescriptor.fromFileName(file.getName()).getMessagingVersion() != 7) {
            logger.debug("(Unopened) segment {} is no longer needed and will be deleted now", (Object)file);
            FileUtils.deleteWithConfirm(file);
            return;
        }
        logger.debug("Recycling {}", (Object)file);
        this.size.addAndGet(DatabaseDescriptor.getCommitLogSegmentSize());
        this.queue.add(new Runnable(){

            @Override
            public void run() {
                CommitLogSegment segment = new CommitLogSegment(file.getPath());
                CommitLogAllocator.this.internalAddReadySegment(segment);
            }
        });
    }

    private void discardSegment(final CommitLogSegment segment, final boolean deleteFile) {
        logger.debug("Segment {} is no longer active and will be deleted {}", (Object)segment, (Object)(deleteFile ? "now" : "by the archive script"));
        this.size.addAndGet(-DatabaseDescriptor.getCommitLogSegmentSize());
        this.queue.add(new Runnable(){

            @Override
            public void run() {
                segment.discard(deleteFile);
            }
        });
    }

    public long bytesUsed() {
        return this.size.get();
    }

    public boolean manages(String name) {
        for (CommitLogSegment segment : Iterables.concat(this.activeSegments, this.availableSegments)) {
            if (!segment.getName().equals(name)) continue;
            return true;
        }
        return false;
    }

    private CommitLogSegment createFreshSegment() {
        this.size.addAndGet(DatabaseDescriptor.getCommitLogSegmentSize());
        return this.internalAddReadySegment(CommitLogSegment.freshSegment());
    }

    private CommitLogSegment internalAddReadySegment(CommitLogSegment segment) {
        assert (!this.activeSegments.contains(segment));
        assert (!this.availableSegments.contains(segment));
        this.availableSegments.add(segment);
        return segment;
    }

    private boolean isCapExceeded() {
        long currentSize = this.size.get();
        logger.debug("Total active commitlog segment space used is {}", (Object)currentSize);
        return currentSize > DatabaseDescriptor.getTotalCommitlogSpaceInMB() * 1024L * 1024L;
    }

    public void enableReserveSegmentCreation() {
        this.createReserveSegments = true;
    }

    private void flushOldestKeyspaces() {
        CommitLogSegment oldestSegment = this.activeSegments.peek();
        if (oldestSegment != null) {
            for (UUID dirtyCFId : oldestSegment.getDirtyCFIDs()) {
                Pair<String, String> pair = Schema.instance.getCF(dirtyCFId);
                if (pair == null) {
                    logger.debug("Marking clean CF {} that doesn't exist anymore", (Object)dirtyCFId);
                    oldestSegment.markClean(dirtyCFId, oldestSegment.getContext());
                    continue;
                }
                String keypace = (String)pair.left;
                final ColumnFamilyStore cfs = Keyspace.open(keypace).getColumnFamilyStore(dirtyCFId);
                Runnable runnable = new Runnable(){

                    @Override
                    public void run() {
                        cfs.forceFlush();
                    }
                };
                StorageService.optionalTasks.execute(runnable);
            }
        }
    }

    public void resetUnsafe() {
        logger.debug("Closing and clearing existing commit log segments...");
        while (!this.queue.isEmpty()) {
            Thread.yield();
        }
        for (CommitLogSegment segment : Iterables.concat(this.activeSegments, this.availableSegments)) {
            segment.close();
        }
        this.activeSegments.clear();
        this.availableSegments.clear();
    }

    public void shutdown() {
        this.run = false;
    }

    public void awaitTermination() throws InterruptedException {
        this.allocationThread.join();
    }

    public Collection<CommitLogSegment> getActiveSegments() {
        return Collections.unmodifiableCollection(this.activeSegments);
    }
}

