/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.Longs;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.File;
import java.io.FileFilter;
import java.io.IOError;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BlacklistedDirectories;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableDeletingTask;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Directories {
    private static final Logger logger = LoggerFactory.getLogger(Directories.class);
    public static final String BACKUPS_SUBDIR = "backups";
    public static final String SNAPSHOT_SUBDIR = "snapshots";
    public static final String SECONDARY_INDEX_NAME_SEPARATOR = ".";
    public static final DataDirectory[] dataFileLocations;
    private final String keyspacename;
    private final String cfname;
    private final File[] sstableDirectories;

    public static Directories create(String keyspacename, String cfname) {
        int idx = cfname.indexOf(SECONDARY_INDEX_NAME_SEPARATOR);
        if (idx > 0) {
            return new Directories(keyspacename, cfname, cfname.substring(0, idx));
        }
        return new Directories(keyspacename, cfname, cfname);
    }

    private Directories(String keyspacename, String cfname, String directoryName) {
        this.keyspacename = keyspacename;
        this.cfname = cfname;
        this.sstableDirectories = new File[dataFileLocations.length];
        for (int i = 0; i < dataFileLocations.length; ++i) {
            this.sstableDirectories[i] = new File(Directories.dataFileLocations[i].location, Directories.join(keyspacename, directoryName));
        }
        if (!StorageService.instance.isClientMode()) {
            for (File dir : this.sstableDirectories) {
                try {
                    FileUtils.createDirectory(dir);
                }
                catch (FSError e) {
                    logger.error("Failed to create {} directory", (Object)dir);
                    FileUtils.handleFSError(e);
                }
            }
        }
    }

    public File getLocationForDisk(DataDirectory dataDirectory) {
        for (File dir : this.sstableDirectories) {
            if (!dir.getAbsolutePath().startsWith(dataDirectory.location.getAbsolutePath())) continue;
            return dir;
        }
        return null;
    }

    public File getDirectoryForNewSSTables() {
        File path = this.getWriteableLocationAsFile();
        if (!(path != null || DatabaseDescriptor.getDiskAccessMode() != Config.DiskAccessMode.mmap && DatabaseDescriptor.getIndexAccessMode() != Config.DiskAccessMode.mmap || FileUtils.isCleanerAvailable())) {
            logger.info("Forcing GC to free up disk space.  Upgrade to the Oracle JVM to avoid this");
            StorageService.instance.requestGC();
            SSTableDeletingTask.rescheduleFailedTasks();
            Uninterruptibles.sleepUninterruptibly((long)10L, (TimeUnit)TimeUnit.SECONDS);
            path = this.getWriteableLocationAsFile();
        }
        return path;
    }

    public File getWriteableLocationAsFile() {
        return this.getLocationForDisk(this.getWriteableLocation());
    }

    public DataDirectory getWriteableLocation() {
        ArrayList<DataDirectory> candidates = new ArrayList<DataDirectory>();
        for (DataDirectory dataDir : dataFileLocations) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) continue;
            candidates.add(dataDir);
        }
        if (candidates.isEmpty()) {
            throw new IOError(new IOException("All configured data directories have been blacklisted as unwritable for erroring out"));
        }
        Collections.sort(candidates);
        Collections.sort(candidates, new Comparator<DataDirectory>(){

            @Override
            public int compare(DataDirectory a, DataDirectory b) {
                return a.currentTasks.get() - b.currentTasks.get();
            }
        });
        return (DataDirectory)candidates.get(0);
    }

    public static File getSnapshotDirectory(Descriptor desc, String snapshotName) {
        return Directories.getOrCreate(desc.directory, SNAPSHOT_SUBDIR, snapshotName);
    }

    public static File getBackupsDirectory(Descriptor desc) {
        return Directories.getOrCreate(desc.directory, BACKUPS_SUBDIR);
    }

    public SSTableLister sstableLister() {
        return new SSTableLister();
    }

    @Deprecated
    public File tryGetLeveledManifest() {
        for (File dir : this.sstableDirectories) {
            File manifestFile = new File(dir, this.cfname + ".json");
            if (!manifestFile.exists()) continue;
            logger.debug("Found manifest at {}", (Object)manifestFile);
            return manifestFile;
        }
        logger.debug("No level manifest found");
        return null;
    }

    @Deprecated
    public void snapshotLeveledManifest(String snapshotName) {
        File manifest = this.tryGetLeveledManifest();
        if (manifest != null) {
            File snapshotDirectory = Directories.getOrCreate(manifest.getParentFile(), SNAPSHOT_SUBDIR, snapshotName);
            File target = new File(snapshotDirectory, manifest.getName());
            FileUtils.createHardLink(manifest, target);
        }
    }

    public boolean snapshotExists(String snapshotName) {
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return true;
        }
        return false;
    }

    public static void clearSnapshot(String snapshotName, List<File> snapshotDirectories) {
        String tag = snapshotName == null ? "" : snapshotName;
        for (File dir : snapshotDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, tag));
            if (!snapshotDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Removing snapshot directory " + snapshotDir);
            }
            FileUtils.deleteRecursive(snapshotDir);
        }
    }

    public long snapshotCreationTime(String snapshotName) {
        for (File dir : this.sstableDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return snapshotDir.lastModified();
        }
        throw new RuntimeException("Snapshot " + snapshotName + " doesn't exist");
    }

    public static List<File> getKSChildDirectories(String ksName) {
        ArrayList<File> result = new ArrayList<File>();
        for (DataDirectory dataDirectory : dataFileLocations) {
            File ksDir = new File(dataDirectory.location, ksName);
            File[] cfDirs = ksDir.listFiles();
            if (cfDirs == null) continue;
            for (File cfDir : cfDirs) {
                if (!cfDir.isDirectory()) continue;
                result.add(cfDir);
            }
        }
        return result;
    }

    public List<File> getCFDirectories() {
        ArrayList<File> result = new ArrayList<File>();
        for (File dataDirectory : this.sstableDirectories) {
            if (!dataDirectory.isDirectory()) continue;
            result.add(dataDirectory);
        }
        return result;
    }

    private static File getOrCreate(File base, String ... subdirs) {
        File dir;
        File file = dir = subdirs == null || subdirs.length == 0 ? base : new File(base, Directories.join(subdirs));
        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new AssertionError((Object)String.format("Invalid directory path %s: path exists but is not a directory", dir));
            }
        } else if (!(dir.mkdirs() || dir.exists() && dir.isDirectory())) {
            throw new FSWriteError((Throwable)new IOException("Unable to create directory " + dir), dir);
        }
        return dir;
    }

    private static String join(String ... s) {
        return StringUtils.join((Object[])s, (String)File.separator);
    }

    static void overrideDataDirectoriesForTest(String loc) {
        for (int i = 0; i < dataFileLocations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(loc));
        }
    }

    static void resetDataDirectoriesAfterTest() {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(locations[i]));
        }
    }

    static {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        dataFileLocations = new DataDirectory[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataFileLocations[i] = new DataDirectory(new File(locations[i]));
        }
    }

    public class SSTableLister {
        private boolean skipTemporary;
        private boolean includeBackups;
        private boolean onlyBackups;
        private int nbFiles;
        private final Map<Descriptor, Set<Component>> components = new HashMap<Descriptor, Set<Component>>();
        private boolean filtered;
        private String snapshotName;

        public SSTableLister skipTemporary(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipTemporary = b;
            return this;
        }

        public SSTableLister includeBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.includeBackups = b;
            return this;
        }

        public SSTableLister onlyBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.onlyBackups = b;
            this.includeBackups = b;
            return this;
        }

        public SSTableLister snapshots(String sn) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.snapshotName = sn;
            return this;
        }

        public Map<Descriptor, Set<Component>> list() {
            this.filter();
            return ImmutableMap.copyOf(this.components);
        }

        public List<File> listFiles() {
            this.filter();
            ArrayList<File> l = new ArrayList<File>(this.nbFiles);
            for (Map.Entry<Descriptor, Set<Component>> entry : this.components.entrySet()) {
                for (Component c : entry.getValue()) {
                    l.add(new File(entry.getKey().filenameFor(c)));
                }
            }
            return l;
        }

        private void filter() {
            if (this.filtered) {
                return;
            }
            for (File location : Directories.this.sstableDirectories) {
                if (BlacklistedDirectories.isUnreadable(location)) continue;
                if (this.snapshotName != null) {
                    new File(location, Directories.join(new String[]{Directories.SNAPSHOT_SUBDIR, this.snapshotName})).listFiles(this.getFilter());
                    continue;
                }
                if (!this.onlyBackups) {
                    location.listFiles(this.getFilter());
                }
                if (!this.includeBackups) continue;
                new File(location, Directories.BACKUPS_SUBDIR).listFiles(this.getFilter());
            }
            this.filtered = true;
        }

        private FileFilter getFilter() {
            final String sstablePrefix = Directories.this.keyspacename + '-' + Directories.this.cfname + '-';
            return new FileFilter(){

                @Override
                public boolean accept(File file) {
                    if (file.isDirectory() || !file.getName().startsWith(sstablePrefix)) {
                        return false;
                    }
                    Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                    if (pair == null) {
                        return false;
                    }
                    if (SSTableLister.this.skipTemporary && ((Descriptor)pair.left).temporary) {
                        return false;
                    }
                    HashSet previous = (HashSet)SSTableLister.this.components.get(pair.left);
                    if (previous == null) {
                        previous = new HashSet();
                        SSTableLister.this.components.put(pair.left, previous);
                    }
                    previous.add(pair.right);
                    SSTableLister.this.nbFiles++;
                    return false;
                }
            };
        }
    }

    public static class DataDirectory
    implements Comparable<DataDirectory> {
        public final File location;
        public final AtomicInteger currentTasks = new AtomicInteger();
        public final AtomicLong estimatedWorkingSize = new AtomicLong();

        public DataDirectory(File location) {
            this.location = location;
        }

        public long getEstimatedAvailableSpace() {
            return this.location.getUsableSpace() - this.estimatedWorkingSize.get();
        }

        @Override
        public int compareTo(DataDirectory o) {
            return -1 * Longs.compare((long)this.getEstimatedAvailableSpace(), (long)o.getEstimatedAvailableSpace());
        }
    }
}

