/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.cassandra.db;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;

import org.apache.cassandra.io.ISerializer;
import org.apache.cassandra.utils.ObjectSizes;

/**
 * A top-level (row) tombstone.
 */
public class DeletionTime implements Comparable<DeletionTime>
{
    /**
     * A special DeletionTime that signifies that there is no top-level (row) tombstone.
     */
    public static final DeletionTime LIVE = new DeletionTime(Long.MIN_VALUE, Integer.MAX_VALUE);

    /**
     * A timestamp (typically in microseconds since the unix epoch, although this is not enforced) after which
     * data should be considered deleted. If set to Long.MIN_VALUE, this implies that the data has not been marked
     * for deletion at all.
     */
    public final long markedForDeleteAt;

    /**
     * The local server timestamp, in seconds since the unix epoch, at which this tombstone was created. This is
     * only used for purposes of purging the tombstone after gc_grace_seconds have elapsed.
     */
    public final int localDeletionTime;

    public static final ISerializer<DeletionTime> serializer = new Serializer();

    @VisibleForTesting
    public DeletionTime(long markedForDeleteAt, int localDeletionTime)
    {
        this.markedForDeleteAt = markedForDeleteAt;
        this.localDeletionTime = localDeletionTime;
    }

    @Override
    public boolean equals(Object o)
    {
        if(!(o instanceof DeletionTime))
            return false;
        DeletionTime that = (DeletionTime)o;
        return markedForDeleteAt == that.markedForDeleteAt && localDeletionTime == that.localDeletionTime;
    }

    @Override
    public final int hashCode()
    {
        return Objects.hashCode(markedForDeleteAt, localDeletionTime);
    }

    @Override
    public String toString()
    {
        return String.format("deletedAt=%d, localDeletion=%d", markedForDeleteAt, localDeletionTime);
    }

    public int compareTo(DeletionTime dt)
    {
        if (markedForDeleteAt < dt.markedForDeleteAt)
            return -1;
        else if (markedForDeleteAt > dt.markedForDeleteAt)
            return 1;
        else if (localDeletionTime < dt.localDeletionTime)
            return -1;
        else if (localDeletionTime > dt.localDeletionTime)
            return -1;
        else
            return 0;
    }

    public boolean isGcAble(int gcBefore)
    {
        return localDeletionTime < gcBefore;
    }

    public boolean isDeleted(Column column)
    {
        return column.timestamp() <= markedForDeleteAt;
    }

    public long memorySize()
    {
        long fields = TypeSizes.NATIVE.sizeof(markedForDeleteAt) + TypeSizes.NATIVE.sizeof(localDeletionTime);
        return ObjectSizes.getFieldSize(fields);
    }

    private static class Serializer implements ISerializer<DeletionTime>
    {
        public void serialize(DeletionTime delTime, DataOutput out) throws IOException
        {
            out.writeInt(delTime.localDeletionTime);
            out.writeLong(delTime.markedForDeleteAt);
        }

        public DeletionTime deserialize(DataInput in) throws IOException
        {
            int ldt = in.readInt();
            long mfda = in.readLong();
            if (mfda == Long.MIN_VALUE && ldt == Integer.MAX_VALUE)
                return LIVE;
            else
                return new DeletionTime(mfda, ldt);
        }

        public long serializedSize(DeletionTime delTime, TypeSizes typeSizes)
        {
            return typeSizes.sizeof(delTime.localDeletionTime)
                 + typeSizes.sizeof(delTime.markedForDeleteAt);
        }
    }
}
