/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.collect.Iterables;
import com.google.common.primitives.Longs;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionTask;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SizeTieredCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(SizeTieredCompactionStrategy.class);
    protected SizeTieredCompactionStrategyOptions options;
    protected volatile int estimatedRemainingTasks = 0;

    public SizeTieredCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        this.options = new SizeTieredCompactionStrategyOptions(options);
    }

    private List<SSTableReader> getNextBackgroundSSTables(int gcBefore) {
        if (!this.isEnabled()) {
            return Collections.emptyList();
        }
        int minThreshold = this.cfs.getMinimumCompactionThreshold();
        int maxThreshold = this.cfs.getMaximumCompactionThreshold();
        Set<SSTableReader> candidates = this.cfs.getUncompactingSSTables();
        List<List<SSTableReader>> buckets = SizeTieredCompactionStrategy.getBuckets(SizeTieredCompactionStrategy.createSSTableAndLengthPairs(SizeTieredCompactionStrategy.filterSuspectSSTables(candidates)), this.options.bucketHigh, this.options.bucketLow, this.options.minSSTableSize);
        logger.debug("Compaction buckets are {}", buckets);
        this.updateEstimatedCompactionsByTasks(buckets);
        List<SSTableReader> mostInteresting = SizeTieredCompactionStrategy.mostInterestingBucket(buckets, minThreshold, maxThreshold);
        if (!mostInteresting.isEmpty()) {
            return mostInteresting;
        }
        ArrayList<SSTableReader> sstablesWithTombstones = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : candidates) {
            if (!this.worthDroppingTombstones(sstable, gcBefore)) continue;
            sstablesWithTombstones.add(sstable);
        }
        if (sstablesWithTombstones.isEmpty()) {
            return Collections.emptyList();
        }
        Collections.sort(sstablesWithTombstones, new SSTableReader.SizeComparator());
        return Collections.singletonList(sstablesWithTombstones.get(0));
    }

    public static List<SSTableReader> mostInterestingBucket(List<List<SSTableReader>> buckets, int minThreshold, int maxThreshold) {
        ArrayList<List<SSTableReader>> prunedBuckets = new ArrayList<List<SSTableReader>>();
        for (List<SSTableReader> bucket : buckets) {
            if (bucket.size() < minThreshold) continue;
            Collections.sort(bucket, new Comparator<SSTableReader>(){

                @Override
                public int compare(SSTableReader o1, SSTableReader o2) {
                    return o1.descriptor.generation - o2.descriptor.generation;
                }
            });
            List<SSTableReader> prunedBucket = bucket.subList(0, Math.min(bucket.size(), maxThreshold));
            prunedBuckets.add(prunedBucket);
        }
        if (prunedBuckets.isEmpty()) {
            return Collections.emptyList();
        }
        return Collections.min(prunedBuckets, new Comparator<List<SSTableReader>>(){

            @Override
            public int compare(List<SSTableReader> o1, List<SSTableReader> o2) {
                return Longs.compare((long)this.avgSize(o1), (long)this.avgSize(o2));
            }

            private long avgSize(List<SSTableReader> sstables) {
                long n = 0L;
                for (SSTableReader sstable : sstables) {
                    n += sstable.bytesOnDisk();
                }
                return n / (long)sstables.size();
            }
        });
    }

    @Override
    public synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        List<SSTableReader> smallestBucket;
        if (!this.isEnabled()) {
            return null;
        }
        do {
            if (!(smallestBucket = this.getNextBackgroundSSTables(gcBefore)).isEmpty()) continue;
            return null;
        } while (!this.cfs.getDataTracker().markCompacting(smallestBucket));
        return new CompactionTask(this.cfs, smallestBucket, gcBefore);
    }

    @Override
    public AbstractCompactionTask getMaximalTask(int gcBefore) {
        Iterable<SSTableReader> sstables = this.cfs.markAllCompacting();
        if (sstables == null) {
            return null;
        }
        return new CompactionTask(this.cfs, sstables, gcBefore);
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        assert (!sstables.isEmpty());
        if (!this.cfs.getDataTracker().markCompacting(sstables)) {
            logger.debug("Unable to mark {} for compaction; probably a background compaction got to it first.  You can disable background compactions temporarily if this is a problem", sstables);
            return null;
        }
        return new CompactionTask(this.cfs, sstables, gcBefore).setUserDefined(true);
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.estimatedRemainingTasks;
    }

    public static List<Pair<SSTableReader, Long>> createSSTableAndLengthPairs(Iterable<SSTableReader> sstables) {
        ArrayList<Pair<SSTableReader, Long>> sstableLengthPairs = new ArrayList<Pair<SSTableReader, Long>>(Iterables.size(sstables));
        for (SSTableReader sstable : sstables) {
            sstableLengthPairs.add(Pair.create(sstable, sstable.onDiskLength()));
        }
        return sstableLengthPairs;
    }

    public static <T> List<List<T>> getBuckets(Collection<Pair<T, Long>> files, double bucketHigh, double bucketLow, long minSSTableSize) {
        ArrayList<Pair<T, Long>> sortedFiles = new ArrayList<Pair<T, Long>>(files);
        Collections.sort(sortedFiles, new Comparator<Pair<T, Long>>(){

            @Override
            public int compare(Pair<T, Long> p1, Pair<T, Long> p2) {
                return ((Long)p1.right).compareTo((Long)p2.right);
            }
        });
        HashMap<Long, List> buckets = new HashMap<Long, List>();
        block0: for (Pair pair : sortedFiles) {
            long size = (Long)pair.right;
            for (Map.Entry entry : buckets.entrySet()) {
                List bucket = (List)entry.getValue();
                long oldAverageSize = (Long)entry.getKey();
                if (!((double)size > (double)oldAverageSize * bucketLow && (double)size < (double)oldAverageSize * bucketHigh) && (size >= minSSTableSize || oldAverageSize >= minSSTableSize)) continue;
                buckets.remove(oldAverageSize);
                long totalSize = (long)bucket.size() * oldAverageSize;
                long newAverageSize = (totalSize + size) / (long)(bucket.size() + 1);
                bucket.add(pair.left);
                buckets.put(newAverageSize, bucket);
                continue block0;
            }
            ArrayList bucket = new ArrayList();
            bucket.add(pair.left);
            buckets.put(size, bucket);
        }
        return new ArrayList<List<T>>(buckets.values());
    }

    private void updateEstimatedCompactionsByTasks(List<List<SSTableReader>> tasks) {
        int n = 0;
        for (List<SSTableReader> bucket : tasks) {
            if (bucket.size() < this.cfs.getMinimumCompactionThreshold()) continue;
            n = (int)((double)n + Math.ceil((double)bucket.size() / (double)this.cfs.getMaximumCompactionThreshold()));
        }
        this.estimatedRemainingTasks = n;
    }

    @Override
    public long getMaxSSTableSize() {
        return Long.MAX_VALUE;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        uncheckedOptions = SizeTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        uncheckedOptions.remove("min_threshold");
        uncheckedOptions.remove("max_threshold");
        return uncheckedOptions;
    }

    public String toString() {
        return String.format("SizeTieredCompactionStrategy[%s/%s]", this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }
}

