/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import java.net.InetAddress;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.ReadResponse;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.metrics.ReadRepairMetrics;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.DigestMismatchException;
import org.apache.cassandra.service.ReadCallback;
import org.apache.cassandra.service.RowDigestResolver;
import org.apache.cassandra.service.StorageProxy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractReadExecutor {
    private static final Logger logger = LoggerFactory.getLogger(AbstractReadExecutor.class);
    protected final ReadCallback<ReadResponse, Row> handler;
    protected final ReadCommand command;
    protected final RowDigestResolver resolver;
    protected final List<InetAddress> unfiltered;
    protected final List<InetAddress> endpoints;
    protected final ColumnFamilyStore cfs;

    AbstractReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistency_level, List<InetAddress> allReplicas, List<InetAddress> queryTargets) throws UnavailableException {
        this.unfiltered = allReplicas;
        this.endpoints = queryTargets;
        this.resolver = new RowDigestResolver(command.ksName, command.key);
        this.handler = new ReadCallback<ReadResponse, Row>(this.resolver, consistency_level, command, this.endpoints);
        this.command = command;
        this.cfs = cfs;
        this.handler.assureSufficientLiveNodes();
        assert (!this.handler.endpoints.isEmpty());
    }

    void executeAsync() {
        InetAddress dataPoint = this.handler.endpoints.get(0);
        if (dataPoint.equals(FBUtilities.getBroadcastAddress())) {
            logger.trace("reading data locally");
            StageManager.getStage(Stage.READ).execute(new StorageProxy.LocalReadRunnable(this.command, this.handler));
        } else {
            logger.trace("reading data from {}", (Object)dataPoint);
            MessagingService.instance().sendRR(this.command.createMessage(), dataPoint, this.handler);
        }
        if (this.handler.endpoints.size() == 1) {
            return;
        }
        ReadCommand digestCommand = this.command.copy();
        digestCommand.setDigestQuery(true);
        MessageOut<ReadCommand> message = null;
        for (int i = 1; i < this.handler.endpoints.size(); ++i) {
            InetAddress digestPoint = this.handler.endpoints.get(i);
            if (digestPoint.equals(FBUtilities.getBroadcastAddress())) {
                logger.trace("reading digest locally");
                StageManager.getStage(Stage.READ).execute(new StorageProxy.LocalReadRunnable(digestCommand, this.handler));
                continue;
            }
            logger.trace("reading digest from {}", (Object)digestPoint);
            if (message == null) {
                message = digestCommand.createMessage();
            }
            MessagingService.instance().sendRR(message, digestPoint, this.handler);
        }
    }

    void speculate() {
    }

    Row get() throws ReadTimeoutException, DigestMismatchException {
        return this.handler.get();
    }

    public static AbstractReadExecutor getReadExecutor(ReadCommand command, ConsistencyLevel consistency_level) throws UnavailableException {
        Keyspace keyspace = Keyspace.open(command.ksName);
        List<InetAddress> allReplicas = StorageProxy.getLiveSortedEndpoints(keyspace, command.key);
        CFMetaData metaData = Schema.instance.getCFMetaData(command.ksName, command.cfName);
        ReadRepairDecision rrDecision = metaData.newReadRepairDecision();
        if (rrDecision != ReadRepairDecision.NONE) {
            ReadRepairMetrics.attempted.mark();
        }
        List<InetAddress> queryTargets = consistency_level.filterForQuery(keyspace, allReplicas, rrDecision);
        if (StorageService.instance.isClientMode()) {
            return new DefaultReadExecutor(null, command, consistency_level, allReplicas, queryTargets);
        }
        ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(command.cfName);
        switch (metaData.getSpeculativeRetry().type) {
            case ALWAYS: {
                return new SpeculateAlwaysExecutor(cfs, command, consistency_level, allReplicas, queryTargets);
            }
            case PERCENTILE: 
            case CUSTOM: {
                return queryTargets.size() < allReplicas.size() ? new SpeculativeReadExecutor(cfs, command, consistency_level, allReplicas, queryTargets) : new DefaultReadExecutor(cfs, command, consistency_level, allReplicas, queryTargets);
            }
        }
        return new DefaultReadExecutor(cfs, command, consistency_level, allReplicas, queryTargets);
    }

    private static class SpeculateAlwaysExecutor
    extends AbstractReadExecutor {
        public SpeculateAlwaysExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistency_level, List<InetAddress> allReplicas, List<InetAddress> queryTargets) throws UnavailableException {
            super(cfs, command, consistency_level, allReplicas, queryTargets);
        }

        @Override
        void executeAsync() {
            int limit = this.unfiltered.size() >= 2 ? 2 : 1;
            for (int i = 0; i < limit; ++i) {
                InetAddress endpoint = (InetAddress)this.unfiltered.get(i);
                if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                    logger.trace("reading full data locally");
                    StageManager.getStage(Stage.READ).execute(new StorageProxy.LocalReadRunnable(this.command, this.handler));
                    continue;
                }
                logger.trace("reading full data from {}", (Object)endpoint);
                MessagingService.instance().sendRR(this.command.createMessage(), endpoint, this.handler);
            }
            if (this.handler.endpoints.size() <= limit) {
                return;
            }
            ReadCommand digestCommand = this.command.copy();
            digestCommand.setDigestQuery(true);
            MessageOut<ReadCommand> message = digestCommand.createMessage();
            for (int i = limit; i < this.handler.endpoints.size(); ++i) {
                InetAddress endpoint = this.handler.endpoints.get(i);
                if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
                    logger.trace("reading data locally, isDigest: {}", (Object)this.command.isDigestQuery());
                    StageManager.getStage(Stage.READ).execute(new StorageProxy.LocalReadRunnable(digestCommand, this.handler));
                    continue;
                }
                logger.trace("reading full data from {}, isDigest: {}", (Object)endpoint, (Object)this.command.isDigestQuery());
                MessagingService.instance().sendRR(message, endpoint, this.handler);
            }
        }
    }

    private static class SpeculativeReadExecutor
    extends AbstractReadExecutor {
        public SpeculativeReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistency_level, List<InetAddress> allReplicas, List<InetAddress> queryTargets) throws UnavailableException {
            super(cfs, command, consistency_level, allReplicas, queryTargets);
            assert (this.handler.endpoints.size() < this.unfiltered.size());
        }

        @Override
        void speculate() {
            if (this.cfs.sampleLatency > TimeUnit.MILLISECONDS.toNanos(this.command.getTimeout())) {
                return;
            }
            if (!this.handler.await(this.cfs.sampleLatency, TimeUnit.NANOSECONDS)) {
                InetAddress endpoint = (InetAddress)this.unfiltered.get(this.handler.endpoints.size());
                ReadCommand scommand = this.command;
                if (this.resolver.getData() != null) {
                    scommand = this.command.copy();
                    scommand.setDigestQuery(true);
                }
                logger.trace("Speculating read retry on {}", (Object)endpoint);
                MessagingService.instance().sendRR(scommand.createMessage(), endpoint, this.handler);
                this.cfs.metric.speculativeRetry.inc();
            }
        }
    }

    private static class DefaultReadExecutor
    extends AbstractReadExecutor {
        public DefaultReadExecutor(ColumnFamilyStore cfs, ReadCommand command, ConsistencyLevel consistency_level, List<InetAddress> allReplicas, List<InetAddress> queryTargets) throws UnavailableException {
            super(cfs, command, consistency_level, allReplicas, queryTargets);
        }
    }
}

