/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ReadRepairDecision;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.BatchlogManager;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.CounterMutation;
import org.apache.cassandra.db.HintedHandOffManager;
import org.apache.cassandra.db.IMutation;
import org.apache.cassandra.db.RangeSliceCommand;
import org.apache.cassandra.db.RangeSliceReply;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.ReadResponse;
import org.apache.cassandra.db.ReadVerbHandler;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.Table;
import org.apache.cassandra.db.Truncation;
import org.apache.cassandra.db.WriteType;
import org.apache.cassandra.db.filter.ColumnSlice;
import org.apache.cassandra.db.filter.IDiskAtomFilter;
import org.apache.cassandra.db.filter.QueryPath;
import org.apache.cassandra.db.filter.SliceQueryFilter;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.RingPosition;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.IsBootstrappingException;
import org.apache.cassandra.exceptions.OverloadedException;
import org.apache.cassandra.exceptions.ReadTimeoutException;
import org.apache.cassandra.exceptions.UnavailableException;
import org.apache.cassandra.exceptions.WriteTimeoutException;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.io.util.FastByteArrayOutputStream;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.locator.IEndpointSnitch;
import org.apache.cassandra.locator.TokenMetadata;
import org.apache.cassandra.metrics.ClientRequestMetrics;
import org.apache.cassandra.metrics.ReadRepairMetrics;
import org.apache.cassandra.metrics.StorageMetrics;
import org.apache.cassandra.net.CompactEndpointSerializationHelper;
import org.apache.cassandra.net.IAsyncCallback;
import org.apache.cassandra.net.MessageIn;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.AbstractRowResolver;
import org.apache.cassandra.service.AbstractWriteResponseHandler;
import org.apache.cassandra.service.DigestMismatchException;
import org.apache.cassandra.service.RangeSliceResponseResolver;
import org.apache.cassandra.service.RangeSliceVerbHandler;
import org.apache.cassandra.service.ReadCallback;
import org.apache.cassandra.service.RowDataResolver;
import org.apache.cassandra.service.RowDigestResolver;
import org.apache.cassandra.service.StorageProxyMBean;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.service.TruncateResponseHandler;
import org.apache.cassandra.service.WriteResponseHandler;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageProxy
implements StorageProxyMBean {
    public static final String MBEAN_NAME = "org.apache.cassandra.db:type=StorageProxy";
    private static final Logger logger = LoggerFactory.getLogger(StorageProxy.class);
    private static final boolean OPTIMIZE_LOCAL_REQUESTS = true;
    public static final String UNREACHABLE = "UNREACHABLE";
    private static final WritePerformer standardWritePerformer;
    private static final WritePerformer counterWritePerformer;
    private static final WritePerformer counterWriteOnCoordinatorPerformer;
    public static final StorageProxy instance;
    private static volatile int maxHintsInProgress;
    private static final CacheLoader<InetAddress, AtomicInteger> hintsInProgress;
    private static final ClientRequestMetrics readMetrics;
    private static final ClientRequestMetrics rangeMetrics;
    private static final ClientRequestMetrics writeMetrics;

    private StorageProxy() {
    }

    public static void mutate(Collection<? extends IMutation> mutations, ConsistencyLevel consistency_level) throws UnavailableException, OverloadedException, WriteTimeoutException {
        Tracing.trace("Determining replicas for mutation");
        String localDataCenter = DatabaseDescriptor.getEndpointSnitch().getDatacenter(FBUtilities.getBroadcastAddress());
        long startTime = System.nanoTime();
        ArrayList<AbstractWriteResponseHandler> responseHandlers = new ArrayList<AbstractWriteResponseHandler>(mutations.size());
        IMutation mostRecentMutation = null;
        try {
            Iterator<IMutation> i$ = mutations.iterator();
            while (i$.hasNext()) {
                IMutation mutation;
                mostRecentMutation = mutation = i$.next();
                if (mutation instanceof CounterMutation) {
                    responseHandlers.add(StorageProxy.mutateCounter((CounterMutation)mutation, localDataCenter));
                    continue;
                }
                WriteType wt = mutations.size() <= 1 ? WriteType.SIMPLE : WriteType.UNLOGGED_BATCH;
                responseHandlers.add(StorageProxy.performWrite(mutation, consistency_level, localDataCenter, standardWritePerformer, null, wt));
            }
            for (AbstractWriteResponseHandler responseHandler : responseHandlers) {
                responseHandler.get();
            }
        }
        catch (WriteTimeoutException ex) {
            StorageProxy.writeMetrics.timeouts.mark();
            ClientRequestMetrics.writeTimeouts.inc();
            Tracing.trace("Write timeout; received {} of {} required replies", ex.received, ex.blockFor);
            throw ex;
        }
        catch (UnavailableException e) {
            StorageProxy.writeMetrics.unavailables.mark();
            ClientRequestMetrics.writeUnavailables.inc();
            Tracing.trace("Unavailable");
            throw e;
        }
        catch (OverloadedException e) {
            ClientRequestMetrics.writeUnavailables.inc();
            Tracing.trace("Overloaded");
            throw e;
        }
        catch (IOException e) {
            assert (mostRecentMutation != null);
            throw new RuntimeException("error writing key " + ByteBufferUtil.bytesToHex(mostRecentMutation.key()), e);
        }
        finally {
            writeMetrics.addNano(System.nanoTime() - startTime);
        }
    }

    public static void mutateAtomically(Collection<RowMutation> mutations, ConsistencyLevel consistency_level) throws UnavailableException, OverloadedException, WriteTimeoutException {
        Tracing.trace("Determining replicas for atomic batch");
        long startTime = System.nanoTime();
        ArrayList<WriteResponseHandlerWrapper> wrappers = new ArrayList<WriteResponseHandlerWrapper>(mutations.size());
        String localDataCenter = DatabaseDescriptor.getEndpointSnitch().getDatacenter(FBUtilities.getBroadcastAddress());
        try {
            for (RowMutation mutation : mutations) {
                WriteResponseHandlerWrapper wrapper = StorageProxy.wrapResponseHandler(mutation, consistency_level, WriteType.BATCH);
                wrapper.handler.assureSufficientLiveNodes();
                wrappers.add(wrapper);
            }
            Collection<InetAddress> batchlogEndpoints = StorageProxy.getBatchlogEndpoints(localDataCenter, consistency_level);
            UUID batchUUID = UUID.randomUUID();
            StorageProxy.syncWriteToBatchlog(mutations, batchlogEndpoints, batchUUID);
            StorageProxy.syncWriteBatchedMutations(wrappers, localDataCenter, consistency_level);
            StorageProxy.asyncRemoveFromBatchlog(batchlogEndpoints, batchUUID);
        }
        catch (UnavailableException e) {
            StorageProxy.writeMetrics.unavailables.mark();
            ClientRequestMetrics.writeUnavailables.inc();
            Tracing.trace("Unavailable");
            throw e;
        }
        catch (WriteTimeoutException e) {
            StorageProxy.writeMetrics.timeouts.mark();
            ClientRequestMetrics.writeTimeouts.inc();
            Tracing.trace("Write timeout; received {} of {} required replies", e.received, e.blockFor);
            throw e;
        }
        finally {
            writeMetrics.addNano(System.nanoTime() - startTime);
        }
    }

    private static void syncWriteToBatchlog(Collection<RowMutation> mutations, Collection<InetAddress> endpoints, UUID uuid) throws WriteTimeoutException {
        RowMutation rm = BatchlogManager.getBatchlogMutationFor(mutations, uuid);
        WriteResponseHandler handler = new WriteResponseHandler(endpoints, Collections.<InetAddress>emptyList(), ConsistencyLevel.ONE, Table.open("system"), null, WriteType.BATCH_LOG);
        StorageProxy.updateBatchlog(rm, endpoints, handler);
        handler.get();
    }

    private static void asyncRemoveFromBatchlog(Collection<InetAddress> endpoints, UUID uuid) {
        RowMutation rm = new RowMutation("system", UUIDType.instance.decompose(uuid));
        rm.delete(new QueryPath("batchlog"), FBUtilities.timestampMicros());
        WriteResponseHandler handler = new WriteResponseHandler(endpoints, Collections.<InetAddress>emptyList(), ConsistencyLevel.ANY, Table.open("system"), null, WriteType.SIMPLE);
        StorageProxy.updateBatchlog(rm, endpoints, handler);
    }

    private static void updateBatchlog(RowMutation rm, Collection<InetAddress> endpoints, AbstractWriteResponseHandler handler) {
        if (endpoints.contains(FBUtilities.getBroadcastAddress())) {
            assert (endpoints.size() == 1);
            StorageProxy.insertLocal(rm, handler);
        } else {
            StorageProxy.sendMessagesToOneDC(rm.createMessage(), endpoints, true, handler);
        }
    }

    private static void syncWriteBatchedMutations(List<WriteResponseHandlerWrapper> wrappers, String localDataCenter, ConsistencyLevel consistencyLevel) throws WriteTimeoutException, OverloadedException {
        for (WriteResponseHandlerWrapper wrapper : wrappers) {
            Iterable endpoints = Iterables.concat(wrapper.handler.naturalEndpoints, wrapper.handler.pendingEndpoints);
            StorageProxy.sendToHintedEndpoints(wrapper.mutation, endpoints, wrapper.handler, localDataCenter, consistencyLevel);
        }
        for (WriteResponseHandlerWrapper wrapper : wrappers) {
            wrapper.handler.get();
        }
    }

    public static AbstractWriteResponseHandler performWrite(IMutation mutation, ConsistencyLevel consistency_level, String localDataCenter, WritePerformer performer, Runnable callback, WriteType writeType) throws UnavailableException, OverloadedException, IOException {
        String table = mutation.getTable();
        AbstractReplicationStrategy rs = Table.open(table).getReplicationStrategy();
        Object tk = StorageService.getPartitioner().getToken(mutation.key());
        List<InetAddress> naturalEndpoints = StorageService.instance.getNaturalEndpoints(table, (RingPosition)tk);
        Collection<InetAddress> pendingEndpoints = StorageService.instance.getTokenMetadata().pendingEndpointsFor((Token)tk, table);
        AbstractWriteResponseHandler responseHandler = rs.getWriteResponseHandler(naturalEndpoints, pendingEndpoints, consistency_level, callback, writeType);
        responseHandler.assureSufficientLiveNodes();
        performer.apply(mutation, Iterables.concat(naturalEndpoints, pendingEndpoints), responseHandler, localDataCenter, consistency_level);
        return responseHandler;
    }

    private static WriteResponseHandlerWrapper wrapResponseHandler(RowMutation mutation, ConsistencyLevel consistency_level, WriteType writeType) {
        AbstractReplicationStrategy rs = Table.open(mutation.getTable()).getReplicationStrategy();
        String table = mutation.getTable();
        Object tk = StorageService.getPartitioner().getToken(mutation.key());
        List<InetAddress> naturalEndpoints = StorageService.instance.getNaturalEndpoints(table, (RingPosition)tk);
        Collection<InetAddress> pendingEndpoints = StorageService.instance.getTokenMetadata().pendingEndpointsFor((Token)tk, table);
        AbstractWriteResponseHandler responseHandler = rs.getWriteResponseHandler(naturalEndpoints, pendingEndpoints, consistency_level, null, writeType);
        return new WriteResponseHandlerWrapper(responseHandler, mutation);
    }

    private static Collection<InetAddress> getBatchlogEndpoints(String localDataCenter, ConsistencyLevel consistencyLevel) throws UnavailableException {
        TokenMetadata.Topology topology = StorageService.instance.getTokenMetadata().cloneOnlyTokenMap().getTopology();
        Collection localMembers = topology.getDatacenterEndpoints().get((Object)localDataCenter);
        if (localMembers.size() == 1) {
            return localMembers;
        }
        localMembers.remove(FBUtilities.getBroadcastAddress());
        List<InetAddress> candidates = new ArrayList<InetAddress>(localMembers.size());
        for (InetAddress member : localMembers) {
            if (!FailureDetector.instance.isAlive(member)) continue;
            candidates.add(member);
        }
        if (candidates.isEmpty()) {
            if (consistencyLevel == ConsistencyLevel.ANY) {
                return Collections.singleton(FBUtilities.getBroadcastAddress());
            }
            throw new UnavailableException(ConsistencyLevel.ONE, 1, 0);
        }
        if (candidates.size() > 2) {
            IEndpointSnitch snitch = DatabaseDescriptor.getEndpointSnitch();
            snitch.sortByProximity(FBUtilities.getBroadcastAddress(), candidates);
            candidates = candidates.subList(0, 2);
        }
        return candidates;
    }

    public static void sendToHintedEndpoints(RowMutation rm, Iterable<InetAddress> targets, AbstractWriteResponseHandler responseHandler, String localDataCenter, ConsistencyLevel consistency_level) throws OverloadedException {
        HashMap dcGroups = null;
        for (InetAddress destination : targets) {
            if (StorageMetrics.totalHintsInProgress.count() > (long)maxHintsInProgress && StorageProxy.getHintsInProgressFor(destination).get() > 0 && StorageProxy.shouldHint(destination)) {
                throw new OverloadedException("Too many in flight hints: " + StorageMetrics.totalHintsInProgress.count());
            }
            if (FailureDetector.instance.isAlive(destination)) {
                ArrayList<InetAddress> dcTargets;
                if (destination.equals(FBUtilities.getBroadcastAddress())) {
                    StorageProxy.insertLocal(rm, responseHandler);
                    continue;
                }
                String dc = DatabaseDescriptor.getEndpointSnitch().getDatacenter(destination);
                ArrayList<InetAddress> arrayList = dcTargets = dcGroups != null ? (ArrayList<InetAddress>)dcGroups.get(dc) : null;
                if (dcTargets == null) {
                    dcTargets = new ArrayList<InetAddress>(3);
                    if (dcGroups == null) {
                        dcGroups = new HashMap();
                    }
                    dcGroups.put(dc, dcTargets);
                }
                dcTargets.add(destination);
                continue;
            }
            if (!StorageProxy.shouldHint(destination)) continue;
            StorageProxy.submitHint(rm, destination, responseHandler, consistency_level);
        }
        if (dcGroups != null) {
            MessageOut<RowMutation> message = rm.createMessage();
            for (Map.Entry entry : dcGroups.entrySet()) {
                boolean isLocalDC = ((String)entry.getKey()).equals(localDataCenter);
                Collection dcTargets = (Collection)entry.getValue();
                StorageProxy.sendMessagesToOneDC(message, dcTargets, isLocalDC, responseHandler);
            }
        }
    }

    private static AtomicInteger getHintsInProgressFor(InetAddress destination) {
        try {
            return (AtomicInteger)hintsInProgress.load((Object)destination);
        }
        catch (Exception e) {
            throw new AssertionError((Object)e);
        }
    }

    public static Future<Void> submitHint(final RowMutation mutation, InetAddress target, final AbstractWriteResponseHandler responseHandler, final ConsistencyLevel consistencyLevel) {
        assert (!target.equals(FBUtilities.getBroadcastAddress())) : target;
        HintRunnable runnable = new HintRunnable(target){

            @Override
            public void runMayThrow() throws IOException {
                int ttl = mutation.calculateHintTTL();
                if (ttl > 0) {
                    logger.debug("Adding hint for {}", (Object)this.target);
                    StorageProxy.writeHintForMutation(mutation, ttl, this.target);
                    if (responseHandler != null && consistencyLevel == ConsistencyLevel.ANY) {
                        responseHandler.response((MessageIn)null);
                    }
                } else {
                    logger.debug("Skipped writing hint for {} (ttl {})", (Object)this.target, (Object)ttl);
                }
            }
        };
        return StorageProxy.submitHint(runnable);
    }

    private static Future<Void> submitHint(HintRunnable runnable) {
        StorageMetrics.totalHintsInProgress.inc();
        StorageProxy.getHintsInProgressFor(runnable.target).incrementAndGet();
        return StageManager.getStage(Stage.MUTATION).submit(runnable);
    }

    public static void writeHintForMutation(RowMutation mutation, int ttl, InetAddress target) throws IOException {
        assert (ttl > 0);
        UUID hostId = StorageService.instance.getTokenMetadata().getHostId(target);
        if (hostId == null && MessagingService.instance().getVersion(target) < 6) {
            logger.warn("Unable to store hint for host with missing ID, {} (old node?)", (Object)target.toString());
            return;
        }
        assert (hostId != null) : "Missing host ID for " + target.getHostAddress();
        mutation.toHint(ttl, hostId).apply();
        StorageMetrics.totalHints.inc();
    }

    private static void sendMessagesToOneDC(MessageOut message, Collection<InetAddress> targets, boolean localDC, AbstractWriteResponseHandler handler) {
        try {
            StorageProxy.sendMessagesToOneDCInternal(message, targets, localDC, handler);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void sendMessagesToOneDCInternal(MessageOut message, Collection<InetAddress> targets, boolean localDC, AbstractWriteResponseHandler handler) throws IOException {
        Iterator<InetAddress> iter = targets.iterator();
        InetAddress target = iter.next();
        if (localDC || MessagingService.instance().getVersion(target) < 4) {
            MessagingService.instance().sendRR(message, target, handler);
            while (iter.hasNext()) {
                target = iter.next();
                MessagingService.instance().sendRR(message, target, handler);
            }
            return;
        }
        FastByteArrayOutputStream bos = new FastByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        dos.writeInt(targets.size() - 1);
        while (iter.hasNext()) {
            InetAddress destination = iter.next();
            CompactEndpointSerializationHelper.serialize(destination, dos);
            String id = MessagingService.instance().addCallback(handler, message, destination, message.getTimeout());
            dos.writeUTF(id);
        }
        message = message.withParameter("FWD_TO", bos.toByteArray());
        Tracing.trace("Enqueuing message to {}", target);
        MessagingService.instance().sendRR(message, target, handler);
    }

    private static void insertLocal(final RowMutation rm, final AbstractWriteResponseHandler responseHandler) {
        DroppableRunnable runnable = new DroppableRunnable(MessagingService.Verb.MUTATION){

            @Override
            public void runMayThrow() throws IOException {
                rm.apply();
                responseHandler.response((MessageIn)null);
            }
        };
        StageManager.getStage(Stage.MUTATION).execute(runnable);
    }

    public static AbstractWriteResponseHandler mutateCounter(CounterMutation cm, String localDataCenter) throws UnavailableException, OverloadedException, IOException {
        InetAddress endpoint = StorageProxy.findSuitableEndpoint(cm.getTable(), cm.key(), localDataCenter, cm.consistency());
        if (endpoint.equals(FBUtilities.getBroadcastAddress())) {
            return StorageProxy.applyCounterMutationOnCoordinator(cm, localDataCenter);
        }
        String table = cm.getTable();
        AbstractReplicationStrategy rs = Table.open(table).getReplicationStrategy();
        Object tk = StorageService.getPartitioner().getToken(cm.key());
        List<InetAddress> naturalEndpoints = StorageService.instance.getNaturalEndpoints(table, (RingPosition)tk);
        Collection<InetAddress> pendingEndpoints = StorageService.instance.getTokenMetadata().pendingEndpointsFor((Token)tk, table);
        rs.getWriteResponseHandler(naturalEndpoints, pendingEndpoints, cm.consistency(), null, WriteType.COUNTER).assureSufficientLiveNodes();
        WriteResponseHandler responseHandler = new WriteResponseHandler(endpoint, WriteType.COUNTER);
        Tracing.trace("Enqueuing counter update to {}", endpoint);
        MessagingService.instance().sendRR(cm.makeMutationMessage(), endpoint, responseHandler);
        return responseHandler;
    }

    private static InetAddress findSuitableEndpoint(String tableName, ByteBuffer key, String localDataCenter, ConsistencyLevel cl) throws UnavailableException {
        Table table = Table.open(tableName);
        IEndpointSnitch snitch = DatabaseDescriptor.getEndpointSnitch();
        List<InetAddress> endpoints = StorageService.instance.getLiveNaturalEndpoints(table, key);
        if (endpoints.isEmpty()) {
            throw new UnavailableException(cl, cl.blockFor(table), 0);
        }
        ArrayList<InetAddress> localEndpoints = new ArrayList<InetAddress>();
        for (InetAddress endpoint : endpoints) {
            if (!snitch.getDatacenter(endpoint).equals(localDataCenter)) continue;
            localEndpoints.add(endpoint);
        }
        if (localEndpoints.isEmpty()) {
            snitch.sortByProximity(FBUtilities.getBroadcastAddress(), endpoints);
            return endpoints.get(0);
        }
        return (InetAddress)localEndpoints.get(FBUtilities.threadLocalRandom().nextInt(localEndpoints.size()));
    }

    public static AbstractWriteResponseHandler applyCounterMutationOnLeader(CounterMutation cm, String localDataCenter, Runnable callback) throws UnavailableException, IOException, OverloadedException {
        return StorageProxy.performWrite(cm, cm.consistency(), localDataCenter, counterWritePerformer, callback, WriteType.COUNTER);
    }

    public static AbstractWriteResponseHandler applyCounterMutationOnCoordinator(CounterMutation cm, String localDataCenter) throws UnavailableException, IOException, OverloadedException {
        return StorageProxy.performWrite(cm, cm.consistency(), localDataCenter, counterWriteOnCoordinatorPerformer, null, WriteType.COUNTER);
    }

    private static Runnable counterWriteTask(final IMutation mutation, final Iterable<InetAddress> targets, final AbstractWriteResponseHandler responseHandler, final String localDataCenter, final ConsistencyLevel consistency_level) {
        return new LocalMutationRunnable(){

            @Override
            public void runMayThrow() throws IOException {
                assert (mutation instanceof CounterMutation);
                final CounterMutation cm = (CounterMutation)mutation;
                cm.apply();
                responseHandler.response((MessageIn)null);
                Sets.SetView remotes = Sets.difference((Set)ImmutableSet.copyOf((Iterable)targets), (Set)ImmutableSet.of((Object)FBUtilities.getBroadcastAddress()));
                if (cm.shouldReplicateOnWrite() && !remotes.isEmpty()) {
                    StageManager.getStage(Stage.REPLICATE_ON_WRITE).execute(new DroppableRunnable(MessagingService.Verb.READ, (Set)remotes){
                        final /* synthetic */ Set val$remotes;
                        {
                            this.val$remotes = set;
                            super(x0);
                        }

                        @Override
                        public void runMayThrow() throws OverloadedException {
                            StorageProxy.sendToHintedEndpoints(cm.makeReplicationMutation(), this.val$remotes, responseHandler, localDataCenter, consistency_level);
                        }
                    });
                }
            }
        };
    }

    private static boolean systemTableQuery(List<ReadCommand> cmds) {
        for (ReadCommand cmd : cmds) {
            if (cmd.table.equals("system")) continue;
            return false;
        }
        return true;
    }

    public static List<Row> read(List<ReadCommand> commands, ConsistencyLevel consistency_level) throws UnavailableException, IsBootstrappingException, ReadTimeoutException {
        if (StorageService.instance.isBootstrapMode() && !StorageProxy.systemTableQuery(commands)) {
            StorageProxy.readMetrics.unavailables.mark();
            ClientRequestMetrics.readUnavailables.inc();
            throw new IsBootstrappingException();
        }
        long startTime = System.nanoTime();
        List<Row> rows = null;
        try {
            rows = StorageProxy.fetchRows(commands, consistency_level);
        }
        catch (UnavailableException e) {
            StorageProxy.readMetrics.unavailables.mark();
            ClientRequestMetrics.readUnavailables.inc();
            throw e;
        }
        catch (ReadTimeoutException e) {
            StorageProxy.readMetrics.timeouts.mark();
            ClientRequestMetrics.readTimeouts.inc();
            throw e;
        }
        finally {
            readMetrics.addNano(System.nanoTime() - startTime);
        }
        return rows;
    }

    private static List<Row> fetchRows(List<ReadCommand> initialCommands, ConsistencyLevel consistency_level) throws UnavailableException, ReadTimeoutException {
        ArrayList<Row> rows = new ArrayList<Row>(initialCommands.size());
        List commandsToRetry = Collections.emptyList();
        do {
            Row row;
            int i;
            AbstractRowResolver resolver;
            List<ReadCommand> commands = commandsToRetry.isEmpty() ? initialCommands : commandsToRetry;
            ReadCallback[] readCallbacks = new ReadCallback[commands.size()];
            if (!commandsToRetry.isEmpty()) {
                Tracing.trace("Retrying {} commands", commandsToRetry.size());
            }
            for (int i2 = 0; i2 < commands.size(); ++i2) {
                ReadCommand command = commands.get(i2);
                Table table = Table.open(command.getKeyspace());
                assert (!command.isDigestQuery());
                List<InetAddress> endpoints = StorageProxy.getLiveSortedEndpoints(table, command.key);
                CFMetaData cfm = Schema.instance.getCFMetaData(command.getKeyspace(), command.getColumnFamilyName());
                ReadRepairDecision rrDecision = cfm.newReadRepairDecision();
                endpoints = consistency_level.filterForQuery(table, endpoints, rrDecision);
                if (rrDecision != ReadRepairDecision.NONE) {
                    ReadRepairMetrics.attempted.mark();
                }
                resolver = new RowDigestResolver(command.table, command.key);
                ReadCallback<ReadResponse, Row> handler = new ReadCallback<ReadResponse, Row>(resolver, consistency_level, command, endpoints);
                handler.assureSufficientLiveNodes();
                assert (!endpoints.isEmpty());
                readCallbacks[i2] = handler;
                InetAddress dataPoint = endpoints.get(0);
                if (dataPoint.equals(FBUtilities.getBroadcastAddress())) {
                    StageManager.getStage(Stage.READ).execute(new LocalReadRunnable(command, handler));
                } else {
                    Tracing.trace("Enqueuing data request to {}", dataPoint);
                    MessagingService.instance().sendRR(command.createMessage(), dataPoint, handler);
                }
                if (endpoints.size() == 1) continue;
                ReadCommand digestCommand = command.copy();
                digestCommand.setDigestQuery(true);
                MessageOut<ReadCommand> message = null;
                for (InetAddress digestPoint : endpoints.subList(1, endpoints.size())) {
                    if (digestPoint.equals(FBUtilities.getBroadcastAddress())) {
                        StageManager.getStage(Stage.READ).execute(new LocalReadRunnable(digestCommand, handler));
                        continue;
                    }
                    Tracing.trace("Enqueuing digest request to {}", digestPoint);
                    if (message == null) {
                        message = digestCommand.createMessage();
                    }
                    MessagingService.instance().sendRR(message, digestPoint, handler);
                }
            }
            ArrayList<ReadCommand> repairCommands = null;
            ArrayList<ReadCallback<ReadResponse, Row>> repairResponseHandlers = null;
            for (i = 0; i < commands.size(); ++i) {
                ReadCallback handler = readCallbacks[i];
                ReadCommand command = commands.get(i);
                try {
                    row = (Row)handler.get();
                    if (row == null) continue;
                    command.maybeTrim(row);
                    rows.add(row);
                    continue;
                }
                catch (ReadTimeoutException ex) {
                    String gotData;
                    int blockFor = consistency_level.blockFor(Table.open(command.getKeyspace()));
                    int responseCount = handler.getReceivedCount();
                    String string = responseCount > 0 ? (gotData = handler.resolver.isDataPresent() ? " (including data)" : " (only digests)") : (gotData = "");
                    if (Tracing.isTracing()) {
                        Tracing.trace("Timed out; received {} of {} responses{}", new Object[]{responseCount, blockFor, gotData});
                    } else if (logger.isDebugEnabled()) {
                        logger.debug("Read timeout; received {} of {} responses{}", new Object[]{responseCount, blockFor, gotData});
                    }
                    throw ex;
                }
                catch (DigestMismatchException ex) {
                    Tracing.trace("Digest mismatch: {}", ex.toString());
                    ReadRepairMetrics.repairedBlocking.mark();
                    resolver = new RowDataResolver(command.table, command.key, command.filter());
                    ReadCallback<ReadResponse, Row> repairHandler = handler.withNewResolver(resolver);
                    if (repairCommands == null) {
                        repairCommands = new ArrayList<ReadCommand>();
                        repairResponseHandlers = new ArrayList<ReadCallback<ReadResponse, Row>>();
                    }
                    repairCommands.add(command);
                    repairResponseHandlers.add(repairHandler);
                    MessageOut<ReadCommand> message = command.createMessage();
                    for (InetAddress endpoint : handler.endpoints) {
                        Tracing.trace("Enqueuing full data read to {}", endpoint);
                        MessagingService.instance().sendRR(message, endpoint, repairHandler);
                    }
                }
            }
            if (commandsToRetry != Collections.EMPTY_LIST) {
                commandsToRetry.clear();
            }
            if (repairResponseHandlers == null) continue;
            for (i = 0; i < repairCommands.size(); ++i) {
                ReadCommand command = (ReadCommand)repairCommands.get(i);
                ReadCallback handler = (ReadCallback)repairResponseHandlers.get(i);
                try {
                    row = (Row)handler.get();
                }
                catch (DigestMismatchException e) {
                    throw new AssertionError((Object)e);
                }
                resolver = (RowDataResolver)handler.resolver;
                try {
                    FBUtilities.waitOnFutures(resolver.repairResults, DatabaseDescriptor.getWriteRpcTimeout());
                }
                catch (TimeoutException e) {
                    Tracing.trace("Timed out on digest mismatch retries");
                    int blockFor = consistency_level.blockFor(Table.open(command.getKeyspace()));
                    throw new ReadTimeoutException(consistency_level, blockFor, blockFor, true);
                }
                ReadCommand retryCommand = command.maybeGenerateRetryCommand((RowDataResolver)resolver, row);
                if (retryCommand != null) {
                    Tracing.trace("Issuing retry for read command");
                    if (commandsToRetry == Collections.EMPTY_LIST) {
                        commandsToRetry = new ArrayList();
                    }
                    commandsToRetry.add(retryCommand);
                    continue;
                }
                if (row == null) continue;
                command.maybeTrim(row);
                rows.add(row);
            }
        } while (!commandsToRetry.isEmpty());
        return rows;
    }

    private static List<InetAddress> getLiveSortedEndpoints(Table table, ByteBuffer key) {
        return StorageProxy.getLiveSortedEndpoints(table, StorageService.instance.getPartitioner().decorateKey(key));
    }

    private static List<InetAddress> getLiveSortedEndpoints(Table table, RingPosition pos) {
        List<InetAddress> liveEndpoints = StorageService.instance.getLiveNaturalEndpoints(table, pos);
        DatabaseDescriptor.getEndpointSnitch().sortByProximity(FBUtilities.getBroadcastAddress(), liveEndpoints);
        return liveEndpoints;
    }

    private static List<InetAddress> intersection(List<InetAddress> l1, List<InetAddress> l2) {
        ArrayList<InetAddress> inter = new ArrayList<InetAddress>(l1);
        inter.retainAll(l2);
        return inter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<Row> getRangeSlice(RangeSliceCommand command, ConsistencyLevel consistency_level) throws UnavailableException, ReadTimeoutException {
        ArrayList<Row> rows;
        Tracing.trace("Determining replicas to query");
        long startTime = System.nanoTime();
        Table table = Table.open(command.keyspace);
        try {
            IDiskAtomFilter commandPredicate = command.predicate;
            int cql3RowCount = 0;
            rows = new ArrayList<Row>();
            List<AbstractBounds<RowPosition>> ranges = StorageProxy.getRestrictedRanges(command.range);
            int i = 0;
            AbstractBounds<RowPosition> nextRange = null;
            List<InetAddress> nextEndpoints = null;
            List<InetAddress> nextFilteredEndpoints = null;
            while (i < ranges.size()) {
                int count;
                AbstractBounds<RowPosition> range = nextRange == null ? ranges.get(i) : nextRange;
                List<InetAddress> liveEndpoints = nextEndpoints == null ? StorageProxy.getLiveSortedEndpoints(table, range.right) : nextEndpoints;
                List<InetAddress> filteredEndpoints = nextFilteredEndpoints == null ? consistency_level.filterForQuery(table, liveEndpoints) : nextFilteredEndpoints;
                ++i;
                while (i < ranges.size()) {
                    List<InetAddress> merged;
                    nextRange = ranges.get(i);
                    nextEndpoints = StorageProxy.getLiveSortedEndpoints(table, nextRange.right);
                    nextFilteredEndpoints = consistency_level.filterForQuery(table, nextEndpoints);
                    if (((RowPosition)range.right).isMinimum() || !consistency_level.isSufficientLiveNodes(table, merged = StorageProxy.intersection(liveEndpoints, nextEndpoints))) break;
                    List<InetAddress> filteredMerged = consistency_level.filterForQuery(table, merged);
                    if (!DatabaseDescriptor.getEndpointSnitch().isWorthMergingForRangeQuery(filteredMerged, filteredEndpoints, nextFilteredEndpoints)) break;
                    range = range.withNewRight((RowPosition)nextRange.right);
                    liveEndpoints = merged;
                    filteredEndpoints = filteredMerged;
                    ++i;
                }
                RangeSliceCommand nodeCmd = new RangeSliceCommand(command.keyspace, command.column_family, command.super_column, commandPredicate, range, command.row_filter, command.maxResults, command.countCQL3Rows, command.isPaging);
                RangeSliceResponseResolver resolver = new RangeSliceResponseResolver(nodeCmd.keyspace);
                ReadCallback<RangeSliceReply, Iterable<Row>> handler = new ReadCallback<RangeSliceReply, Iterable<Row>>(resolver, consistency_level, nodeCmd, filteredEndpoints);
                handler.assureSufficientLiveNodes();
                resolver.setSources(filteredEndpoints);
                if (filteredEndpoints.size() == 1 && filteredEndpoints.get(0).equals(FBUtilities.getBroadcastAddress())) {
                    StageManager.getStage(Stage.READ).execute(new LocalRangeSliceRunnable(nodeCmd, handler));
                } else {
                    MessageOut<RangeSliceCommand> message = nodeCmd.createMessage();
                    for (InetAddress endpoint : filteredEndpoints) {
                        Tracing.trace("Enqueuing request to {}", endpoint);
                        MessagingService.instance().sendRR(message, endpoint, handler);
                    }
                }
                try {
                    for (Row row : handler.get()) {
                        rows.add(row);
                        if (!nodeCmd.countCQL3Rows) continue;
                        cql3RowCount += row.getLiveCount(commandPredicate);
                    }
                    FBUtilities.waitOnFutures(resolver.repairResults, DatabaseDescriptor.getWriteRpcTimeout());
                }
                catch (ReadTimeoutException ex) {
                    String gotData;
                    int blockFor = consistency_level.blockFor(table);
                    int responseCount = resolver.responses.size();
                    String string = responseCount > 0 ? (resolver.isDataPresent() ? " (including data)" : " (only digests)") : (gotData = "");
                    if (Tracing.isTracing()) {
                        Tracing.trace("Timed out; received {} of {} responses{} for range {} of {}", new Object[]{responseCount, blockFor, gotData, i, ranges.size()});
                    } else if (logger.isDebugEnabled()) {
                        logger.debug("Range slice timeout; received {} of {} responses{} for range {} of {}", new Object[]{responseCount, blockFor, gotData, i, ranges.size()});
                    }
                    throw ex;
                }
                catch (TimeoutException ex) {
                    int blockFor = consistency_level.blockFor(table);
                    if (Tracing.isTracing()) {
                        Tracing.trace("Timed out while read-repairing after receiving all {} data and digest responses", blockFor);
                    } else {
                        logger.debug("Range slice timeout while read-repairing after receiving all {} data and digest responses", (Object)blockFor);
                    }
                    throw new ReadTimeoutException(consistency_level, blockFor, blockFor, true);
                }
                catch (DigestMismatchException e) {
                    throw new AssertionError((Object)e);
                }
                int n = count = nodeCmd.countCQL3Rows ? cql3RowCount : rows.size();
                if (count >= nodeCmd.maxResults) {
                    break;
                }
                if (rows.isEmpty() || !command.isPaging) continue;
                assert (commandPredicate instanceof SliceQueryFilter);
                commandPredicate = ((SliceQueryFilter)commandPredicate).withUpdatedSlices(ColumnSlice.ALL_COLUMNS_ARRAY);
            }
        }
        finally {
            rangeMetrics.addNano(System.nanoTime() - startTime);
        }
        return StorageProxy.trim(command, rows);
    }

    private static List<Row> trim(RangeSliceCommand command, List<Row> rows) {
        if (command.countCQL3Rows) {
            return rows;
        }
        return rows.size() > command.maxResults ? rows.subList(0, command.maxResults) : rows;
    }

    @Override
    public Map<String, List<String>> getSchemaVersions() {
        return this.describeSchemaVersions();
    }

    public static Map<String, List<String>> describeSchemaVersions() {
        String myVersion = Schema.instance.getVersion().toString();
        final ConcurrentHashMap versions = new ConcurrentHashMap();
        Set<InetAddress> liveHosts = Gossiper.instance.getLiveMembers();
        final CountDownLatch latch = new CountDownLatch(liveHosts.size());
        IAsyncCallback<UUID> cb = new IAsyncCallback<UUID>(){

            @Override
            public void response(MessageIn<UUID> message) {
                versions.put(message.from, message.payload);
                latch.countDown();
            }

            @Override
            public boolean isLatencyForSnitch() {
                return false;
            }
        };
        MessageOut message = new MessageOut(MessagingService.Verb.SCHEMA_CHECK);
        for (InetAddress endpoint : liveHosts) {
            MessagingService.instance().sendRR(message, endpoint, cb);
        }
        try {
            latch.await(DatabaseDescriptor.getRpcTimeout(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException ex) {
            throw new AssertionError((Object)"This latch shouldn't have been interrupted.");
        }
        HashMap<String, List<String>> results = new HashMap<String, List<String>>();
        Iterable allHosts = Iterables.concat(Gossiper.instance.getLiveMembers(), Gossiper.instance.getUnreachableMembers());
        for (InetAddress inetAddress : allHosts) {
            UUID version = (UUID)versions.get(inetAddress);
            String stringVersion = version == null ? UNREACHABLE : version.toString();
            ArrayList<String> hosts = (ArrayList<String>)results.get(stringVersion);
            if (hosts == null) {
                hosts = new ArrayList<String>();
                results.put(stringVersion, hosts);
            }
            hosts.add(inetAddress.getHostAddress());
        }
        if (results.get(UNREACHABLE) != null) {
            logger.debug("Hosts not in agreement. Didn't get a response from everybody: {}", (Object)StringUtils.join((Collection)((Collection)results.get(UNREACHABLE)), (String)","));
        }
        for (Map.Entry entry : results.entrySet()) {
            if (((String)entry.getKey()).equals(UNREACHABLE) || ((String)entry.getKey()).equals(myVersion)) continue;
            for (String host : (List)entry.getValue()) {
                logger.debug("{} disagrees ({})", (Object)host, entry.getKey());
            }
        }
        if (results.size() == 1) {
            logger.debug("Schemas are in agreement.");
        }
        return results;
    }

    static <T extends RingPosition> List<AbstractBounds<T>> getRestrictedRanges(AbstractBounds<T> queryRange) {
        Token upperBoundToken;
        Object upperBound;
        if (queryRange instanceof Bounds && queryRange.left.equals(queryRange.right) && !queryRange.left.isMinimum(StorageService.getPartitioner())) {
            return Collections.singletonList(queryRange);
        }
        TokenMetadata tokenMetadata = StorageService.instance.getTokenMetadata();
        ArrayList<AbstractBounds<T>> ranges = new ArrayList<AbstractBounds<T>>();
        Iterator<Token> ringIter = TokenMetadata.ringIterator(tokenMetadata.sortedTokens(), queryRange.left.getToken(), true);
        AbstractBounds remainder = queryRange;
        while (ringIter.hasNext() && (remainder.left.equals(upperBound = (upperBoundToken = ringIter.next()).upperBound(queryRange.left.getClass())) || remainder.contains(upperBound))) {
            Pair<AbstractBounds<T>, AbstractBounds<T>> splits = remainder.split(upperBound);
            if (splits == null) continue;
            ranges.add((AbstractBounds<T>)splits.left);
            remainder = (AbstractBounds)splits.right;
        }
        ranges.add(remainder);
        return ranges;
    }

    @Override
    public long getReadOperations() {
        return StorageProxy.readMetrics.latency.count();
    }

    @Override
    public long getTotalReadLatencyMicros() {
        return StorageProxy.readMetrics.totalLatency.count();
    }

    @Override
    public double getRecentReadLatencyMicros() {
        return readMetrics.getRecentLatency();
    }

    @Override
    public long[] getTotalReadLatencyHistogramMicros() {
        return StorageProxy.readMetrics.totalLatencyHistogram.getBuckets(false);
    }

    @Override
    public long[] getRecentReadLatencyHistogramMicros() {
        return StorageProxy.readMetrics.recentLatencyHistogram.getBuckets(true);
    }

    @Override
    public long getRangeOperations() {
        return StorageProxy.rangeMetrics.latency.count();
    }

    @Override
    public long getTotalRangeLatencyMicros() {
        return StorageProxy.rangeMetrics.totalLatency.count();
    }

    @Override
    public double getRecentRangeLatencyMicros() {
        return rangeMetrics.getRecentLatency();
    }

    @Override
    public long[] getTotalRangeLatencyHistogramMicros() {
        return StorageProxy.rangeMetrics.totalLatencyHistogram.getBuckets(false);
    }

    @Override
    public long[] getRecentRangeLatencyHistogramMicros() {
        return StorageProxy.rangeMetrics.recentLatencyHistogram.getBuckets(true);
    }

    @Override
    public long getWriteOperations() {
        return StorageProxy.writeMetrics.latency.count();
    }

    @Override
    public long getTotalWriteLatencyMicros() {
        return StorageProxy.writeMetrics.totalLatency.count();
    }

    @Override
    public double getRecentWriteLatencyMicros() {
        return writeMetrics.getRecentLatency();
    }

    @Override
    public long[] getTotalWriteLatencyHistogramMicros() {
        return StorageProxy.writeMetrics.totalLatencyHistogram.getBuckets(false);
    }

    @Override
    public long[] getRecentWriteLatencyHistogramMicros() {
        return StorageProxy.writeMetrics.recentLatencyHistogram.getBuckets(true);
    }

    @Override
    public boolean getHintedHandoffEnabled() {
        return DatabaseDescriptor.hintedHandoffEnabled();
    }

    @Override
    public void setHintedHandoffEnabled(boolean b) {
        DatabaseDescriptor.setHintedHandoffEnabled(b);
    }

    @Override
    public int getMaxHintWindow() {
        return DatabaseDescriptor.getMaxHintWindow();
    }

    @Override
    public void setMaxHintWindow(int ms) {
        DatabaseDescriptor.setMaxHintWindow(ms);
    }

    public static boolean shouldHint(InetAddress ep) {
        boolean hintWindowExpired;
        if (!DatabaseDescriptor.hintedHandoffEnabled()) {
            HintedHandOffManager.instance.metrics.incrPastWindow(ep);
            return false;
        }
        boolean bl = hintWindowExpired = Gossiper.instance.getEndpointDowntime(ep) > (long)DatabaseDescriptor.getMaxHintWindow();
        if (hintWindowExpired) {
            HintedHandOffManager.instance.metrics.incrPastWindow(ep);
            Tracing.trace("Not hinting {} which has been down {}ms", ep, Gossiper.instance.getEndpointDowntime(ep));
        }
        return !hintWindowExpired;
    }

    public static void truncateBlocking(String keyspace, String cfname) throws UnavailableException, TimeoutException, IOException {
        logger.debug("Starting a blocking truncate operation on keyspace {}, CF {}", (Object)keyspace, (Object)cfname);
        if (StorageProxy.isAnyHostDown()) {
            logger.info("Cannot perform truncate, some hosts are down");
            int liveMembers = Gossiper.instance.getLiveMembers().size();
            throw new UnavailableException(ConsistencyLevel.ALL, liveMembers + Gossiper.instance.getUnreachableMembers().size(), liveMembers);
        }
        Set<InetAddress> allEndpoints = Gossiper.instance.getLiveMembers();
        int blockFor = allEndpoints.size();
        TruncateResponseHandler responseHandler = new TruncateResponseHandler(blockFor);
        Tracing.trace("Enqueuing truncate messages to hosts {}", allEndpoints);
        Truncation truncation = new Truncation(keyspace, cfname);
        MessageOut<Truncation> message = truncation.createMessage();
        for (InetAddress endpoint : allEndpoints) {
            MessagingService.instance().sendRR(message, endpoint, responseHandler);
        }
        try {
            responseHandler.get();
        }
        catch (TimeoutException e) {
            Tracing.trace("Timed out");
            throw e;
        }
    }

    private static boolean isAnyHostDown() {
        return !Gossiper.instance.getUnreachableMembers().isEmpty();
    }

    @Override
    public long getTotalHints() {
        return StorageMetrics.totalHints.count();
    }

    @Override
    public int getMaxHintsInProgress() {
        return maxHintsInProgress;
    }

    @Override
    public void setMaxHintsInProgress(int qs) {
        maxHintsInProgress = qs;
    }

    @Override
    public int getHintsInProgress() {
        return (int)StorageMetrics.totalHintsInProgress.count();
    }

    public void verifyNoHintsInProgress() {
        if (this.getHintsInProgress() > 0) {
            logger.warn("Some hints were not written before shutdown.  This is not supposed to happen.  You should (a) run repair, and (b) file a bug report");
        }
    }

    @Override
    public Long getRpcTimeout() {
        return DatabaseDescriptor.getRpcTimeout();
    }

    @Override
    public void setRpcTimeout(Long timeoutInMillis) {
        DatabaseDescriptor.setRpcTimeout(timeoutInMillis);
    }

    @Override
    public Long getReadRpcTimeout() {
        return DatabaseDescriptor.getReadRpcTimeout();
    }

    @Override
    public void setReadRpcTimeout(Long timeoutInMillis) {
        DatabaseDescriptor.setReadRpcTimeout(timeoutInMillis);
    }

    @Override
    public Long getWriteRpcTimeout() {
        return DatabaseDescriptor.getWriteRpcTimeout();
    }

    @Override
    public void setWriteRpcTimeout(Long timeoutInMillis) {
        DatabaseDescriptor.setWriteRpcTimeout(timeoutInMillis);
    }

    @Override
    public Long getRangeRpcTimeout() {
        return DatabaseDescriptor.getRangeRpcTimeout();
    }

    @Override
    public void setRangeRpcTimeout(Long timeoutInMillis) {
        DatabaseDescriptor.setRangeRpcTimeout(timeoutInMillis);
    }

    @Override
    public Long getTruncateRpcTimeout() {
        return DatabaseDescriptor.getTruncateRpcTimeout();
    }

    @Override
    public void setTruncateRpcTimeout(Long timeoutInMillis) {
        DatabaseDescriptor.setTruncateRpcTimeout(timeoutInMillis);
    }

    @Override
    public long getReadRepairAttempted() {
        return ReadRepairMetrics.attempted.count();
    }

    @Override
    public long getReadRepairRepairedBlocking() {
        return ReadRepairMetrics.repairedBlocking.count();
    }

    @Override
    public long getReadRepairRepairedBackground() {
        return ReadRepairMetrics.repairedBackground.count();
    }

    static {
        instance = new StorageProxy();
        maxHintsInProgress = 1024 * FBUtilities.getAvailableProcessors();
        hintsInProgress = new CacheLoader<InetAddress, AtomicInteger>(){

            public AtomicInteger load(InetAddress inetAddress) {
                return new AtomicInteger(0);
            }
        };
        readMetrics = new ClientRequestMetrics("Read");
        rangeMetrics = new ClientRequestMetrics("RangeSlice");
        writeMetrics = new ClientRequestMetrics("Write");
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(new StorageProxy(), new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        standardWritePerformer = new WritePerformer(){

            @Override
            public void apply(IMutation mutation, Iterable<InetAddress> targets, AbstractWriteResponseHandler responseHandler, String localDataCenter, ConsistencyLevel consistency_level) throws IOException, OverloadedException {
                assert (mutation instanceof RowMutation);
                StorageProxy.sendToHintedEndpoints((RowMutation)mutation, targets, responseHandler, localDataCenter, consistency_level);
            }
        };
        counterWritePerformer = new WritePerformer(){

            @Override
            public void apply(IMutation mutation, Iterable<InetAddress> targets, AbstractWriteResponseHandler responseHandler, String localDataCenter, ConsistencyLevel consistency_level) throws IOException {
                Runnable runnable = StorageProxy.counterWriteTask(mutation, targets, responseHandler, localDataCenter, consistency_level);
                runnable.run();
            }
        };
        counterWriteOnCoordinatorPerformer = new WritePerformer(){

            @Override
            public void apply(IMutation mutation, Iterable<InetAddress> targets, AbstractWriteResponseHandler responseHandler, String localDataCenter, ConsistencyLevel consistency_level) throws IOException {
                Runnable runnable = StorageProxy.counterWriteTask(mutation, targets, responseHandler, localDataCenter, consistency_level);
                StageManager.getStage(Stage.MUTATION).execute(runnable);
            }
        };
    }

    private static abstract class HintRunnable
    implements Runnable {
        public final InetAddress target;

        protected HintRunnable(InetAddress target) {
            this.target = target;
        }

        @Override
        public void run() {
            try {
                this.runMayThrow();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            finally {
                StorageMetrics.totalHintsInProgress.dec();
                StorageProxy.getHintsInProgressFor(this.target).decrementAndGet();
            }
        }

        protected abstract void runMayThrow() throws Exception;
    }

    private static abstract class LocalMutationRunnable
    implements Runnable {
        private final long constructionTime = System.currentTimeMillis();

        private LocalMutationRunnable() {
        }

        @Override
        public final void run() {
            if (System.currentTimeMillis() > this.constructionTime + DatabaseDescriptor.getTimeout(MessagingService.Verb.MUTATION)) {
                MessagingService.instance().incrementDroppedMessages(MessagingService.Verb.MUTATION);
                HintRunnable runnable = new HintRunnable(FBUtilities.getBroadcastAddress()){

                    @Override
                    protected void runMayThrow() throws Exception {
                        LocalMutationRunnable.this.runMayThrow();
                    }
                };
                StorageProxy.submitHint(runnable);
                return;
            }
            try {
                this.runMayThrow();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        protected abstract void runMayThrow() throws Exception;
    }

    private static abstract class DroppableRunnable
    implements Runnable {
        private final long constructionTime = System.currentTimeMillis();
        private final MessagingService.Verb verb;

        public DroppableRunnable(MessagingService.Verb verb) {
            this.verb = verb;
        }

        @Override
        public final void run() {
            if (System.currentTimeMillis() > this.constructionTime + DatabaseDescriptor.getTimeout(this.verb)) {
                MessagingService.instance().incrementDroppedMessages(this.verb);
                return;
            }
            try {
                this.runMayThrow();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        protected abstract void runMayThrow() throws Exception;
    }

    public static interface WritePerformer {
        public void apply(IMutation var1, Iterable<InetAddress> var2, AbstractWriteResponseHandler var3, String var4, ConsistencyLevel var5) throws IOException, OverloadedException;
    }

    static class LocalRangeSliceRunnable
    extends DroppableRunnable {
        private final RangeSliceCommand command;
        private final ReadCallback<RangeSliceReply, Iterable<Row>> handler;
        private final long start = System.currentTimeMillis();

        LocalRangeSliceRunnable(RangeSliceCommand command, ReadCallback<RangeSliceReply, Iterable<Row>> handler) {
            super(MessagingService.Verb.READ);
            this.command = command;
            this.handler = handler;
        }

        @Override
        protected void runMayThrow() throws ExecutionException, InterruptedException {
            RangeSliceReply result = new RangeSliceReply(RangeSliceVerbHandler.executeLocally(this.command));
            MessagingService.instance().addLatency(FBUtilities.getBroadcastAddress(), System.currentTimeMillis() - this.start);
            this.handler.response(result);
        }
    }

    static class LocalReadRunnable
    extends DroppableRunnable {
        private final ReadCommand command;
        private final ReadCallback<ReadResponse, Row> handler;
        private final long start = System.currentTimeMillis();

        LocalReadRunnable(ReadCommand command, ReadCallback<ReadResponse, Row> handler) {
            super(MessagingService.Verb.READ);
            this.command = command;
            this.handler = handler;
        }

        @Override
        protected void runMayThrow() {
            Table table = Table.open(this.command.table);
            Row r = this.command.getRow(table);
            ReadResponse result = ReadVerbHandler.getResponse(this.command, r);
            MessagingService.instance().addLatency(FBUtilities.getBroadcastAddress(), System.currentTimeMillis() - this.start);
            this.handler.response(result);
        }
    }

    private static class WriteResponseHandlerWrapper {
        final AbstractWriteResponseHandler handler;
        final RowMutation mutation;

        WriteResponseHandlerWrapper(AbstractWriteResponseHandler handler, RowMutation mutation) {
            this.handler = handler;
            this.mutation = mutation;
        }
    }
}

