/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionTask;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SizeTieredCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(SizeTieredCompactionStrategy.class);
    protected static final long DEFAULT_MIN_SSTABLE_SIZE = 0x3200000L;
    protected static final double DEFAULT_BUCKET_LOW = 0.5;
    protected static final double DEFAULT_BUCKET_HIGH = 1.5;
    protected static final String MIN_SSTABLE_SIZE_KEY = "min_sstable_size";
    protected static final String BUCKET_LOW_KEY = "bucket_low";
    protected static final String BUCKET_HIGH_KEY = "bucket_high";
    protected long minSSTableSize;
    protected double bucketLow;
    protected double bucketHigh;
    protected volatile int estimatedRemainingTasks = 0;

    public SizeTieredCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        String optionValue = options.get(MIN_SSTABLE_SIZE_KEY);
        this.minSSTableSize = optionValue == null ? 0x3200000L : Long.parseLong(optionValue);
        optionValue = options.get(BUCKET_LOW_KEY);
        this.bucketLow = optionValue == null ? 0.5 : Double.parseDouble(optionValue);
        optionValue = options.get(BUCKET_HIGH_KEY);
        this.bucketHigh = optionValue == null ? 1.5 : Double.parseDouble(optionValue);
        cfs.setCompactionThresholds(cfs.metadata.getMinCompactionThreshold(), cfs.metadata.getMaxCompactionThreshold());
    }

    @Override
    public synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        if (this.cfs.isCompactionDisabled()) {
            logger.debug("Compaction is currently disabled.");
            return null;
        }
        Set<SSTableReader> candidates = this.cfs.getUncompactingSSTables();
        List<List<SSTableReader>> buckets = this.getBuckets(SizeTieredCompactionStrategy.createSSTableAndLengthPairs(SizeTieredCompactionStrategy.filterSuspectSSTables(candidates)));
        logger.debug("Compaction buckets are {}", buckets);
        this.updateEstimatedCompactionsByTasks(buckets);
        ArrayList<List<SSTableReader>> prunedBuckets = new ArrayList<List<SSTableReader>>();
        for (List<SSTableReader> bucket : buckets) {
            if (bucket.size() < this.cfs.getMinimumCompactionThreshold()) continue;
            Collections.sort(bucket, new Comparator<SSTableReader>(){

                @Override
                public int compare(SSTableReader o1, SSTableReader o2) {
                    return o1.descriptor.generation - o2.descriptor.generation;
                }
            });
            prunedBuckets.add(bucket.subList(0, Math.min(bucket.size(), this.cfs.getMaximumCompactionThreshold())));
        }
        if (prunedBuckets.isEmpty()) {
            for (List<SSTableReader> bucket : buckets) {
                for (SSTableReader table : bucket) {
                    if (!this.worthDroppingTombstones(table, gcBefore)) continue;
                    prunedBuckets.add(Collections.singletonList(table));
                }
            }
            if (prunedBuckets.isEmpty()) {
                return null;
            }
        }
        List<SSTableReader> smallestBucket = Collections.min(prunedBuckets, new Comparator<List<SSTableReader>>(){

            @Override
            public int compare(List<SSTableReader> o1, List<SSTableReader> o2) {
                long n = this.avgSize(o1) - this.avgSize(o2);
                if (n < 0L) {
                    return -1;
                }
                if (n > 0L) {
                    return 1;
                }
                return 0;
            }

            private long avgSize(List<SSTableReader> sstables) {
                long n = 0L;
                for (SSTableReader sstable : sstables) {
                    n += sstable.bytesOnDisk();
                }
                return n / (long)sstables.size();
            }
        });
        if (!this.cfs.getDataTracker().markCompacting(smallestBucket)) {
            logger.debug("Unable to mark {} for compaction; probably a user-defined compaction got in the way", smallestBucket);
            return null;
        }
        return new CompactionTask(this.cfs, smallestBucket, gcBefore);
    }

    @Override
    public AbstractCompactionTask getMaximalTask(int gcBefore) {
        return this.cfs.getSSTables().isEmpty() ? null : new CompactionTask(this.cfs, SizeTieredCompactionStrategy.filterSuspectSSTables(this.cfs.getSSTables()), gcBefore);
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        return new CompactionTask(this.cfs, sstables, gcBefore).setUserDefined(true);
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.estimatedRemainingTasks;
    }

    private static List<Pair<SSTableReader, Long>> createSSTableAndLengthPairs(Collection<SSTableReader> collection) {
        ArrayList<Pair<SSTableReader, Long>> tableLengthPairs = new ArrayList<Pair<SSTableReader, Long>>(collection.size());
        for (SSTableReader table : collection) {
            tableLengthPairs.add(Pair.create(table, table.onDiskLength()));
        }
        return tableLengthPairs;
    }

    <T> List<List<T>> getBuckets(Collection<Pair<T, Long>> files) {
        ArrayList<Pair<T, Long>> sortedFiles = new ArrayList<Pair<T, Long>>(files);
        Collections.sort(sortedFiles, new Comparator<Pair<T, Long>>(){

            @Override
            public int compare(Pair<T, Long> p1, Pair<T, Long> p2) {
                return ((Long)p1.right).compareTo((Long)p2.right);
            }
        });
        HashMap<Long, List> buckets = new HashMap<Long, List>();
        block0: for (Pair pair : sortedFiles) {
            long size = (Long)pair.right;
            for (Map.Entry entry : buckets.entrySet()) {
                List bucket = (List)entry.getValue();
                long oldAverageSize = (Long)entry.getKey();
                if (!((double)size > (double)oldAverageSize * this.bucketLow && (double)size < (double)oldAverageSize * this.bucketHigh) && (size >= this.minSSTableSize || oldAverageSize >= this.minSSTableSize)) continue;
                buckets.remove(oldAverageSize);
                long totalSize = (long)bucket.size() * oldAverageSize;
                long newAverageSize = (totalSize + size) / (long)(bucket.size() + 1);
                bucket.add(pair.left);
                buckets.put(newAverageSize, bucket);
                continue block0;
            }
            ArrayList bucket = new ArrayList();
            bucket.add(pair.left);
            buckets.put(size, bucket);
        }
        return new ArrayList<List<T>>(buckets.values());
    }

    private void updateEstimatedCompactionsByTasks(List<List<SSTableReader>> tasks) {
        int n = 0;
        for (List<SSTableReader> bucket : tasks) {
            if (bucket.size() < this.cfs.getMinimumCompactionThreshold()) continue;
            n = (int)((double)n + Math.ceil((double)bucket.size() / (double)this.cfs.getMaximumCompactionThreshold()));
        }
        this.estimatedRemainingTasks = n;
    }

    @Override
    public long getMaxSSTableSize() {
        return Long.MAX_VALUE;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        double bucketHigh;
        double bucketLow;
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        String optionValue = options.get(MIN_SSTABLE_SIZE_KEY);
        try {
            long minSSTableSize;
            long l = minSSTableSize = optionValue == null ? 0x3200000L : Long.parseLong(optionValue);
            if (minSSTableSize < 0L) {
                throw new ConfigurationException(String.format("%s must be non negative: %d", MIN_SSTABLE_SIZE_KEY, minSSTableSize));
            }
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", optionValue, MIN_SSTABLE_SIZE_KEY), e);
        }
        optionValue = options.get(BUCKET_LOW_KEY);
        try {
            bucketLow = optionValue == null ? 0.5 : Double.parseDouble(optionValue);
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", optionValue, 0.5), e);
        }
        optionValue = options.get(BUCKET_HIGH_KEY);
        try {
            bucketHigh = optionValue == null ? 1.5 : Double.parseDouble(optionValue);
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", optionValue, 1.5), e);
        }
        if (bucketHigh <= bucketLow) {
            throw new ConfigurationException(String.format("BucketHigh value (%s) is less than or equal BucketLow value (%s)", bucketHigh, bucketLow));
        }
        uncheckedOptions.remove(MIN_SSTABLE_SIZE_KEY);
        uncheckedOptions.remove(BUCKET_LOW_KEY);
        uncheckedOptions.remove(BUCKET_HIGH_KEY);
        uncheckedOptions.remove("min_threshold");
        uncheckedOptions.remove("max_threshold");
        return uncheckedOptions;
    }

    public String toString() {
        return String.format("SizeTieredCompactionStrategy[%s/%s]", this.cfs.getMinimumCompactionThreshold(), this.cfs.getMaximumCompactionThreshold());
    }
}

