/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOError;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.config.Config;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.BatchCommitLogExecutorService;
import org.apache.cassandra.db.commitlog.CommitLogAllocator;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.ICommitLogExecutorService;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogExecutorService;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = new CommitLog();
    private final ICommitLogExecutorService executor;
    public final CommitLogAllocator allocator;
    public final CommitLogArchiver archiver = new CommitLogArchiver();
    public static final int END_OF_SEGMENT_MARKER = 0;
    public static final int END_OF_SEGMENT_MARKER_SIZE = 4;
    public CommitLogSegment activeSegment;

    private CommitLog() {
        try {
            DatabaseDescriptor.createAllDirectories();
            this.allocator = new CommitLogAllocator();
            this.activateNextSegment();
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        this.executor = DatabaseDescriptor.getCommitLogSync() == Config.CommitLogSync.batch ? new BatchCommitLogExecutorService() : new PeriodicCommitLogExecutorService(this);
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(this, new ObjectName("org.apache.cassandra.db:type=Commitlog"));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void resetUnsafe() throws IOException {
        this.allocator.resetUnsafe();
        this.activateNextSegment();
    }

    public int recover() throws IOException {
        this.archiver.maybeRestoreArchive();
        Object[] files = new File(DatabaseDescriptor.getCommitLogLocation()).listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return CommitLogSegment.possibleCommitLogFile(name) && !CommitLog.instance.allocator.manages(name);
            }
        });
        int replayed = 0;
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new FileUtils.FileComparator());
            logger.info("Replaying " + StringUtils.join((Object[])files, (String)", "));
            replayed = this.recover((File[])files);
            logger.info("Log replay complete, " + replayed + " replayed mutations");
            for (Object f : files) {
                CommitLog.instance.allocator.recycleSegment((File)f);
            }
        }
        this.allocator.enableReserveSegmentCreation();
        return replayed;
    }

    public int recover(File ... clogs) throws IOException {
        CommitLogReplayer recovery = new CommitLogReplayer();
        recovery.recover(clogs);
        return recovery.blockForWrites();
    }

    @Override
    public void recover(String path) throws IOException {
        this.recover(new File(path));
    }

    public Future<ReplayPosition> getContext() {
        Callable<ReplayPosition> task = new Callable<ReplayPosition>(){

            @Override
            public ReplayPosition call() throws Exception {
                return CommitLog.this.activeSegment.getContext();
            }
        };
        return this.executor.submit(task);
    }

    public int activeSegments() {
        return this.allocator.getActiveSegments().size();
    }

    public void add(RowMutation rm) throws IOException {
        long totalSize = RowMutation.serializer().serializedSize(rm, 4) + 20L;
        if (totalSize > (long)DatabaseDescriptor.getCommitLogSegmentSize()) {
            logger.warn("Skipping commitlog append of extremely large mutation ({} bytes)", (Object)totalSize);
            return;
        }
        this.executor.add(new LogRecordAdder(rm));
    }

    public void discardCompletedSegments(final Integer cfId, final ReplayPosition context) throws IOException {
        Callable task = new Callable(){

            public Object call() throws IOException {
                logger.debug("discard completed log segments for {}, column family {}", (Object)context, (Object)cfId);
                Iterator<CommitLogSegment> iter = CommitLog.this.allocator.getActiveSegments().iterator();
                while (iter.hasNext()) {
                    CommitLogSegment segment = iter.next();
                    segment.markClean(cfId, context);
                    if (segment.isUnused() && iter.hasNext()) {
                        logger.debug("Commit log segment {} is unused", (Object)segment);
                        CommitLog.this.allocator.recycleSegment(segment);
                    } else if (logger.isDebugEnabled()) {
                        logger.debug(String.format("Not safe to delete commit log %s; dirty is %s; hasNext: %s", segment, segment.dirtyString(), iter.hasNext()));
                    }
                    if (!segment.contains(context)) continue;
                    break;
                }
                return null;
            }
        };
        try {
            this.executor.submit(task).get();
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public void sync() throws IOException {
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segment.sync();
        }
    }

    @Override
    public long getCompletedTasks() {
        return this.executor.getCompletedTasks();
    }

    @Override
    public long getPendingTasks() {
        return this.executor.getPendingTasks();
    }

    @Override
    public long getTotalCommitlogSize() {
        return this.allocator.bytesUsed();
    }

    public void forceNewSegment() throws ExecutionException, InterruptedException {
        logger.debug("Forcing new segment creation");
        Callable task = new Callable(){

            public Object call() throws IOException {
                if (CommitLog.this.activeSegment.position() > 0) {
                    CommitLog.this.activateNextSegment();
                }
                return null;
            }
        };
        this.executor.submit(task).get();
    }

    private void activateNextSegment() throws IOException {
        this.activeSegment = this.allocator.fetchSegment();
        logger.debug("Active segment is now {}", (Object)this.activeSegment);
    }

    @Override
    public List<String> getActiveSegmentNames() {
        ArrayList<String> segmentNames = new ArrayList<String>();
        for (CommitLogSegment segment : this.allocator.getActiveSegments()) {
            segmentNames.add(segment.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    public void shutdownBlocking() throws InterruptedException {
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.allocator.shutdown();
        this.allocator.awaitTermination();
    }

    class LogRecordAdder
    implements Callable,
    Runnable {
        final RowMutation rowMutation;

        LogRecordAdder(RowMutation rm) {
            this.rowMutation = rm;
        }

        @Override
        public void run() {
            try {
                if (!CommitLog.this.activeSegment.hasCapacityFor(this.rowMutation)) {
                    CommitLogSegment oldSegment = CommitLog.this.activeSegment;
                    CommitLog.this.activateNextSegment();
                    CommitLog.this.archiver.maybeArchive(oldSegment.getPath(), oldSegment.getName());
                }
                CommitLog.this.activeSegment.write(this.rowMutation);
            }
            catch (IOException e) {
                throw new IOError(e);
            }
        }

        public Object call() throws Exception {
            this.run();
            return null;
        }
    }
}

