/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOError;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.dht.IPartitioner;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.util.BufferedRandomAccessFile;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class SSTable {
    static final Logger logger = LoggerFactory.getLogger(SSTable.class);
    public static final String COMPONENT_DATA = Component.Type.DATA.repr;
    public static final String COMPONENT_INDEX = Component.Type.PRIMARY_INDEX.repr;
    public static final String COMPONENT_FILTER = Component.Type.FILTER.repr;
    public static final String COMPONENT_STATS = Component.Type.STATS.repr;
    public static final String TEMPFILE_MARKER = "tmp";
    public final Descriptor descriptor;
    protected final Set<Component> components;
    public final CFMetaData metadata;
    public final IPartitioner partitioner;
    protected final EstimatedHistogram estimatedRowSize;
    protected final EstimatedHistogram estimatedColumnCount;

    protected SSTable(Descriptor descriptor, CFMetaData metadata, IPartitioner partitioner) {
        this(descriptor, new HashSet<Component>(), metadata, partitioner);
    }

    protected SSTable(Descriptor descriptor, Set<Component> components, CFMetaData metadata, IPartitioner partitioner) {
        this(descriptor, components, metadata, partitioner, SSTable.defaultRowHistogram(), SSTable.defaultColumnHistogram());
    }

    static EstimatedHistogram defaultColumnHistogram() {
        return new EstimatedHistogram(114);
    }

    static EstimatedHistogram defaultRowHistogram() {
        return new EstimatedHistogram(150);
    }

    protected SSTable(Descriptor descriptor, Set<Component> components, CFMetaData metadata, IPartitioner partitioner, EstimatedHistogram rowSizes, EstimatedHistogram columnCounts) {
        this.descriptor = descriptor;
        HashSet<Component> dataComponents = new HashSet<Component>(components);
        for (Component component : components) {
            assert (component.type != Component.Type.COMPACTED_MARKER);
        }
        this.components = Collections.unmodifiableSet(dataComponents);
        this.metadata = metadata;
        this.partitioner = partitioner;
        this.estimatedRowSize = rowSizes;
        this.estimatedColumnCount = columnCounts;
    }

    public EstimatedHistogram getEstimatedRowSize() {
        return this.estimatedRowSize;
    }

    public EstimatedHistogram getEstimatedColumnCount() {
        return this.estimatedColumnCount;
    }

    public static boolean delete(Descriptor desc, Set<Component> components) {
        try {
            if (components.contains(Component.DATA)) {
                FileUtils.deleteWithConfirm(desc.filenameFor(Component.DATA));
            }
            for (Component component : components) {
                if (component.equals(Component.DATA) || component.equals(Component.COMPACTED_MARKER)) continue;
                FileUtils.deleteWithConfirm(desc.filenameFor(component));
            }
            FileUtils.delete(desc.filenameFor(Component.COMPACTED_MARKER));
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        logger.info("Deleted " + desc);
        return true;
    }

    public String getFilename() {
        return this.descriptor.filenameFor(COMPONENT_DATA);
    }

    public String getColumnFamilyName() {
        return this.descriptor.cfname;
    }

    public String getTableName() {
        return this.descriptor.ksname;
    }

    public static Pair<Descriptor, Component> tryComponentFromFilename(File dir, String name) {
        try {
            return Component.fromFilename(dir, name);
        }
        catch (Exception e) {
            if (!"snapshots".equals(name) && !"backups".equals(name)) {
                logger.warn("Invalid file '{}' in data directory {}.", (Object)name, (Object)dir);
            }
            return null;
        }
    }

    static Set<Component> componentsFor(final Descriptor desc) {
        final HashSet<Component> components = new HashSet<Component>();
        desc.directory.list(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                Pair<Descriptor, Component> component = SSTable.tryComponentFromFilename(dir, name);
                if (component != null && ((Descriptor)component.left).equals(desc)) {
                    components.add(component.right);
                }
                return false;
            }
        });
        return components;
    }

    static long estimateRowsFromData(Descriptor desc, BufferedRandomAccessFile dfile) throws IOException {
        int keys;
        int SAMPLES_CAP = 1000;
        int BYTES_CAP = (int)Math.min(100000000L, dfile.length());
        long dataPosition = 0L;
        for (keys = 0; dataPosition < (long)BYTES_CAP && keys < 1000; ++keys) {
            dfile.seek(dataPosition);
            ByteBufferUtil.skipShortLength(dfile);
            long dataSize = SSTableReader.readRowSize(dfile, desc);
            dataPosition = dfile.getFilePointer() + dataSize;
        }
        dfile.seek(0L);
        return dfile.length() / (dataPosition / (long)keys);
    }

    static long estimateRowsFromIndex(BufferedRandomAccessFile ifile) throws IOException {
        int keys;
        int SAMPLES_CAP = 10000;
        int BYTES_CAP = (int)Math.min(10000000L, ifile.length());
        for (keys = 0; ifile.getFilePointer() < (long)BYTES_CAP && keys < 10000; ++keys) {
            ByteBufferUtil.skipShortLength(ifile);
            ifile.skipBytes(8);
        }
        assert (keys > 0 && ifile.getFilePointer() > 0L && ifile.length() > 0L);
        long estimatedRows = ifile.length() / (ifile.getFilePointer() / (long)keys);
        ifile.seek(0L);
        return estimatedRows;
    }

    public static long getTotalBytes(Iterable<SSTableReader> sstables) {
        long sum = 0L;
        for (SSTableReader sstable : sstables) {
            sum += sstable.length();
        }
        return sum;
    }

    public long bytesOnDisk() {
        long bytes = 0L;
        for (Component component : this.components) {
            bytes += new File(this.descriptor.filenameFor(component)).length();
        }
        return bytes;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(" + "path='" + this.getFilename() + '\'' + ')';
    }
}

