/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.pqc.lifecycle;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.camel.component.pqc.PQCKeyEncapsulationAlgorithms;
import org.apache.camel.component.pqc.PQCSignatureAlgorithms;
import org.apache.camel.component.pqc.lifecycle.KeyFormatConverter;
import org.apache.camel.component.pqc.lifecycle.KeyLifecycleManager;
import org.apache.camel.component.pqc.lifecycle.KeyMetadata;
import org.bouncycastle.pqc.crypto.lms.LMOtsParameters;
import org.bouncycastle.pqc.crypto.lms.LMSigParameters;
import org.bouncycastle.pqc.jcajce.spec.BIKEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.CMCEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.DilithiumParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FalconParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FrodoParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.HQCParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.LMSKeyGenParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRULPRimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRUParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SABERParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SNTRUPrimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SPHINCSPlusParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSMTParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSParameterSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.vault.authentication.ClientAuthentication;
import org.springframework.vault.authentication.TokenAuthentication;
import org.springframework.vault.client.VaultEndpoint;
import org.springframework.vault.core.VaultKeyValueOperations;
import org.springframework.vault.core.VaultKeyValueOperationsSupport;
import org.springframework.vault.core.VaultTemplate;
import org.springframework.vault.support.VaultResponse;

public class HashicorpVaultKeyLifecycleManager
implements KeyLifecycleManager {
    private static final Logger LOG = LoggerFactory.getLogger(HashicorpVaultKeyLifecycleManager.class);
    private final VaultTemplate vaultTemplate;
    private final String secretsEngine;
    private final String keyPrefix;
    private final boolean cloud;
    private final String namespace;
    private final ConcurrentHashMap<String, KeyPair> keyCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, KeyMetadata> metadataCache = new ConcurrentHashMap();

    public HashicorpVaultKeyLifecycleManager(VaultTemplate vaultTemplate, String secretsEngine, String keyPrefix) {
        this(vaultTemplate, secretsEngine, keyPrefix, false, null);
    }

    public HashicorpVaultKeyLifecycleManager(VaultTemplate vaultTemplate, String secretsEngine, String keyPrefix, boolean cloud, String namespace) {
        this.vaultTemplate = vaultTemplate;
        this.secretsEngine = secretsEngine != null ? secretsEngine : "secret";
        this.keyPrefix = keyPrefix != null ? keyPrefix : "pqc/keys";
        this.cloud = cloud;
        this.namespace = namespace;
        LOG.info("Initialized HashicorpVaultKeyLifecycleManager with secretsEngine: {}, keyPrefix: {}, cloud: {}, namespace: {}", new Object[]{this.secretsEngine, this.keyPrefix, this.cloud, this.namespace});
        try {
            this.loadExistingKeys();
        }
        catch (Exception e) {
            LOG.warn("Failed to load existing keys from Vault", (Throwable)e);
        }
    }

    public HashicorpVaultKeyLifecycleManager(String host, int port, String scheme, String token) {
        this(host, port, scheme, token, "secret", "pqc/keys", false, null);
    }

    public HashicorpVaultKeyLifecycleManager(String host, int port, String scheme, String token, String secretsEngine, String keyPrefix) {
        this(host, port, scheme, token, secretsEngine, keyPrefix, false, null);
    }

    public HashicorpVaultKeyLifecycleManager(String host, int port, String scheme, String token, String secretsEngine, String keyPrefix, boolean cloud, String namespace) {
        this.secretsEngine = secretsEngine != null ? secretsEngine : "secret";
        this.keyPrefix = keyPrefix != null ? keyPrefix : "pqc/keys";
        this.cloud = cloud;
        this.namespace = namespace;
        VaultEndpoint vaultEndpoint = new VaultEndpoint();
        vaultEndpoint.setHost(host);
        vaultEndpoint.setPort(port);
        vaultEndpoint.setScheme(scheme != null ? scheme : "https");
        this.vaultTemplate = new VaultTemplate(vaultEndpoint, (ClientAuthentication)new TokenAuthentication(token));
        LOG.info("Initialized HashicorpVaultKeyLifecycleManager with Vault at: {}://{}:{}, secretsEngine: {}, keyPrefix: {}, cloud: {}, namespace: {}", new Object[]{scheme, host, port, this.secretsEngine, this.keyPrefix, this.cloud, this.namespace});
        try {
            this.loadExistingKeys();
        }
        catch (Exception e) {
            LOG.warn("Failed to load existing keys from Vault", (Throwable)e);
        }
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId) throws Exception {
        return this.generateKeyPair(algorithm, keyId, null);
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId, Object parameterSpec) throws Exception {
        LOG.info("Generating key pair for algorithm: {}, keyId: {}", (Object)algorithm, (Object)keyId);
        String provider = this.determineProvider(algorithm);
        KeyPairGenerator generator = provider != null ? KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm), provider) : KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm));
        if (parameterSpec != null) {
            if (parameterSpec instanceof AlgorithmParameterSpec) {
                generator.initialize((AlgorithmParameterSpec)parameterSpec, new SecureRandom());
            } else if (parameterSpec instanceof Integer) {
                generator.initialize((Integer)parameterSpec, new SecureRandom());
            }
        } else {
            AlgorithmParameterSpec defaultSpec = this.getDefaultParameterSpec(algorithm);
            if (defaultSpec != null) {
                generator.initialize(defaultSpec, new SecureRandom());
            } else {
                generator.initialize(this.getDefaultKeySize(algorithm), new SecureRandom());
            }
        }
        KeyPair keyPair = generator.generateKeyPair();
        KeyMetadata metadata = new KeyMetadata(keyId, algorithm);
        metadata.setDescription("Generated on " + String.valueOf(new Date()));
        this.storeKey(keyId, keyPair, metadata);
        LOG.info("Generated key pair in Vault: {}", (Object)metadata);
        return keyPair;
    }

    @Override
    public byte[] exportKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format, boolean includePrivate) throws Exception {
        return KeyFormatConverter.exportKeyPair(keyPair, format, includePrivate);
    }

    @Override
    public byte[] exportPublicKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format) throws Exception {
        return KeyFormatConverter.exportPublicKey(keyPair.getPublic(), format);
    }

    @Override
    public KeyPair importKey(byte[] keyData, KeyLifecycleManager.KeyFormat format, String algorithm) throws Exception {
        try {
            PrivateKey privateKey = KeyFormatConverter.importPrivateKey(keyData, format, this.getAlgorithmName(algorithm));
            LOG.warn("Importing private key only - public key derivation may be needed");
            return new KeyPair(null, privateKey);
        }
        catch (Exception e) {
            PublicKey publicKey = KeyFormatConverter.importPublicKey(keyData, format, this.getAlgorithmName(algorithm));
            return new KeyPair(publicKey, null);
        }
    }

    @Override
    public KeyPair rotateKey(String oldKeyId, String newKeyId, String algorithm) throws Exception {
        LOG.info("Rotating key from {} to {}", (Object)oldKeyId, (Object)newKeyId);
        KeyMetadata oldMetadata = this.getKeyMetadata(oldKeyId);
        if (oldMetadata == null) {
            throw new IllegalArgumentException("Old key not found: " + oldKeyId);
        }
        oldMetadata.setStatus(KeyMetadata.KeyStatus.DEPRECATED);
        this.updateKeyMetadata(oldKeyId, oldMetadata);
        KeyPair newKeyPair = this.generateKeyPair(algorithm, newKeyId);
        LOG.info("Key rotation completed in Vault: {} -> {}", (Object)oldKeyId, (Object)newKeyId);
        return newKeyPair;
    }

    @Override
    public void storeKey(String keyId, KeyPair keyPair, KeyMetadata metadata) throws Exception {
        byte[] privateKeyBytes = keyPair.getPrivate().getEncoded();
        byte[] publicKeyBytes = keyPair.getPublic().getEncoded();
        String privateKeyBase64 = Base64.getEncoder().encodeToString(privateKeyBytes);
        String publicKeyBase64 = Base64.getEncoder().encodeToString(publicKeyBytes);
        String metadataBase64 = this.serializeMetadata(metadata);
        VaultKeyValueOperations keyValue = this.vaultTemplate.opsForKeyValue(this.secretsEngine, VaultKeyValueOperationsSupport.KeyValueBackend.versioned());
        HashMap<String, String> privateKeyData = new HashMap<String, String>();
        privateKeyData.put("key", privateKeyBase64);
        privateKeyData.put("format", "PKCS8");
        privateKeyData.put("algorithm", metadata.getAlgorithm());
        keyValue.put(this.getKeyPath(keyId) + "/private", privateKeyData);
        HashMap<String, String> publicKeyData = new HashMap<String, String>();
        publicKeyData.put("key", publicKeyBase64);
        publicKeyData.put("format", "X509");
        publicKeyData.put("algorithm", metadata.getAlgorithm());
        keyValue.put(this.getKeyPath(keyId) + "/public", publicKeyData);
        HashMap<String, String> metadataData = new HashMap<String, String>();
        metadataData.put("metadata", metadataBase64);
        metadataData.put("keyId", keyId);
        metadataData.put("algorithm", metadata.getAlgorithm());
        keyValue.put(this.getKeyPath(keyId) + "/metadata", metadataData);
        this.keyCache.put(keyId, keyPair);
        this.metadataCache.put(keyId, metadata);
        LOG.debug("Stored private key, public key, and metadata separately in Vault for: {}", (Object)keyId);
    }

    @Override
    public KeyPair getKey(String keyId) throws Exception {
        if (this.keyCache.containsKey(keyId)) {
            return this.keyCache.get(keyId);
        }
        String privateKeyPath = this.buildDataPath(this.getKeyPath(keyId) + "/private");
        VaultResponse privateResponse = this.vaultTemplate.read(privateKeyPath);
        if (privateResponse == null || privateResponse.getData() == null) {
            throw new IllegalArgumentException("Private key not found in Vault: " + keyId);
        }
        String publicKeyPath = this.buildDataPath(this.getKeyPath(keyId) + "/public");
        VaultResponse publicResponse = this.vaultTemplate.read(publicKeyPath);
        if (publicResponse == null || publicResponse.getData() == null) {
            throw new IllegalArgumentException("Public key not found in Vault: " + keyId);
        }
        Map privateResponseData = (Map)privateResponse.getData();
        Map privateData = (Map)privateResponseData.get("data");
        Map publicResponseData = (Map)publicResponse.getData();
        Map publicData = (Map)publicResponseData.get("data");
        if (privateData == null || publicData == null) {
            throw new IllegalArgumentException("Key data not found in Vault: " + keyId);
        }
        String privateKeyBase64 = (String)privateData.get("key");
        String publicKeyBase64 = (String)publicData.get("key");
        String algorithm = (String)privateData.get("algorithm");
        byte[] privateKeyBytes = Base64.getDecoder().decode(privateKeyBase64);
        byte[] publicKeyBytes = Base64.getDecoder().decode(publicKeyBase64);
        PrivateKey privateKey = KeyFormatConverter.importPrivateKey(privateKeyBytes, KeyLifecycleManager.KeyFormat.DER, this.getAlgorithmName(algorithm));
        PublicKey publicKey = KeyFormatConverter.importPublicKey(publicKeyBytes, KeyLifecycleManager.KeyFormat.DER, this.getAlgorithmName(algorithm));
        KeyPair keyPair = new KeyPair(publicKey, privateKey);
        this.keyCache.put(keyId, keyPair);
        return keyPair;
    }

    @Override
    public KeyMetadata getKeyMetadata(String keyId) throws Exception {
        if (this.metadataCache.containsKey(keyId)) {
            return this.metadataCache.get(keyId);
        }
        String metadataPath = this.buildDataPath(this.getKeyPath(keyId) + "/metadata");
        VaultResponse response = this.vaultTemplate.read(metadataPath);
        if (response == null || response.getData() == null) {
            return null;
        }
        Map responseData = (Map)response.getData();
        Map secretData = (Map)responseData.get("data");
        if (secretData == null) {
            return null;
        }
        String metadataBase64 = (String)secretData.get("metadata");
        KeyMetadata metadata = this.deserializeMetadata(metadataBase64);
        this.metadataCache.put(keyId, metadata);
        return metadata;
    }

    @Override
    public void updateKeyMetadata(String keyId, KeyMetadata metadata) throws Exception {
        KeyPair keyPair = this.getKey(keyId);
        this.storeKey(keyId, keyPair, metadata);
    }

    @Override
    public void deleteKey(String keyId) throws Exception {
        VaultKeyValueOperations keyValue = this.vaultTemplate.opsForKeyValue(this.secretsEngine, VaultKeyValueOperationsSupport.KeyValueBackend.versioned());
        keyValue.delete(this.getKeyPath(keyId) + "/private");
        keyValue.delete(this.getKeyPath(keyId) + "/public");
        keyValue.delete(this.getKeyPath(keyId) + "/metadata");
        this.keyCache.remove(keyId);
        this.metadataCache.remove(keyId);
        LOG.info("Deleted private key, public key, and metadata from Vault: {}", (Object)keyId);
    }

    @Override
    public List<KeyMetadata> listKeys() throws Exception {
        String metadataPath = this.buildMetadataPath(this.keyPrefix);
        List keyIds = this.vaultTemplate.list(metadataPath);
        ArrayList<KeyMetadata> metadataList = new ArrayList<KeyMetadata>();
        if (keyIds != null) {
            for (String keyId : keyIds) {
                try {
                    String cleanKeyId = keyId.endsWith("/") ? keyId.substring(0, keyId.length() - 1) : keyId;
                    KeyMetadata metadata = this.getKeyMetadata(cleanKeyId);
                    if (metadata == null) continue;
                    metadataList.add(metadata);
                }
                catch (Exception e) {
                    LOG.warn("Failed to load metadata for key: {}", (Object)keyId, (Object)e);
                }
            }
        }
        return metadataList;
    }

    @Override
    public boolean needsRotation(String keyId, Duration maxAge, long maxUsage) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata == null) {
            return false;
        }
        if (metadata.needsRotation()) {
            return true;
        }
        if (maxAge != null && metadata.getAgeInDays() > maxAge.toDays()) {
            return true;
        }
        return maxUsage > 0L && metadata.getUsageCount() >= maxUsage;
    }

    @Override
    public void expireKey(String keyId) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.EXPIRED);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Expired key in Vault: {}", (Object)keyId);
        }
    }

    @Override
    public void revokeKey(String keyId, String reason) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.REVOKED);
            metadata.setDescription((String)(metadata.getDescription() != null ? metadata.getDescription() + "; " : "") + "Revoked: " + reason);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Revoked key in Vault: {} - {}", (Object)keyId, (Object)reason);
        }
    }

    private void loadExistingKeys() throws Exception {
        String metadataPath = this.buildMetadataPath(this.keyPrefix);
        List keyIds = this.vaultTemplate.list(metadataPath);
        if (keyIds != null) {
            LOG.info("Found {} existing keys in Vault", (Object)keyIds.size());
            for (String keyId : keyIds) {
                try {
                    String cleanKeyId = keyId.endsWith("/") ? keyId.substring(0, keyId.length() - 1) : keyId;
                    KeyMetadata metadata = this.getKeyMetadata(cleanKeyId);
                    if (metadata == null) continue;
                    LOG.debug("Loaded existing key from Vault: {}", (Object)metadata);
                }
                catch (Exception e) {
                    LOG.warn("Failed to load key from Vault: {}", (Object)keyId, (Object)e);
                }
            }
        }
    }

    private String getKeyPath(String keyId) {
        return this.keyPrefix + "/" + keyId;
    }

    private String buildDataPath(String secretPath) {
        if (!this.cloud) {
            return this.secretsEngine + "/data/" + secretPath;
        }
        if (this.namespace != null && !this.namespace.isEmpty()) {
            return this.namespace + "/" + this.secretsEngine + "/data/" + secretPath;
        }
        return this.secretsEngine + "/data/" + secretPath;
    }

    private String buildMetadataPath(String secretPath) {
        if (!this.cloud) {
            return this.secretsEngine + "/metadata/" + secretPath;
        }
        if (this.namespace != null && !this.namespace.isEmpty()) {
            return this.namespace + "/" + this.secretsEngine + "/metadata/" + secretPath;
        }
        return this.secretsEngine + "/metadata/" + secretPath;
    }

    private String serializeMetadata(KeyMetadata metadata) throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (ObjectOutputStream oos = new ObjectOutputStream(baos);){
            oos.writeObject(metadata);
        }
        return Base64.getEncoder().encodeToString(baos.toByteArray());
    }

    private KeyMetadata deserializeMetadata(String base64) throws Exception {
        byte[] data = Base64.getDecoder().decode(base64);
        ByteArrayInputStream bais = new ByteArrayInputStream(data);
        try (ObjectInputStream ois = new ObjectInputStream(bais);){
            KeyMetadata keyMetadata = (KeyMetadata)ois.readObject();
            return keyMetadata;
        }
    }

    private String determineProvider(String algorithm) {
        try {
            PQCSignatureAlgorithms sigAlg = PQCSignatureAlgorithms.valueOf(algorithm);
            return sigAlg.getBcProvider();
        }
        catch (IllegalArgumentException e1) {
            try {
                PQCKeyEncapsulationAlgorithms kemAlg = PQCKeyEncapsulationAlgorithms.valueOf(algorithm);
                return kemAlg.getBcProvider();
            }
            catch (IllegalArgumentException e2) {
                return null;
            }
        }
    }

    private String getAlgorithmName(String algorithm) {
        try {
            return PQCSignatureAlgorithms.valueOf(algorithm).getAlgorithm();
        }
        catch (IllegalArgumentException e1) {
            try {
                return PQCKeyEncapsulationAlgorithms.valueOf(algorithm).getAlgorithm();
            }
            catch (IllegalArgumentException e2) {
                return algorithm;
            }
        }
    }

    private AlgorithmParameterSpec getDefaultParameterSpec(String algorithm) {
        try {
            switch (algorithm) {
                case "DILITHIUM": {
                    return DilithiumParameterSpec.dilithium2;
                }
                case "MLDSA": 
                case "SLHDSA": {
                    return null;
                }
                case "FALCON": {
                    return FalconParameterSpec.falcon_512;
                }
                case "SPHINCSPLUS": {
                    return SPHINCSPlusParameterSpec.sha2_128s;
                }
                case "XMSS": {
                    return new XMSSParameterSpec(10, "SHA256");
                }
                case "XMSSMT": {
                    return XMSSMTParameterSpec.XMSSMT_SHA2_20d2_256;
                }
                case "LMS": 
                case "HSS": {
                    return new LMSKeyGenParameterSpec(LMSigParameters.lms_sha256_n32_h10, LMOtsParameters.sha256_n32_w4);
                }
                case "MLKEM": 
                case "KYBER": {
                    return null;
                }
                case "NTRU": {
                    return NTRUParameterSpec.ntruhps2048509;
                }
                case "NTRULPRime": {
                    return NTRULPRimeParameterSpec.ntrulpr653;
                }
                case "SNTRUPrime": {
                    return SNTRUPrimeParameterSpec.sntrup761;
                }
                case "SABER": {
                    return SABERParameterSpec.lightsaberkem128r3;
                }
                case "FRODO": {
                    return FrodoParameterSpec.frodokem640aes;
                }
                case "BIKE": {
                    return BIKEParameterSpec.bike128;
                }
                case "HQC": {
                    return HQCParameterSpec.hqc128;
                }
                case "CMCE": {
                    return CMCEParameterSpec.mceliece348864;
                }
            }
            return null;
        }
        catch (Exception e) {
            LOG.warn("Failed to create default parameter spec for algorithm: {}", (Object)algorithm, (Object)e);
            return null;
        }
    }

    private int getDefaultKeySize(String algorithm) {
        return 256;
    }

    public VaultTemplate getVaultTemplate() {
        return this.vaultTemplate;
    }
}

