/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.pqc.lifecycle;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.camel.component.pqc.PQCKeyEncapsulationAlgorithms;
import org.apache.camel.component.pqc.PQCSignatureAlgorithms;
import org.apache.camel.component.pqc.lifecycle.KeyFormatConverter;
import org.apache.camel.component.pqc.lifecycle.KeyLifecycleManager;
import org.apache.camel.component.pqc.lifecycle.KeyMetadata;
import org.bouncycastle.pqc.crypto.lms.LMOtsParameters;
import org.bouncycastle.pqc.crypto.lms.LMSigParameters;
import org.bouncycastle.pqc.jcajce.spec.BIKEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.CMCEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.DilithiumParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FalconParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FrodoParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.HQCParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.LMSKeyGenParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRULPRimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRUParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SABERParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SNTRUPrimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SPHINCSPlusParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSMTParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSParameterSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedKeyLifecycleManager
implements KeyLifecycleManager {
    private static final Logger LOG = LoggerFactory.getLogger(FileBasedKeyLifecycleManager.class);
    private final Path keyDirectory;
    private final ConcurrentHashMap<String, KeyPair> keyCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, KeyMetadata> metadataCache = new ConcurrentHashMap();

    public FileBasedKeyLifecycleManager(String keyDirectoryPath) throws IOException {
        this.keyDirectory = Paths.get(keyDirectoryPath, new String[0]);
        Files.createDirectories(this.keyDirectory, new FileAttribute[0]);
        LOG.info("Initialized FileBasedKeyLifecycleManager with directory: {}", (Object)this.keyDirectory);
        this.loadExistingKeys();
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId) throws Exception {
        return this.generateKeyPair(algorithm, keyId, null);
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId, Object parameterSpec) throws Exception {
        LOG.info("Generating key pair for algorithm: {}, keyId: {}", (Object)algorithm, (Object)keyId);
        String provider = this.determineProvider(algorithm);
        KeyPairGenerator generator = provider != null ? KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm), provider) : KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm));
        if (parameterSpec != null) {
            if (parameterSpec instanceof AlgorithmParameterSpec) {
                generator.initialize((AlgorithmParameterSpec)parameterSpec, new SecureRandom());
            } else if (parameterSpec instanceof Integer) {
                generator.initialize((Integer)parameterSpec, new SecureRandom());
            }
        } else {
            AlgorithmParameterSpec defaultSpec = this.getDefaultParameterSpec(algorithm);
            if (defaultSpec != null) {
                generator.initialize(defaultSpec, new SecureRandom());
            } else {
                generator.initialize(this.getDefaultKeySize(algorithm), new SecureRandom());
            }
        }
        KeyPair keyPair = generator.generateKeyPair();
        KeyMetadata metadata = new KeyMetadata(keyId, algorithm);
        metadata.setDescription("Generated on " + String.valueOf(new Date()));
        this.storeKey(keyId, keyPair, metadata);
        LOG.info("Generated key pair: {}", (Object)metadata);
        return keyPair;
    }

    @Override
    public byte[] exportKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format, boolean includePrivate) throws Exception {
        return KeyFormatConverter.exportKeyPair(keyPair, format, includePrivate);
    }

    @Override
    public byte[] exportPublicKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format) throws Exception {
        return KeyFormatConverter.exportPublicKey(keyPair.getPublic(), format);
    }

    @Override
    public KeyPair importKey(byte[] keyData, KeyLifecycleManager.KeyFormat format, String algorithm) throws Exception {
        try {
            PrivateKey privateKey = KeyFormatConverter.importPrivateKey(keyData, format, this.getAlgorithmName(algorithm));
            LOG.warn("Importing private key only - public key derivation may be needed");
            return new KeyPair(null, privateKey);
        }
        catch (Exception e) {
            PublicKey publicKey = KeyFormatConverter.importPublicKey(keyData, format, this.getAlgorithmName(algorithm));
            return new KeyPair(publicKey, null);
        }
    }

    @Override
    public KeyPair rotateKey(String oldKeyId, String newKeyId, String algorithm) throws Exception {
        LOG.info("Rotating key from {} to {}", (Object)oldKeyId, (Object)newKeyId);
        KeyMetadata oldMetadata = this.getKeyMetadata(oldKeyId);
        if (oldMetadata == null) {
            throw new IllegalArgumentException("Old key not found: " + oldKeyId);
        }
        oldMetadata.setStatus(KeyMetadata.KeyStatus.DEPRECATED);
        this.updateKeyMetadata(oldKeyId, oldMetadata);
        KeyPair newKeyPair = this.generateKeyPair(algorithm, newKeyId);
        LOG.info("Key rotation completed: {} -> {}", (Object)oldKeyId, (Object)newKeyId);
        return newKeyPair;
    }

    @Override
    public void storeKey(String keyId, KeyPair keyPair, KeyMetadata metadata) throws Exception {
        Path keyFile = this.getKeyFile(keyId);
        try (ObjectOutputStream oos = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(keyFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)));){
            oos.writeObject(keyPair);
        }
        Path metadataFile = this.getMetadataFile(keyId);
        try (ObjectOutputStream oos = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(metadataFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)));){
            oos.writeObject(metadata);
        }
        this.keyCache.put(keyId, keyPair);
        this.metadataCache.put(keyId, metadata);
        LOG.debug("Stored key and metadata for: {}", (Object)keyId);
    }

    @Override
    public KeyPair getKey(String keyId) throws Exception {
        if (this.keyCache.containsKey(keyId)) {
            return this.keyCache.get(keyId);
        }
        Path keyFile = this.getKeyFile(keyId);
        if (!Files.exists(keyFile, new LinkOption[0])) {
            throw new IllegalArgumentException("Key not found: " + keyId);
        }
        try (ObjectInputStream ois = new ObjectInputStream(new BufferedInputStream(Files.newInputStream(keyFile, new OpenOption[0])));){
            KeyPair keyPair = (KeyPair)ois.readObject();
            this.keyCache.put(keyId, keyPair);
            KeyPair keyPair2 = keyPair;
            return keyPair2;
        }
    }

    @Override
    public KeyMetadata getKeyMetadata(String keyId) throws Exception {
        if (this.metadataCache.containsKey(keyId)) {
            return this.metadataCache.get(keyId);
        }
        Path metadataFile = this.getMetadataFile(keyId);
        if (!Files.exists(metadataFile, new LinkOption[0])) {
            return null;
        }
        try (ObjectInputStream ois = new ObjectInputStream(new BufferedInputStream(Files.newInputStream(metadataFile, new OpenOption[0])));){
            KeyMetadata metadata = (KeyMetadata)ois.readObject();
            this.metadataCache.put(keyId, metadata);
            KeyMetadata keyMetadata = metadata;
            return keyMetadata;
        }
    }

    @Override
    public void updateKeyMetadata(String keyId, KeyMetadata metadata) throws Exception {
        Path metadataFile = this.getMetadataFile(keyId);
        try (ObjectOutputStream oos = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(metadataFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING)));){
            oos.writeObject(metadata);
        }
        this.metadataCache.put(keyId, metadata);
    }

    @Override
    public void deleteKey(String keyId) throws Exception {
        Files.deleteIfExists(this.getKeyFile(keyId));
        Files.deleteIfExists(this.getMetadataFile(keyId));
        this.keyCache.remove(keyId);
        this.metadataCache.remove(keyId);
        LOG.info("Deleted key: {}", (Object)keyId);
    }

    @Override
    public List<KeyMetadata> listKeys() throws Exception {
        return new ArrayList<KeyMetadata>(this.metadataCache.values());
    }

    @Override
    public boolean needsRotation(String keyId, Duration maxAge, long maxUsage) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata == null) {
            return false;
        }
        if (metadata.needsRotation()) {
            return true;
        }
        if (maxAge != null && metadata.getAgeInDays() > maxAge.toDays()) {
            return true;
        }
        return maxUsage > 0L && metadata.getUsageCount() >= maxUsage;
    }

    @Override
    public void expireKey(String keyId) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.EXPIRED);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Expired key: {}", (Object)keyId);
        }
    }

    @Override
    public void revokeKey(String keyId, String reason) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.REVOKED);
            metadata.setDescription((String)(metadata.getDescription() != null ? metadata.getDescription() + "; " : "") + "Revoked: " + reason);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Revoked key: {} - {}", (Object)keyId, (Object)reason);
        }
    }

    private void loadExistingKeys() {
        try {
            Files.list(this.keyDirectory).filter(path -> path.toString().endsWith(".metadata")).forEach(path -> {
                try {
                    String keyId = path.getFileName().toString().replace(".metadata", "");
                    KeyMetadata metadata = this.getKeyMetadata(keyId);
                    if (metadata != null) {
                        LOG.debug("Loaded existing key: {}", (Object)metadata);
                    }
                }
                catch (Exception e) {
                    LOG.warn("Failed to load key metadata: {}", path, (Object)e);
                }
            });
        }
        catch (IOException e) {
            LOG.warn("Failed to list existing keys", (Throwable)e);
        }
    }

    private Path getKeyFile(String keyId) {
        return this.keyDirectory.resolve(keyId + ".key");
    }

    private Path getMetadataFile(String keyId) {
        return this.keyDirectory.resolve(keyId + ".metadata");
    }

    private String determineProvider(String algorithm) {
        try {
            PQCSignatureAlgorithms sigAlg = PQCSignatureAlgorithms.valueOf(algorithm);
            return sigAlg.getBcProvider();
        }
        catch (IllegalArgumentException e1) {
            try {
                PQCKeyEncapsulationAlgorithms kemAlg = PQCKeyEncapsulationAlgorithms.valueOf(algorithm);
                return kemAlg.getBcProvider();
            }
            catch (IllegalArgumentException e2) {
                return null;
            }
        }
    }

    private String getAlgorithmName(String algorithm) {
        try {
            return PQCSignatureAlgorithms.valueOf(algorithm).getAlgorithm();
        }
        catch (IllegalArgumentException e1) {
            try {
                return PQCKeyEncapsulationAlgorithms.valueOf(algorithm).getAlgorithm();
            }
            catch (IllegalArgumentException e2) {
                return algorithm;
            }
        }
    }

    private AlgorithmParameterSpec getDefaultParameterSpec(String algorithm) {
        try {
            switch (algorithm) {
                case "DILITHIUM": {
                    return DilithiumParameterSpec.dilithium2;
                }
                case "MLDSA": 
                case "SLHDSA": {
                    return null;
                }
                case "FALCON": {
                    return FalconParameterSpec.falcon_512;
                }
                case "SPHINCSPLUS": {
                    return SPHINCSPlusParameterSpec.sha2_128s;
                }
                case "XMSS": {
                    return new XMSSParameterSpec(10, "SHA256");
                }
                case "XMSSMT": {
                    return XMSSMTParameterSpec.XMSSMT_SHA2_20d2_256;
                }
                case "LMS": 
                case "HSS": {
                    return new LMSKeyGenParameterSpec(LMSigParameters.lms_sha256_n32_h10, LMOtsParameters.sha256_n32_w4);
                }
                case "MLKEM": 
                case "KYBER": {
                    return null;
                }
                case "NTRU": {
                    return NTRUParameterSpec.ntruhps2048509;
                }
                case "NTRULPRime": {
                    return NTRULPRimeParameterSpec.ntrulpr653;
                }
                case "SNTRUPrime": {
                    return SNTRUPrimeParameterSpec.sntrup761;
                }
                case "SABER": {
                    return SABERParameterSpec.lightsaberkem128r3;
                }
                case "FRODO": {
                    return FrodoParameterSpec.frodokem640aes;
                }
                case "BIKE": {
                    return BIKEParameterSpec.bike128;
                }
                case "HQC": {
                    return HQCParameterSpec.hqc128;
                }
                case "CMCE": {
                    return CMCEParameterSpec.mceliece348864;
                }
            }
            return null;
        }
        catch (Exception e) {
            LOG.warn("Failed to create default parameter spec for algorithm: {}", (Object)algorithm, (Object)e);
            return null;
        }
    }

    private int getDefaultKeySize(String algorithm) {
        return 256;
    }
}

