/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.pqc.lifecycle;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URI;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.camel.component.pqc.PQCKeyEncapsulationAlgorithms;
import org.apache.camel.component.pqc.PQCSignatureAlgorithms;
import org.apache.camel.component.pqc.lifecycle.KeyFormatConverter;
import org.apache.camel.component.pqc.lifecycle.KeyLifecycleManager;
import org.apache.camel.component.pqc.lifecycle.KeyMetadata;
import org.bouncycastle.pqc.crypto.lms.LMOtsParameters;
import org.bouncycastle.pqc.crypto.lms.LMSigParameters;
import org.bouncycastle.pqc.jcajce.spec.BIKEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.CMCEParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.DilithiumParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FalconParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.FrodoParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.HQCParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.LMSKeyGenParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRULPRimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.NTRUParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SABERParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SNTRUPrimeParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.SPHINCSPlusParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSMTParameterSpec;
import org.bouncycastle.pqc.jcajce.spec.XMSSParameterSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.secretsmanager.SecretsManagerClient;
import software.amazon.awssdk.services.secretsmanager.SecretsManagerClientBuilder;
import software.amazon.awssdk.services.secretsmanager.model.CreateSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.DeleteSecretRequest;
import software.amazon.awssdk.services.secretsmanager.model.GetSecretValueRequest;
import software.amazon.awssdk.services.secretsmanager.model.GetSecretValueResponse;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretsRequest;
import software.amazon.awssdk.services.secretsmanager.model.ListSecretsResponse;
import software.amazon.awssdk.services.secretsmanager.model.PutSecretValueRequest;
import software.amazon.awssdk.services.secretsmanager.model.ResourceNotFoundException;
import software.amazon.awssdk.services.secretsmanager.model.SecretListEntry;
import software.amazon.awssdk.services.secretsmanager.model.Tag;

public class AwsSecretsManagerKeyLifecycleManager
implements KeyLifecycleManager {
    private static final Logger LOG = LoggerFactory.getLogger(AwsSecretsManagerKeyLifecycleManager.class);
    private final SecretsManagerClient secretsManagerClient;
    private final String keyPrefix;
    private final ConcurrentHashMap<String, KeyPair> keyCache = new ConcurrentHashMap();
    private final ConcurrentHashMap<String, KeyMetadata> metadataCache = new ConcurrentHashMap();
    private final ObjectMapper objectMapper = new ObjectMapper();

    public AwsSecretsManagerKeyLifecycleManager(SecretsManagerClient secretsManagerClient, String keyPrefix) {
        this.secretsManagerClient = secretsManagerClient;
        this.keyPrefix = keyPrefix != null ? keyPrefix : "pqc/keys";
        LOG.info("Initialized AwsSecretsManagerKeyLifecycleManager with keyPrefix: {}", (Object)this.keyPrefix);
        try {
            this.loadExistingKeys();
        }
        catch (Exception e) {
            LOG.warn("Failed to load existing keys from AWS Secrets Manager", (Throwable)e);
        }
    }

    public AwsSecretsManagerKeyLifecycleManager(String region) {
        this(region, null, null, null);
    }

    public AwsSecretsManagerKeyLifecycleManager(String region, String accessKey, String secretKey, String keyPrefix) {
        this(region, accessKey, secretKey, keyPrefix, null);
    }

    public AwsSecretsManagerKeyLifecycleManager(String region, String accessKey, String secretKey, String keyPrefix, String endpointOverride) {
        this.keyPrefix = keyPrefix != null ? keyPrefix : "pqc/keys";
        SecretsManagerClientBuilder clientBuilder = SecretsManagerClient.builder();
        if (region != null) {
            clientBuilder.region(Region.of((String)region));
        }
        if (accessKey != null && secretKey != null) {
            AwsBasicCredentials credentials = AwsBasicCredentials.create((String)accessKey, (String)secretKey);
            clientBuilder.credentialsProvider((AwsCredentialsProvider)StaticCredentialsProvider.create((AwsCredentials)credentials));
        }
        if (endpointOverride != null) {
            clientBuilder.endpointOverride(URI.create(endpointOverride));
        }
        this.secretsManagerClient = (SecretsManagerClient)clientBuilder.build();
        LOG.info("Initialized AwsSecretsManagerKeyLifecycleManager with region: {}, keyPrefix: {}, endpointOverride: {}", new Object[]{region, this.keyPrefix, endpointOverride});
        try {
            this.loadExistingKeys();
        }
        catch (Exception e) {
            LOG.warn("Failed to load existing keys from AWS Secrets Manager", (Throwable)e);
        }
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId) throws Exception {
        return this.generateKeyPair(algorithm, keyId, null);
    }

    @Override
    public KeyPair generateKeyPair(String algorithm, String keyId, Object parameterSpec) throws Exception {
        LOG.info("Generating key pair for algorithm: {}, keyId: {}", (Object)algorithm, (Object)keyId);
        String provider = this.determineProvider(algorithm);
        KeyPairGenerator generator = provider != null ? KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm), provider) : KeyPairGenerator.getInstance(this.getAlgorithmName(algorithm));
        if (parameterSpec != null) {
            if (parameterSpec instanceof AlgorithmParameterSpec) {
                generator.initialize((AlgorithmParameterSpec)parameterSpec, new SecureRandom());
            } else if (parameterSpec instanceof Integer) {
                generator.initialize((Integer)parameterSpec, new SecureRandom());
            }
        } else {
            AlgorithmParameterSpec defaultSpec = this.getDefaultParameterSpec(algorithm);
            if (defaultSpec != null) {
                generator.initialize(defaultSpec, new SecureRandom());
            } else {
                generator.initialize(this.getDefaultKeySize(algorithm), new SecureRandom());
            }
        }
        KeyPair keyPair = generator.generateKeyPair();
        KeyMetadata metadata = new KeyMetadata(keyId, algorithm);
        metadata.setDescription("Generated on " + String.valueOf(new Date()));
        this.storeKey(keyId, keyPair, metadata);
        LOG.info("Generated key pair in AWS Secrets Manager: {}", (Object)metadata);
        return keyPair;
    }

    @Override
    public byte[] exportKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format, boolean includePrivate) throws Exception {
        return KeyFormatConverter.exportKeyPair(keyPair, format, includePrivate);
    }

    @Override
    public byte[] exportPublicKey(KeyPair keyPair, KeyLifecycleManager.KeyFormat format) throws Exception {
        return KeyFormatConverter.exportPublicKey(keyPair.getPublic(), format);
    }

    @Override
    public KeyPair importKey(byte[] keyData, KeyLifecycleManager.KeyFormat format, String algorithm) throws Exception {
        try {
            PrivateKey privateKey = KeyFormatConverter.importPrivateKey(keyData, format, this.getAlgorithmName(algorithm));
            LOG.warn("Importing private key only - public key derivation may be needed");
            return new KeyPair(null, privateKey);
        }
        catch (Exception e) {
            PublicKey publicKey = KeyFormatConverter.importPublicKey(keyData, format, this.getAlgorithmName(algorithm));
            return new KeyPair(publicKey, null);
        }
    }

    @Override
    public KeyPair rotateKey(String oldKeyId, String newKeyId, String algorithm) throws Exception {
        LOG.info("Rotating key from {} to {}", (Object)oldKeyId, (Object)newKeyId);
        KeyMetadata oldMetadata = this.getKeyMetadata(oldKeyId);
        if (oldMetadata == null) {
            throw new IllegalArgumentException("Old key not found: " + oldKeyId);
        }
        oldMetadata.setStatus(KeyMetadata.KeyStatus.DEPRECATED);
        this.updateKeyMetadata(oldKeyId, oldMetadata);
        KeyPair newKeyPair = this.generateKeyPair(algorithm, newKeyId);
        LOG.info("Key rotation completed in AWS Secrets Manager: {} -> {}", (Object)oldKeyId, (Object)newKeyId);
        return newKeyPair;
    }

    @Override
    public void storeKey(String keyId, KeyPair keyPair, KeyMetadata metadata) throws Exception {
        byte[] privateKeyBytes = keyPair.getPrivate().getEncoded();
        byte[] publicKeyBytes = keyPair.getPublic().getEncoded();
        String privateKeyBase64 = Base64.getEncoder().encodeToString(privateKeyBytes);
        String publicKeyBase64 = Base64.getEncoder().encodeToString(publicKeyBytes);
        String metadataBase64 = this.serializeMetadata(metadata);
        String privateSecretName = this.getSecretName(keyId, "private");
        String privateSecretValue = this.objectMapper.writeValueAsString((Object)new SecretData(privateKeyBase64, "PKCS8", metadata.getAlgorithm()));
        this.createOrUpdateSecret(privateSecretName, privateSecretValue, "PQC Private Key: " + keyId);
        String publicSecretName = this.getSecretName(keyId, "public");
        String publicSecretValue = this.objectMapper.writeValueAsString((Object)new SecretData(publicKeyBase64, "X509", metadata.getAlgorithm()));
        this.createOrUpdateSecret(publicSecretName, publicSecretValue, "PQC Public Key: " + keyId);
        String metadataSecretName = this.getSecretName(keyId, "metadata");
        String metadataSecretValue = this.objectMapper.writeValueAsString((Object)new MetadataData(metadataBase64, keyId, metadata.getAlgorithm()));
        this.createOrUpdateSecret(metadataSecretName, metadataSecretValue, "PQC Key Metadata: " + keyId);
        this.keyCache.put(keyId, keyPair);
        this.metadataCache.put(keyId, metadata);
        LOG.debug("Stored private key, public key, and metadata separately in AWS Secrets Manager for: {}", (Object)keyId);
    }

    @Override
    public KeyPair getKey(String keyId) throws Exception {
        if (this.keyCache.containsKey(keyId)) {
            return this.keyCache.get(keyId);
        }
        String privateSecretName = this.getSecretName(keyId, "private");
        GetSecretValueResponse privateResponse = this.getSecret(privateSecretName);
        String publicSecretName = this.getSecretName(keyId, "public");
        GetSecretValueResponse publicResponse = this.getSecret(publicSecretName);
        SecretData privateData = (SecretData)this.objectMapper.readValue(privateResponse.secretString(), SecretData.class);
        SecretData publicData = (SecretData)this.objectMapper.readValue(publicResponse.secretString(), SecretData.class);
        byte[] privateKeyBytes = Base64.getDecoder().decode(privateData.getKey());
        byte[] publicKeyBytes = Base64.getDecoder().decode(publicData.getKey());
        PrivateKey privateKey = KeyFormatConverter.importPrivateKey(privateKeyBytes, KeyLifecycleManager.KeyFormat.DER, this.getAlgorithmName(privateData.getAlgorithm()));
        PublicKey publicKey = KeyFormatConverter.importPublicKey(publicKeyBytes, KeyLifecycleManager.KeyFormat.DER, this.getAlgorithmName(publicData.getAlgorithm()));
        KeyPair keyPair = new KeyPair(publicKey, privateKey);
        this.keyCache.put(keyId, keyPair);
        return keyPair;
    }

    @Override
    public KeyMetadata getKeyMetadata(String keyId) throws Exception {
        if (this.metadataCache.containsKey(keyId)) {
            return this.metadataCache.get(keyId);
        }
        String metadataSecretName = this.getSecretName(keyId, "metadata");
        try {
            GetSecretValueResponse response = this.getSecret(metadataSecretName);
            MetadataData metadataData = (MetadataData)this.objectMapper.readValue(response.secretString(), MetadataData.class);
            KeyMetadata metadata = this.deserializeMetadata(metadataData.getMetadata());
            this.metadataCache.put(keyId, metadata);
            return metadata;
        }
        catch (ResourceNotFoundException e) {
            return null;
        }
    }

    @Override
    public void updateKeyMetadata(String keyId, KeyMetadata metadata) throws Exception {
        KeyPair keyPair = this.getKey(keyId);
        this.storeKey(keyId, keyPair, metadata);
    }

    @Override
    public void deleteKey(String keyId) throws Exception {
        this.deleteSecret(this.getSecretName(keyId, "private"));
        this.deleteSecret(this.getSecretName(keyId, "public"));
        this.deleteSecret(this.getSecretName(keyId, "metadata"));
        this.keyCache.remove(keyId);
        this.metadataCache.remove(keyId);
        LOG.info("Deleted private key, public key, and metadata from AWS Secrets Manager: {}", (Object)keyId);
    }

    @Override
    public List<KeyMetadata> listKeys() throws Exception {
        ListSecretsResponse response;
        ArrayList<KeyMetadata> metadataList = new ArrayList<KeyMetadata>();
        String nextToken = null;
        do {
            ListSecretsRequest.Builder requestBuilder = ListSecretsRequest.builder().maxResults(Integer.valueOf(100));
            if (nextToken != null) {
                requestBuilder.nextToken(nextToken);
            }
            response = this.secretsManagerClient.listSecrets((ListSecretsRequest)requestBuilder.build());
            for (SecretListEntry secret : response.secretList()) {
                String secretName = secret.name();
                if (!secretName.startsWith(this.keyPrefix) || !secretName.endsWith("/metadata")) continue;
                String keyId = this.extractKeyIdFromSecretName(secretName);
                try {
                    KeyMetadata metadata = this.getKeyMetadata(keyId);
                    if (metadata == null) continue;
                    metadataList.add(metadata);
                }
                catch (Exception e) {
                    LOG.warn("Failed to load metadata for key: {}", (Object)keyId, (Object)e);
                }
            }
        } while ((nextToken = response.nextToken()) != null);
        return metadataList;
    }

    @Override
    public boolean needsRotation(String keyId, Duration maxAge, long maxUsage) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata == null) {
            return false;
        }
        if (metadata.needsRotation()) {
            return true;
        }
        if (maxAge != null && metadata.getAgeInDays() > maxAge.toDays()) {
            return true;
        }
        return maxUsage > 0L && metadata.getUsageCount() >= maxUsage;
    }

    @Override
    public void expireKey(String keyId) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.EXPIRED);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Expired key in AWS Secrets Manager: {}", (Object)keyId);
        }
    }

    @Override
    public void revokeKey(String keyId, String reason) throws Exception {
        KeyMetadata metadata = this.getKeyMetadata(keyId);
        if (metadata != null) {
            metadata.setStatus(KeyMetadata.KeyStatus.REVOKED);
            metadata.setDescription((String)(metadata.getDescription() != null ? metadata.getDescription() + "; " : "") + "Revoked: " + reason);
            this.updateKeyMetadata(keyId, metadata);
            LOG.info("Revoked key in AWS Secrets Manager: {} - {}", (Object)keyId, (Object)reason);
        }
    }

    private void loadExistingKeys() throws Exception {
        List<KeyMetadata> keys = this.listKeys();
        if (!keys.isEmpty()) {
            LOG.info("Found {} existing keys in AWS Secrets Manager", (Object)keys.size());
            for (KeyMetadata metadata : keys) {
                LOG.debug("Loaded existing key from AWS Secrets Manager: {}", (Object)metadata);
            }
        }
    }

    private String getSecretName(String keyId, String type) {
        return this.keyPrefix + "/" + keyId + "/" + type;
    }

    private String extractKeyIdFromSecretName(String secretName) {
        String withoutPrefix = secretName.substring(this.keyPrefix.length() + 1);
        int lastSlash = withoutPrefix.lastIndexOf(47);
        return withoutPrefix.substring(0, lastSlash);
    }

    private void createOrUpdateSecret(String secretName, String secretValue, String description) {
        try {
            PutSecretValueRequest putRequest = (PutSecretValueRequest)PutSecretValueRequest.builder().secretId(secretName).secretString(secretValue).build();
            this.secretsManagerClient.putSecretValue(putRequest);
            LOG.debug("Updated secret: {}", (Object)secretName);
        }
        catch (ResourceNotFoundException e) {
            CreateSecretRequest createRequest = (CreateSecretRequest)CreateSecretRequest.builder().name(secretName).secretString(secretValue).description(description).tags(new Tag[]{(Tag)Tag.builder().key("ManagedBy").value("camel-pqc").build()}).build();
            this.secretsManagerClient.createSecret(createRequest);
            LOG.debug("Created secret: {}", (Object)secretName);
        }
    }

    private GetSecretValueResponse getSecret(String secretName) {
        GetSecretValueRequest request = (GetSecretValueRequest)GetSecretValueRequest.builder().secretId(secretName).build();
        return this.secretsManagerClient.getSecretValue(request);
    }

    private void deleteSecret(String secretName) {
        try {
            DeleteSecretRequest request = (DeleteSecretRequest)DeleteSecretRequest.builder().secretId(secretName).forceDeleteWithoutRecovery(Boolean.valueOf(true)).build();
            this.secretsManagerClient.deleteSecret(request);
            LOG.debug("Deleted secret: {}", (Object)secretName);
        }
        catch (ResourceNotFoundException e) {
            LOG.debug("Secret not found, skipping deletion: {}", (Object)secretName);
        }
    }

    private String serializeMetadata(KeyMetadata metadata) throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (ObjectOutputStream oos = new ObjectOutputStream(baos);){
            oos.writeObject(metadata);
        }
        return Base64.getEncoder().encodeToString(baos.toByteArray());
    }

    private KeyMetadata deserializeMetadata(String base64) throws Exception {
        byte[] data = Base64.getDecoder().decode(base64);
        ByteArrayInputStream bais = new ByteArrayInputStream(data);
        try (ObjectInputStream ois = new ObjectInputStream(bais);){
            KeyMetadata keyMetadata = (KeyMetadata)ois.readObject();
            return keyMetadata;
        }
    }

    private String determineProvider(String algorithm) {
        try {
            PQCSignatureAlgorithms sigAlg = PQCSignatureAlgorithms.valueOf(algorithm);
            return sigAlg.getBcProvider();
        }
        catch (IllegalArgumentException e1) {
            try {
                PQCKeyEncapsulationAlgorithms kemAlg = PQCKeyEncapsulationAlgorithms.valueOf(algorithm);
                return kemAlg.getBcProvider();
            }
            catch (IllegalArgumentException e2) {
                return null;
            }
        }
    }

    private String getAlgorithmName(String algorithm) {
        try {
            return PQCSignatureAlgorithms.valueOf(algorithm).getAlgorithm();
        }
        catch (IllegalArgumentException e1) {
            try {
                return PQCKeyEncapsulationAlgorithms.valueOf(algorithm).getAlgorithm();
            }
            catch (IllegalArgumentException e2) {
                return algorithm;
            }
        }
    }

    private AlgorithmParameterSpec getDefaultParameterSpec(String algorithm) {
        try {
            switch (algorithm) {
                case "DILITHIUM": {
                    return DilithiumParameterSpec.dilithium2;
                }
                case "MLDSA": 
                case "SLHDSA": {
                    return null;
                }
                case "FALCON": {
                    return FalconParameterSpec.falcon_512;
                }
                case "SPHINCSPLUS": {
                    return SPHINCSPlusParameterSpec.sha2_128s;
                }
                case "XMSS": {
                    return new XMSSParameterSpec(10, "SHA256");
                }
                case "XMSSMT": {
                    return XMSSMTParameterSpec.XMSSMT_SHA2_20d2_256;
                }
                case "LMS": 
                case "HSS": {
                    return new LMSKeyGenParameterSpec(LMSigParameters.lms_sha256_n32_h10, LMOtsParameters.sha256_n32_w4);
                }
                case "MLKEM": 
                case "KYBER": {
                    return null;
                }
                case "NTRU": {
                    return NTRUParameterSpec.ntruhps2048509;
                }
                case "NTRULPRime": {
                    return NTRULPRimeParameterSpec.ntrulpr653;
                }
                case "SNTRUPrime": {
                    return SNTRUPrimeParameterSpec.sntrup761;
                }
                case "SABER": {
                    return SABERParameterSpec.lightsaberkem128r3;
                }
                case "FRODO": {
                    return FrodoParameterSpec.frodokem640aes;
                }
                case "BIKE": {
                    return BIKEParameterSpec.bike128;
                }
                case "HQC": {
                    return HQCParameterSpec.hqc128;
                }
                case "CMCE": {
                    return CMCEParameterSpec.mceliece348864;
                }
            }
            return null;
        }
        catch (Exception e) {
            LOG.warn("Failed to create default parameter spec for algorithm: {}", (Object)algorithm, (Object)e);
            return null;
        }
    }

    private int getDefaultKeySize(String algorithm) {
        return 256;
    }

    public SecretsManagerClient getSecretsManagerClient() {
        return this.secretsManagerClient;
    }

    private static class SecretData {
        private String key;
        private String format;
        private String algorithm;

        public SecretData() {
        }

        public SecretData(String key, String format, String algorithm) {
            this.key = key;
            this.format = format;
            this.algorithm = algorithm;
        }

        public String getKey() {
            return this.key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public String getFormat() {
            return this.format;
        }

        public void setFormat(String format) {
            this.format = format;
        }

        public String getAlgorithm() {
            return this.algorithm;
        }

        public void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }
    }

    private static class MetadataData {
        private String metadata;
        private String keyId;
        private String algorithm;

        public MetadataData() {
        }

        public MetadataData(String metadata, String keyId, String algorithm) {
            this.metadata = metadata;
            this.keyId = keyId;
            this.algorithm = algorithm;
        }

        public String getMetadata() {
            return this.metadata;
        }

        public void setMetadata(String metadata) {
            this.metadata = metadata;
        }

        public String getKeyId() {
            return this.keyId;
        }

        public void setKeyId(String keyId) {
            this.keyId = keyId;
        }

        public String getAlgorithm() {
            return this.algorithm;
        }

        public void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }
    }
}

