/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.keycloak.security.cache;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.camel.component.keycloak.security.KeycloakTokenIntrospector;
import org.apache.camel.component.keycloak.security.cache.TokenCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConcurrentMapTokenCache
implements TokenCache {
    private static final Logger LOG = LoggerFactory.getLogger(ConcurrentMapTokenCache.class);
    private final Map<String, CachedEntry> cache;
    private final long ttlMillis;
    private final AtomicLong hitCount = new AtomicLong(0L);
    private final AtomicLong missCount = new AtomicLong(0L);
    private final AtomicLong evictionCount = new AtomicLong(0L);

    public ConcurrentMapTokenCache(long ttlSeconds) {
        this.cache = new ConcurrentHashMap<String, CachedEntry>();
        this.ttlMillis = ttlSeconds * 1000L;
    }

    @Override
    public KeycloakTokenIntrospector.IntrospectionResult get(String token) {
        CachedEntry entry = this.cache.get(token);
        if (entry != null) {
            if (!entry.isExpired()) {
                this.hitCount.incrementAndGet();
                LOG.trace("Cache hit for token");
                return entry.result;
            }
            this.cache.remove(token);
            this.evictionCount.incrementAndGet();
            LOG.trace("Cache entry expired and removed");
        }
        this.missCount.incrementAndGet();
        LOG.trace("Cache miss for token");
        return null;
    }

    @Override
    public void put(String token, KeycloakTokenIntrospector.IntrospectionResult result) {
        this.cache.put(token, new CachedEntry(result, this.ttlMillis));
        LOG.trace("Token introspection result cached");
        this.cleanupExpiredEntries();
    }

    @Override
    public void remove(String token) {
        this.cache.remove(token);
        LOG.trace("Token removed from cache");
    }

    @Override
    public void clear() {
        this.cache.clear();
        this.hitCount.set(0L);
        this.missCount.set(0L);
        this.evictionCount.set(0L);
        LOG.debug("Cache cleared");
    }

    @Override
    public long size() {
        return this.cache.size();
    }

    @Override
    public TokenCache.CacheStats getStats() {
        return new TokenCache.CacheStats(this.hitCount.get(), this.missCount.get(), this.evictionCount.get());
    }

    private void cleanupExpiredEntries() {
        if (this.cache.isEmpty()) {
            return;
        }
        this.cache.entrySet().removeIf(entry -> {
            if (((CachedEntry)entry.getValue()).isExpired()) {
                this.evictionCount.incrementAndGet();
                LOG.trace("Removing expired cache entry during cleanup");
                return true;
            }
            return false;
        });
    }

    private static class CachedEntry {
        private final KeycloakTokenIntrospector.IntrospectionResult result;
        private final long expirationTime;

        CachedEntry(KeycloakTokenIntrospector.IntrospectionResult result, long ttlMillis) {
            this.result = result;
            this.expirationTime = System.currentTimeMillis() + ttlMillis;
        }

        boolean isExpired() {
            return System.currentTimeMillis() >= this.expirationTime;
        }
    }
}

