/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.keycloak.security;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import org.apache.camel.component.keycloak.security.cache.TokenCache;
import org.apache.camel.component.keycloak.security.cache.TokenCacheFactory;
import org.apache.camel.component.keycloak.security.cache.TokenCacheType;
import org.apache.camel.util.ObjectHelper;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.entity.UrlEncodedFormEntity;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ClassicHttpResponse;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.ParseException;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeycloakTokenIntrospector {
    private static final Logger LOG = LoggerFactory.getLogger(KeycloakTokenIntrospector.class);
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final String serverUrl;
    private final String realm;
    private final String clientId;
    private final String clientSecret;
    private final CloseableHttpClient httpClient;
    private final TokenCache cache;

    public KeycloakTokenIntrospector(String serverUrl, String realm, String clientId, String clientSecret, boolean cacheEnabled, long cacheTtlSeconds) {
        this(serverUrl, realm, clientId, clientSecret, cacheEnabled ? TokenCacheFactory.createCache(cacheTtlSeconds) : null);
    }

    public KeycloakTokenIntrospector(String serverUrl, String realm, String clientId, String clientSecret, TokenCacheType cacheType, long cacheTtlSeconds, long maxCacheSize, boolean recordStats) {
        this(serverUrl, realm, clientId, clientSecret, TokenCacheFactory.createCache(cacheType, cacheTtlSeconds, maxCacheSize, recordStats));
    }

    public KeycloakTokenIntrospector(String serverUrl, String realm, String clientId, String clientSecret, TokenCache cache) {
        this.serverUrl = serverUrl;
        this.realm = realm;
        this.clientId = clientId;
        this.clientSecret = clientSecret;
        this.httpClient = HttpClients.createDefault();
        this.cache = cache;
    }

    public IntrospectionResult introspect(String token) throws IOException {
        IntrospectionResult cached;
        if (ObjectHelper.isEmpty((String)token)) {
            throw new IllegalArgumentException("Token cannot be null or empty");
        }
        if (this.cache != null && (cached = this.cache.get(token)) != null) {
            LOG.debug("Returning cached introspection result for token");
            return cached;
        }
        String introspectionUrl = String.format("%s/realms/%s/protocol/openid-connect/token/introspect", this.serverUrl, this.realm);
        LOG.debug("Introspecting token at: {}", (Object)introspectionUrl);
        HttpPost request = new HttpPost(introspectionUrl);
        String auth = this.clientId + ":" + this.clientSecret;
        String encodedAuth = Base64.getEncoder().encodeToString(auth.getBytes());
        request.setHeader("Authorization", (Object)("Basic " + encodedAuth));
        request.setHeader("Content-Type", (Object)"application/x-www-form-urlencoded");
        BasicNameValuePair tokenParam = new BasicNameValuePair("token", token);
        request.setEntity((HttpEntity)new UrlEncodedFormEntity(List.of(tokenParam)));
        try {
            IntrospectionResult result = (IntrospectionResult)this.httpClient.execute((ClassicHttpRequest)request, response -> this.parseIntrospectionResponse(response));
            if (this.cache != null && result != null) {
                this.cache.put(token, result);
            }
            return result;
        }
        catch (Exception e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            throw new IOException("Failed to introspect token", e);
        }
    }

    private IntrospectionResult parseIntrospectionResponse(ClassicHttpResponse response) throws IOException {
        try {
            int statusCode = response.getCode();
            if (statusCode != 200) {
                String errorBody = EntityUtils.toString((HttpEntity)response.getEntity());
                LOG.error("Introspection request failed with status {}: {}", (Object)statusCode, (Object)errorBody);
                throw new IOException("Token introspection failed with status " + statusCode + ": " + errorBody);
            }
            String responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            LOG.debug("Introspection response: {}", (Object)responseBody);
            Map responseMap = (Map)OBJECT_MAPPER.readValue(responseBody, Map.class);
            return new IntrospectionResult(responseMap);
        }
        catch (ParseException e) {
            throw new IOException("Failed to parse HTTP response", e);
        }
    }

    public void clearCache() {
        if (this.cache != null) {
            this.cache.clear();
            LOG.debug("Introspection cache cleared");
        }
    }

    public TokenCache.CacheStats getCacheStats() {
        return this.cache != null ? this.cache.getStats() : null;
    }

    public long getCacheSize() {
        return this.cache != null ? this.cache.size() : 0L;
    }

    public void close() {
        try {
            if (this.httpClient != null) {
                this.httpClient.close();
            }
        }
        catch (IOException e) {
            LOG.warn("Error closing HTTP client", (Throwable)e);
        }
        if (this.cache != null) {
            this.cache.close();
        }
    }

    public static class IntrospectionResult {
        private final Map<String, Object> claims;

        public IntrospectionResult(Map<String, Object> claims) {
            this.claims = claims;
        }

        public boolean isActive() {
            Object active = this.claims.get("active");
            return active instanceof Boolean && (Boolean)active != false;
        }

        public String getSubject() {
            return (String)this.claims.get("sub");
        }

        public String getUsername() {
            return (String)this.claims.get("username");
        }

        public String getScope() {
            return (String)this.claims.get("scope");
        }

        public String getClientId() {
            return (String)this.claims.get("client_id");
        }

        public String getTokenType() {
            return (String)this.claims.get("token_type");
        }

        public Long getExpiration() {
            Object exp = this.claims.get("exp");
            if (exp instanceof Number) {
                return ((Number)exp).longValue();
            }
            return null;
        }

        public Long getIssuedAt() {
            Object iat = this.claims.get("iat");
            if (iat instanceof Number) {
                return ((Number)iat).longValue();
            }
            return null;
        }

        public Map<String, Object> getAllClaims() {
            return this.claims;
        }

        public Object getClaim(String claimName) {
            return this.claims.get(claimName);
        }
    }
}

