/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.docling;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.apache.camel.component.docling.AuthenticationScheme;
import org.apache.camel.component.docling.BatchConversionResult;
import org.apache.camel.component.docling.BatchProcessingResults;
import org.apache.camel.component.docling.ConversionStatus;
import org.apache.camel.component.docling.DoclingMetadataFields;
import org.apache.camel.component.docling.DocumentMetadata;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.config.ConnectionConfig;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClientBuilder;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManager;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.ParseException;
import org.apache.hc.core5.http.io.SocketConfig;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.pool.PoolStats;
import org.apache.hc.core5.util.TimeValue;
import org.apache.hc.core5.util.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DoclingServeClient {
    private static final Logger LOG = LoggerFactory.getLogger(DoclingServeClient.class);
    private static final String DEFAULT_CONVERT_ENDPOINT = "/v1/convert/source";
    private static final String DEFAULT_ASYNC_CONVERT_ENDPOINT = "/v1/convert/source/async";
    private final String baseUrl;
    private final ObjectMapper objectMapper;
    private final CloseableHttpClient httpClient;
    private final PoolingHttpClientConnectionManager connectionManager;
    private final AuthenticationScheme authenticationScheme;
    private final String authenticationToken;
    private final String apiKeyHeader;
    private final String convertEndpoint;
    private final long asyncPollInterval;
    private final long asyncTimeout;

    public DoclingServeClient(String baseUrl) {
        this(baseUrl, AuthenticationScheme.NONE, null, "X-API-Key", DEFAULT_CONVERT_ENDPOINT, 2000L, 300000L, 20, 10, 30000, 60000, 30000, -1L, 2000, true, 60000L);
    }

    public DoclingServeClient(String baseUrl, AuthenticationScheme authenticationScheme, String authenticationToken, String apiKeyHeader) {
        this(baseUrl, authenticationScheme, authenticationToken, apiKeyHeader, DEFAULT_CONVERT_ENDPOINT, 2000L, 300000L, 20, 10, 30000, 60000, 30000, -1L, 2000, true, 60000L);
    }

    public DoclingServeClient(String baseUrl, AuthenticationScheme authenticationScheme, String authenticationToken, String apiKeyHeader, String convertEndpoint) {
        this(baseUrl, authenticationScheme, authenticationToken, apiKeyHeader, convertEndpoint, 2000L, 300000L, 20, 10, 30000, 60000, 30000, -1L, 2000, true, 60000L);
    }

    public DoclingServeClient(String baseUrl, AuthenticationScheme authenticationScheme, String authenticationToken, String apiKeyHeader, String convertEndpoint, long asyncPollInterval, long asyncTimeout) {
        this(baseUrl, authenticationScheme, authenticationToken, apiKeyHeader, convertEndpoint, asyncPollInterval, asyncTimeout, 20, 10, 30000, 60000, 30000, -1L, 2000, true, 60000L);
    }

    public DoclingServeClient(String baseUrl, AuthenticationScheme authenticationScheme, String authenticationToken, String apiKeyHeader, String convertEndpoint, long asyncPollInterval, long asyncTimeout, int maxTotalConnections, int maxConnectionsPerRoute, int connectionTimeout, int socketTimeout, int connectionRequestTimeout, long connectionTimeToLive, int validateAfterInactivity, boolean evictIdleConnections, long maxIdleTime) {
        this.baseUrl = baseUrl.endsWith("/") ? baseUrl.substring(0, baseUrl.length() - 1) : baseUrl;
        this.objectMapper = new ObjectMapper();
        this.authenticationScheme = authenticationScheme != null ? authenticationScheme : AuthenticationScheme.NONE;
        this.authenticationToken = authenticationToken;
        this.apiKeyHeader = apiKeyHeader != null ? apiKeyHeader : "X-API-Key";
        this.convertEndpoint = convertEndpoint != null ? convertEndpoint : DEFAULT_CONVERT_ENDPOINT;
        this.asyncPollInterval = asyncPollInterval;
        this.asyncTimeout = asyncTimeout;
        this.connectionManager = this.buildConnectionManager(maxTotalConnections, maxConnectionsPerRoute, connectionTimeout, socketTimeout, connectionTimeToLive, validateAfterInactivity);
        this.httpClient = this.buildHttpClient(this.connectionManager, connectionRequestTimeout, socketTimeout, evictIdleConnections, maxIdleTime);
        LOG.info("DoclingServeClient initialized with connection pool: maxTotal={}, maxPerRoute={}, connTimeout={}ms, socketTimeout={}ms", new Object[]{maxTotalConnections, maxConnectionsPerRoute, connectionTimeout, socketTimeout});
    }

    private PoolingHttpClientConnectionManager buildConnectionManager(int maxTotalConnections, int maxConnectionsPerRoute, int connectionTimeout, int socketTimeout, long connectionTimeToLive, int validateAfterInactivity) {
        SocketConfig socketConfig = SocketConfig.custom().setSoTimeout(Timeout.ofMilliseconds((long)socketTimeout)).build();
        ConnectionConfig connectionConfig = ConnectionConfig.custom().setConnectTimeout(Timeout.ofMilliseconds((long)connectionTimeout)).setSocketTimeout(Timeout.ofMilliseconds((long)socketTimeout)).setValidateAfterInactivity(TimeValue.ofMilliseconds((long)validateAfterInactivity)).setTimeToLive(connectionTimeToLive > 0L ? TimeValue.ofMilliseconds((long)connectionTimeToLive) : TimeValue.NEG_ONE_MILLISECOND).build();
        PoolingHttpClientConnectionManager connManager = PoolingHttpClientConnectionManagerBuilder.create().setMaxConnTotal(maxTotalConnections).setMaxConnPerRoute(maxConnectionsPerRoute).setDefaultSocketConfig(socketConfig).setDefaultConnectionConfig(connectionConfig).build();
        LOG.debug("Connection manager configured: maxTotal={}, maxPerRoute={}, validateAfterInactivity={}ms, ttl={}ms", new Object[]{maxTotalConnections, maxConnectionsPerRoute, validateAfterInactivity, connectionTimeToLive > 0L ? Long.valueOf(connectionTimeToLive) : "infinite"});
        return connManager;
    }

    private CloseableHttpClient buildHttpClient(PoolingHttpClientConnectionManager connectionManager, int connectionRequestTimeout, int socketTimeout, boolean evictIdleConnections, long maxIdleTime) {
        RequestConfig requestConfig = RequestConfig.custom().setConnectionRequestTimeout(Timeout.ofMilliseconds((long)connectionRequestTimeout)).setResponseTimeout(Timeout.ofMilliseconds((long)socketTimeout)).build();
        HttpClientBuilder clientBuilder = HttpClients.custom().setConnectionManager((HttpClientConnectionManager)connectionManager).setDefaultRequestConfig(requestConfig);
        if (evictIdleConnections) {
            clientBuilder.evictIdleConnections(TimeValue.ofMilliseconds((long)maxIdleTime));
            LOG.debug("Idle connection eviction enabled: maxIdleTime={}ms", (Object)maxIdleTime);
        }
        CloseableHttpClient client = clientBuilder.build();
        LOG.debug("HTTP client configured: connectionRequestTimeout={}ms, socketTimeout={}ms", (Object)connectionRequestTimeout, (Object)socketTimeout);
        return client;
    }

    public String convertDocument(String inputSource, String outputFormat) throws IOException {
        LOG.debug("Converting document using docling-serve API: {}", (Object)inputSource);
        if (inputSource.startsWith("http://") || inputSource.startsWith("https://")) {
            return this.convertFromUrl(inputSource, outputFormat);
        }
        return this.convertFromFile(inputSource, outputFormat);
    }

    private String convertFromUrl(String url, String outputFormat) throws IOException {
        HashMap<String, Object> requestBody = new HashMap<String, Object>();
        HashMap<String, String> source = new HashMap<String, String>();
        source.put("kind", "http");
        source.put("url", url);
        requestBody.put("sources", Collections.singletonList(source));
        if (outputFormat != null && !outputFormat.isEmpty()) {
            HashMap<String, List<String>> options = new HashMap<String, List<String>>();
            options.put("to_formats", Collections.singletonList(this.mapOutputFormat(outputFormat)));
            requestBody.put("options", options);
        }
        String jsonRequest = this.objectMapper.writeValueAsString(requestBody);
        LOG.debug("Request body: {}", (Object)jsonRequest);
        HttpPost httpPost = new HttpPost(this.baseUrl + this.convertEndpoint);
        httpPost.setEntity((HttpEntity)new StringEntity(jsonRequest, ContentType.APPLICATION_JSON));
        httpPost.setHeader("Accept", (Object)"application/json");
        this.applyAuthentication(httpPost);
        try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)httpPost);){
            String responseBody;
            int statusCode = response.getCode();
            try {
                responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            }
            catch (ParseException e) {
                throw new IOException("Failed to parse response from docling-serve API", e);
            }
            if (statusCode >= 200 && statusCode < 300) {
                String string = this.extractConvertedContent(responseBody, outputFormat);
                return string;
            }
            throw new IOException("Docling-serve API request failed with status " + statusCode + ": " + responseBody);
        }
    }

    private String convertFromFile(String filePath, String outputFormat) throws IOException {
        File file = new File(filePath);
        if (!file.exists()) {
            throw new IOException("File not found: " + filePath);
        }
        byte[] fileBytes = Files.readAllBytes(file.toPath());
        String base64Content = Base64.getEncoder().encodeToString(fileBytes);
        HashMap<String, Object> requestBody = new HashMap<String, Object>();
        HashMap<String, String> source = new HashMap<String, String>();
        source.put("kind", "file");
        source.put("base64_string", base64Content);
        source.put("filename", file.getName());
        requestBody.put("sources", Collections.singletonList(source));
        if (outputFormat != null && !outputFormat.isEmpty()) {
            HashMap<String, List<String>> options = new HashMap<String, List<String>>();
            options.put("to_formats", Collections.singletonList(this.mapOutputFormat(outputFormat)));
            requestBody.put("options", options);
        }
        String jsonRequest = this.objectMapper.writeValueAsString(requestBody);
        LOG.debug("Request body: {}", (Object)jsonRequest);
        HttpPost httpPost = new HttpPost(this.baseUrl + this.convertEndpoint);
        httpPost.setEntity((HttpEntity)new StringEntity(jsonRequest, ContentType.APPLICATION_JSON));
        httpPost.setHeader("Accept", (Object)"application/json");
        this.applyAuthentication(httpPost);
        try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)httpPost);){
            String responseBody;
            int statusCode = response.getCode();
            try {
                responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            }
            catch (ParseException e) {
                throw new IOException("Failed to parse response from docling-serve API", e);
            }
            if (statusCode >= 200 && statusCode < 300) {
                String string = this.extractConvertedContent(responseBody, outputFormat);
                return string;
            }
            throw new IOException("Docling-serve API request failed with status " + statusCode + ": " + responseBody);
        }
    }

    private String extractConvertedContent(String responseBody, String outputFormat) throws IOException {
        try {
            JsonNode rootNode = this.objectMapper.readTree(responseBody);
            if (rootNode.has("documents") && rootNode.get("documents").isArray() && rootNode.get("documents").size() > 0) {
                JsonNode firstDoc = rootNode.get("documents").get(0);
                if (firstDoc.has("content")) {
                    return firstDoc.get("content").asText();
                }
                if (firstDoc.has("markdown")) {
                    return firstDoc.get("markdown").asText();
                }
                if (firstDoc.has("text")) {
                    return firstDoc.get("text").asText();
                }
                return this.objectMapper.writeValueAsString((Object)firstDoc);
            }
            if (rootNode.has("content")) {
                return rootNode.get("content").asText();
            }
            return this.objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString((Object)rootNode);
        }
        catch (Exception e) {
            LOG.warn("Failed to parse JSON response, returning raw response", (Throwable)e);
            return responseBody;
        }
    }

    private String mapOutputFormat(String outputFormat) {
        if (outputFormat == null) {
            return "md";
        }
        switch (outputFormat.toLowerCase()) {
            case "markdown": 
            case "md": {
                return "md";
            }
            case "html": {
                return "html";
            }
            case "json": {
                return "json";
            }
            case "text": 
            case "txt": {
                return "text";
            }
        }
        return "md";
    }

    private void applyAuthentication(HttpPost httpPost) {
        if (this.authenticationScheme == null || this.authenticationScheme == AuthenticationScheme.NONE) {
            return;
        }
        if (this.authenticationToken == null || this.authenticationToken.isEmpty()) {
            LOG.warn("Authentication scheme is set to {} but no authentication token provided", (Object)this.authenticationScheme);
            return;
        }
        switch (this.authenticationScheme) {
            case BEARER: {
                httpPost.setHeader("Authorization", (Object)("Bearer " + this.authenticationToken));
                LOG.debug("Applied Bearer token authentication");
                break;
            }
            case API_KEY: {
                httpPost.setHeader(this.apiKeyHeader, (Object)this.authenticationToken);
                LOG.debug("Applied API Key authentication with header: {}", (Object)this.apiKeyHeader);
                break;
            }
            default: {
                LOG.warn("Unknown authentication scheme: {}", (Object)this.authenticationScheme);
            }
        }
    }

    private void applyAuthenticationGet(HttpGet httpGet) {
        if (this.authenticationScheme == null || this.authenticationScheme == AuthenticationScheme.NONE) {
            return;
        }
        if (this.authenticationToken == null || this.authenticationToken.isEmpty()) {
            LOG.warn("Authentication scheme is set to {} but no authentication token provided", (Object)this.authenticationScheme);
            return;
        }
        switch (this.authenticationScheme) {
            case BEARER: {
                httpGet.setHeader("Authorization", (Object)("Bearer " + this.authenticationToken));
                LOG.debug("Applied Bearer token authentication");
                break;
            }
            case API_KEY: {
                httpGet.setHeader(this.apiKeyHeader, (Object)this.authenticationToken);
                LOG.debug("Applied API Key authentication with header: {}", (Object)this.apiKeyHeader);
                break;
            }
            default: {
                LOG.warn("Unknown authentication scheme: {}", (Object)this.authenticationScheme);
            }
        }
    }

    public String convertDocumentAsync(String inputSource, String outputFormat) throws IOException {
        LOG.debug("Starting async document conversion using docling-serve API: {}", (Object)inputSource);
        String asyncEndpoint = this.convertEndpoint.replace(DEFAULT_CONVERT_ENDPOINT, DEFAULT_ASYNC_CONVERT_ENDPOINT);
        Map<String, Object> requestBody = this.buildRequestBody(inputSource, outputFormat);
        String jsonRequest = this.objectMapper.writeValueAsString(requestBody);
        LOG.debug("Async request body: {}", (Object)jsonRequest);
        HttpPost httpPost = new HttpPost(this.baseUrl + asyncEndpoint);
        httpPost.setEntity((HttpEntity)new StringEntity(jsonRequest, ContentType.APPLICATION_JSON));
        httpPost.setHeader("Accept", (Object)"application/json");
        this.applyAuthentication(httpPost);
        try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)httpPost);){
            String responseBody;
            int statusCode = response.getCode();
            try {
                responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            }
            catch (ParseException e) {
                throw new IOException("Failed to parse response from docling-serve API", e);
            }
            if (statusCode >= 200 && statusCode < 300) {
                JsonNode rootNode = this.objectMapper.readTree(responseBody);
                if (rootNode.has("task_id")) {
                    String string = rootNode.get("task_id").asText();
                    return string;
                }
                if (rootNode.has("id")) {
                    String string = rootNode.get("id").asText();
                    return string;
                }
                throw new IOException("No task ID found in async conversion response: " + responseBody);
            }
            throw new IOException("Docling-serve async API request failed with status " + statusCode + ": " + responseBody);
        }
    }

    public ConversionStatus checkConversionStatus(String taskId) throws IOException {
        LOG.debug("Checking status for task: {}", (Object)taskId);
        String statusEndpoint = "/v1/status/poll/" + taskId;
        HttpGet httpGet = new HttpGet(this.baseUrl + statusEndpoint);
        httpGet.setHeader("Accept", (Object)"application/json");
        this.applyAuthenticationGet(httpGet);
        try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)httpGet);){
            String responseBody;
            int statusCode = response.getCode();
            try {
                responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            }
            catch (ParseException e) {
                throw new IOException("Failed to parse response from docling-serve API", e);
            }
            if (statusCode >= 200 && statusCode < 300) {
                JsonNode rootNode = this.objectMapper.readTree(responseBody);
                ConversionStatus conversionStatus = this.parseConversionStatus(taskId, rootNode);
                return conversionStatus;
            }
            throw new IOException("Failed to check task status. Status code: " + statusCode + ", Response: " + responseBody);
        }
    }

    public String convertDocumentAsyncAndWait(String inputSource, String outputFormat) throws IOException {
        String taskId = this.convertDocumentAsync(inputSource, outputFormat);
        LOG.debug("Started async conversion with task ID: {}", (Object)taskId);
        long startTime = System.currentTimeMillis();
        long deadline = startTime + this.asyncTimeout;
        while (System.currentTimeMillis() < deadline) {
            ConversionStatus status = this.checkConversionStatus(taskId);
            LOG.debug("Task {} status: {}", (Object)taskId, (Object)status.getStatus());
            if (status.isCompleted()) {
                LOG.debug("Task {} completed successfully", (Object)taskId);
                return status.getResult();
            }
            if (status.isFailed()) {
                throw new IOException("Async conversion failed: " + status.getErrorMessage());
            }
            try {
                Thread.sleep(this.asyncPollInterval);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Async conversion interrupted", e);
            }
        }
        throw new IOException("Async conversion timed out after " + this.asyncTimeout + "ms for task: " + taskId);
    }

    private Map<String, Object> buildRequestBody(String inputSource, String outputFormat) {
        HashMap<String, Object> requestBody = new HashMap<String, Object>();
        if (inputSource.startsWith("http://") || inputSource.startsWith("https://")) {
            HashMap<String, String> source = new HashMap<String, String>();
            source.put("kind", "http");
            source.put("url", inputSource);
            requestBody.put("sources", Collections.singletonList(source));
        } else {
            try {
                File file = new File(inputSource);
                byte[] fileBytes = Files.readAllBytes(file.toPath());
                String base64Content = Base64.getEncoder().encodeToString(fileBytes);
                HashMap<String, String> source = new HashMap<String, String>();
                source.put("kind", "file");
                source.put("base64_string", base64Content);
                source.put("filename", file.getName());
                requestBody.put("sources", Collections.singletonList(source));
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to read file: " + inputSource, e);
            }
        }
        if (outputFormat != null && !outputFormat.isEmpty()) {
            HashMap<String, List<String>> options = new HashMap<String, List<String>>();
            options.put("to_formats", Collections.singletonList(this.mapOutputFormat(outputFormat)));
            requestBody.put("options", options);
        }
        return requestBody;
    }

    private ConversionStatus parseConversionStatus(String taskId, JsonNode statusNode) {
        String statusStr = statusNode.has("task_status") ? statusNode.get("task_status").asText() : "unknown";
        ConversionStatus.Status status = switch (statusStr.toLowerCase()) {
            case "pending" -> ConversionStatus.Status.PENDING;
            case "started", "in_progress", "running", "processing" -> ConversionStatus.Status.IN_PROGRESS;
            case "success", "completed" -> ConversionStatus.Status.COMPLETED;
            case "failure", "failed", "error" -> ConversionStatus.Status.FAILED;
            default -> {
                LOG.warn("Unknown task status: {}", (Object)statusStr);
                yield ConversionStatus.Status.UNKNOWN;
            }
        };
        String result = null;
        String errorMessage = null;
        if (status == ConversionStatus.Status.COMPLETED) {
            try {
                result = this.fetchTaskResult(taskId);
            }
            catch (IOException e) {
                LOG.warn("Failed to fetch result for completed task: {}", (Object)taskId, (Object)e);
            }
        }
        if (status == ConversionStatus.Status.FAILED) {
            errorMessage = statusNode.has("task_meta") && statusNode.get("task_meta").has("error") ? statusNode.get("task_meta").get("error").asText() : (statusNode.has("error") ? statusNode.get("error").asText() : "Task failed without error message");
        }
        Integer progress = statusNode.has("task_position") ? Integer.valueOf(statusNode.get("task_position").asInt()) : null;
        return new ConversionStatus(taskId, status, result, errorMessage, progress);
    }

    private String fetchTaskResult(String taskId) throws IOException {
        LOG.debug("Fetching result for task: {}", (Object)taskId);
        String resultEndpoint = "/v1/result/" + taskId;
        HttpGet httpGet = new HttpGet(this.baseUrl + resultEndpoint);
        httpGet.setHeader("Accept", (Object)"application/json");
        this.applyAuthenticationGet(httpGet);
        try (CloseableHttpResponse response = this.httpClient.execute((ClassicHttpRequest)httpGet);){
            String responseBody;
            int statusCode = response.getCode();
            try {
                responseBody = EntityUtils.toString((HttpEntity)response.getEntity());
            }
            catch (ParseException e) {
                throw new IOException("Failed to parse response from docling-serve API", e);
            }
            if (statusCode >= 200 && statusCode < 300) {
                String string = this.extractConvertedContent(responseBody, null);
                return string;
            }
            throw new IOException("Failed to fetch task result. Status code: " + statusCode + ", Response: " + responseBody);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BatchProcessingResults convertDocumentsBatch(List<String> inputSources, String outputFormat, int batchSize, int parallelism, boolean failOnFirstError, boolean useAsync, long batchTimeout) {
        LOG.info("Starting batch conversion of {} documents with parallelism={}, failOnFirstError={}, timeout={}ms", new Object[]{inputSources.size(), parallelism, failOnFirstError, batchTimeout});
        BatchProcessingResults results = new BatchProcessingResults();
        results.setStartTimeMs(System.currentTimeMillis());
        ExecutorService executor = Executors.newFixedThreadPool(parallelism);
        AtomicInteger index = new AtomicInteger(0);
        AtomicBoolean shouldCancel = new AtomicBoolean(false);
        try {
            ArrayList<CompletableFuture<BatchConversionResult>> futures = new ArrayList<CompletableFuture<BatchConversionResult>>();
            for (String inputSource : inputSources) {
                int n = index.getAndIncrement();
                String documentId = "doc-" + n;
                CompletableFuture<BatchConversionResult> future = CompletableFuture.supplyAsync(() -> {
                    BatchConversionResult result;
                    block3: {
                        if (failOnFirstError && shouldCancel.get()) {
                            BatchConversionResult cancelledResult = new BatchConversionResult(documentId, inputSource);
                            cancelledResult.setBatchIndex(currentIndex);
                            cancelledResult.setSuccess(false);
                            cancelledResult.setErrorMessage("Cancelled due to previous failure");
                            return cancelledResult;
                        }
                        result = new BatchConversionResult(documentId, inputSource);
                        result.setBatchIndex(currentIndex);
                        long startTime = System.currentTimeMillis();
                        try {
                            LOG.debug("Processing document {} (index {}): {}", new Object[]{documentId, currentIndex, inputSource});
                            String converted = useAsync ? this.convertDocumentAsyncAndWait(inputSource, outputFormat) : this.convertDocument(inputSource, outputFormat);
                            result.setResult(converted);
                            result.setSuccess(true);
                            result.setProcessingTimeMs(System.currentTimeMillis() - startTime);
                            LOG.debug("Successfully processed document {} in {}ms", (Object)documentId, (Object)result.getProcessingTimeMs());
                        }
                        catch (Exception e) {
                            result.setSuccess(false);
                            result.setErrorMessage(e.getMessage());
                            result.setProcessingTimeMs(System.currentTimeMillis() - startTime);
                            LOG.error("Failed to process document {} (index {}): {}", new Object[]{documentId, currentIndex, e.getMessage(), e});
                            if (!failOnFirstError) break block3;
                            shouldCancel.set(true);
                        }
                    }
                    return result;
                }, executor);
                futures.add(future);
            }
            CompletableFuture<Void> allOf = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
            try {
                allOf.get(batchTimeout, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                LOG.error("Batch processing timed out after {}ms", (Object)batchTimeout);
                futures.forEach(f -> f.cancel(true));
                throw new RuntimeException("Batch processing timed out after " + batchTimeout + "ms", e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOG.error("Batch processing interrupted", (Throwable)e);
                futures.forEach(f -> f.cancel(true));
                throw new RuntimeException("Batch processing interrupted", e);
            }
            catch (Exception e) {
                LOG.error("Batch processing failed", (Throwable)e);
                futures.forEach(f -> f.cancel(true));
                throw new RuntimeException("Batch processing failed", e);
            }
            for (CompletableFuture completableFuture : futures) {
                try {
                    BatchConversionResult result = completableFuture.getNow(null);
                    if (result == null) continue;
                    results.addResult(result);
                    if (!failOnFirstError || result.isSuccess()) continue;
                    LOG.warn("Failing batch due to error in document {}: {}", (Object)result.getDocumentId(), (Object)result.getErrorMessage());
                    break;
                }
                catch (Exception e) {
                    LOG.error("Error retrieving result", (Throwable)e);
                }
            }
        }
        finally {
            executor.shutdown();
            try {
                long shutdownTimeout = Math.max(10000L, batchTimeout / 10L);
                if (!executor.awaitTermination(shutdownTimeout, TimeUnit.MILLISECONDS)) {
                    LOG.warn("Executor did not terminate within {}ms, forcing shutdown", (Object)shutdownTimeout);
                    executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        results.setEndTimeMs(System.currentTimeMillis());
        LOG.info("Batch conversion completed: total={}, success={}, failed={}, time={}ms", new Object[]{results.getTotalDocuments(), results.getSuccessCount(), results.getFailureCount(), results.getTotalProcessingTimeMs()});
        if (failOnFirstError && results.hasAnyFailures()) {
            BatchConversionResult firstFailure = results.getFailed().get(0);
            throw new RuntimeException("Batch processing failed for document: " + firstFailure.getOriginalPath() + " - " + firstFailure.getErrorMessage());
        }
        return results;
    }

    public DocumentMetadata extractMetadata(String inputSource, boolean extractAll, boolean includeRawMetadata) throws IOException {
        LOG.debug("Extracting metadata using docling-serve API: {}", (Object)inputSource);
        String jsonOutput = this.convertDocument(inputSource, "json");
        DocumentMetadata metadata = new DocumentMetadata();
        metadata.setFilePath(inputSource);
        try {
            JsonNode mainTextNode;
            JsonNode docNode;
            File file;
            JsonNode rootNode = this.objectMapper.readTree(jsonOutput);
            if (!inputSource.startsWith("http://") && !inputSource.startsWith("https://") && (file = new File(inputSource)).exists()) {
                metadata.setFileName(file.getName());
                metadata.setFileSizeBytes(file.length());
            }
            if (rootNode.has("metadata")) {
                JsonNode metadataNode = rootNode.get("metadata");
                this.extractMetadataFields(metadata, metadataNode, extractAll);
            }
            if (rootNode.has("document") && (docNode = rootNode.get("document")).has("name") && metadata.getTitle() == null) {
                metadata.setTitle(docNode.get("name").asText());
            }
            if (rootNode.has("main-text") && (mainTextNode = rootNode.get("main-text")).isArray() && mainTextNode.size() > 0) {
                metadata.setDocumentType("Text Document");
            }
            if (rootNode.has("pages")) {
                if (rootNode.get("pages").isArray()) {
                    metadata.setPageCount(rootNode.get("pages").size());
                } else if (rootNode.get("pages").isInt()) {
                    metadata.setPageCount(rootNode.get("pages").asInt());
                }
            } else if (rootNode.has("num_pages")) {
                metadata.setPageCount(rootNode.get("num_pages").asInt());
            } else if (rootNode.has("page_count")) {
                metadata.setPageCount(rootNode.get("page_count").asInt());
            }
            if (includeRawMetadata) {
                Map rawMap = (Map)this.objectMapper.convertValue((Object)rootNode, Map.class);
                metadata.setRawMetadata(rawMap);
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to parse metadata from docling-serve response: {}", (Object)e.getMessage(), (Object)e);
            throw new IOException("Failed to extract metadata", e);
        }
        return metadata;
    }

    private void extractMetadataFields(DocumentMetadata metadata, JsonNode metadataNode, boolean extractAll) {
        if (metadataNode.has("title")) {
            metadata.setTitle(metadataNode.get("title").asText());
        }
        if (metadataNode.has("author") || metadataNode.has("Author")) {
            String author = metadataNode.has("author") ? metadataNode.get("author").asText() : metadataNode.get("Author").asText();
            metadata.setAuthor(author);
        }
        if (metadataNode.has("creator") || metadataNode.has("Creator")) {
            String creator = metadataNode.has("creator") ? metadataNode.get("creator").asText() : metadataNode.get("Creator").asText();
            metadata.setCreator(creator);
        }
        if (metadataNode.has("producer") || metadataNode.has("Producer")) {
            String producer = metadataNode.has("producer") ? metadataNode.get("producer").asText() : metadataNode.get("Producer").asText();
            metadata.setProducer(producer);
        }
        if (metadataNode.has("subject") || metadataNode.has("Subject")) {
            String subject = metadataNode.has("subject") ? metadataNode.get("subject").asText() : metadataNode.get("Subject").asText();
            metadata.setSubject(subject);
        }
        if (metadataNode.has("keywords") || metadataNode.has("Keywords")) {
            String keywords = metadataNode.has("keywords") ? metadataNode.get("keywords").asText() : metadataNode.get("Keywords").asText();
            metadata.setKeywords(keywords);
        }
        if (metadataNode.has("language") || metadataNode.has("Language")) {
            String language = metadataNode.has("language") ? metadataNode.get("language").asText() : metadataNode.get("Language").asText();
            metadata.setLanguage(language);
        }
        if (metadataNode.has("format") || metadataNode.has("Format")) {
            String format = metadataNode.has("format") ? metadataNode.get("format").asText() : metadataNode.get("Format").asText();
            metadata.setFormat(format);
        }
        this.extractDateField(metadata, metadataNode, "creation_date", "CreationDate", "created", "Created", date -> metadata.setCreationDate((Instant)date));
        this.extractDateField(metadata, metadataNode, "modification_date", "ModificationDate", "modified", "Modified", "ModDate", date -> metadata.setModificationDate((Instant)date));
        if (extractAll) {
            metadataNode.fields().forEachRemaining(entry -> {
                String key = (String)entry.getKey();
                if (!DoclingMetadataFields.isStandardField(key)) {
                    JsonNode value = (JsonNode)entry.getValue();
                    if (value.isTextual()) {
                        metadata.addCustomMetadata(key, value.asText());
                    } else if (value.isInt()) {
                        metadata.addCustomMetadata(key, value.asInt());
                    } else if (value.isLong()) {
                        metadata.addCustomMetadata(key, value.asLong());
                    } else if (value.isBoolean()) {
                        metadata.addCustomMetadata(key, value.asBoolean());
                    } else if (value.isDouble()) {
                        metadata.addCustomMetadata(key, value.asDouble());
                    } else {
                        metadata.addCustomMetadata(key, value.toString());
                    }
                }
            });
        }
    }

    private void extractDateField(DocumentMetadata metadata, JsonNode metadataNode, String fieldName1, String fieldName2, String fieldName3, String fieldName4, String fieldName5, Consumer<Instant> setter) {
        String dateStr = null;
        if (metadataNode.has(fieldName1)) {
            dateStr = metadataNode.get(fieldName1).asText();
        } else if (metadataNode.has(fieldName2)) {
            dateStr = metadataNode.get(fieldName2).asText();
        } else if (metadataNode.has(fieldName3)) {
            dateStr = metadataNode.get(fieldName3).asText();
        } else if (metadataNode.has(fieldName4)) {
            dateStr = metadataNode.get(fieldName4).asText();
        } else if (fieldName5 != null && metadataNode.has(fieldName5)) {
            dateStr = metadataNode.get(fieldName5).asText();
        }
        if (dateStr != null && !dateStr.isEmpty()) {
            try {
                Instant instant = Instant.parse(dateStr);
                setter.accept(instant);
            }
            catch (Exception e) {
                LOG.debug("Failed to parse date field {}: {}", (Object)fieldName1, (Object)dateStr);
                try {
                    LocalDateTime ldt = LocalDateTime.parse(dateStr);
                    Instant instant = ldt.atZone(ZoneId.systemDefault()).toInstant();
                    setter.accept(instant);
                }
                catch (Exception e2) {
                    LOG.debug("Failed to parse date as LocalDateTime: {}", (Object)dateStr);
                }
            }
        }
    }

    private void extractDateField(DocumentMetadata metadata, JsonNode metadataNode, String fieldName1, String fieldName2, String fieldName3, String fieldName4, Consumer<Instant> setter) {
        this.extractDateField(metadata, metadataNode, fieldName1, fieldName2, fieldName3, fieldName4, null, setter);
    }

    public void close() throws IOException {
        if (this.httpClient != null) {
            try {
                this.httpClient.close();
                LOG.debug("HTTP client closed successfully");
            }
            catch (IOException e) {
                LOG.warn("Error closing HTTP client", (Throwable)e);
                throw e;
            }
        }
        if (this.connectionManager != null) {
            this.connectionManager.close();
            LOG.debug("Connection manager closed successfully");
        }
    }

    public String getPoolStats() {
        if (this.connectionManager != null) {
            PoolStats stats = this.connectionManager.getTotalStats();
            return String.format("ConnectionPool[available=%d, leased=%d, pending=%d, max=%d]", stats.getAvailable(), stats.getLeased(), stats.getPending(), stats.getMax());
        }
        return "ConnectionPool[not initialized]";
    }
}

