/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.docling;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.InvalidPayloadException;
import org.apache.camel.component.docling.BatchProcessingResults;
import org.apache.camel.component.docling.ConversionStatus;
import org.apache.camel.component.docling.DoclingConfiguration;
import org.apache.camel.component.docling.DoclingEndpoint;
import org.apache.camel.component.docling.DoclingMetadataFields;
import org.apache.camel.component.docling.DoclingOperations;
import org.apache.camel.component.docling.DoclingServeClient;
import org.apache.camel.component.docling.DocumentMetadata;
import org.apache.camel.support.DefaultProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DoclingProducer
extends DefaultProducer {
    private static final Logger LOG = LoggerFactory.getLogger(DoclingProducer.class);
    private DoclingEndpoint endpoint;
    private DoclingConfiguration configuration;
    private DoclingServeClient doclingServeClient;

    public DoclingProducer(DoclingEndpoint endpoint) {
        super((Endpoint)endpoint);
        this.endpoint = endpoint;
        this.configuration = endpoint.getConfiguration();
    }

    protected void doStart() throws Exception {
        super.doStart();
        if (this.configuration.isUseDoclingServe()) {
            this.doclingServeClient = new DoclingServeClient(this.configuration.getDoclingServeUrl(), this.configuration.getAuthenticationScheme(), this.configuration.getAuthenticationToken(), this.configuration.getApiKeyHeader(), this.configuration.getConvertEndpoint(), this.configuration.getAsyncPollInterval(), this.configuration.getAsyncTimeout(), this.configuration.getMaxTotalConnections(), this.configuration.getMaxConnectionsPerRoute(), this.configuration.getConnectionTimeout(), this.configuration.getSocketTimeout(), this.configuration.getConnectionRequestTimeout(), this.configuration.getConnectionTimeToLive(), this.configuration.getValidateAfterInactivity(), this.configuration.isEvictIdleConnections(), this.configuration.getMaxIdleTime());
            LOG.info("DoclingProducer configured to use docling-serve API at: {}{} with authentication: {} (async mode: {})", new Object[]{this.configuration.getDoclingServeUrl(), this.configuration.getConvertEndpoint(), this.configuration.getAuthenticationScheme(), this.configuration.isUseAsyncMode()});
            LOG.debug("Connection pool stats: {}", (Object)this.doclingServeClient.getPoolStats());
        } else {
            LOG.info("DoclingProducer configured to use docling CLI command");
        }
    }

    protected void doStop() throws Exception {
        super.doStop();
        if (this.doclingServeClient != null) {
            LOG.debug("Shutting down DoclingServeClient. Final pool stats: {}", (Object)this.doclingServeClient.getPoolStats());
            this.doclingServeClient.close();
            this.doclingServeClient = null;
            LOG.info("DoclingServeClient closed successfully");
        }
    }

    public void process(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer processing exchange with message ID: {}", (Object)exchange.getExchangeId());
        DoclingOperations operation = this.getOperation(exchange);
        LOG.debug("DoclingProducer performing operation: {}", (Object)operation);
        switch (operation) {
            case CONVERT_TO_MARKDOWN: {
                this.processConvertToMarkdown(exchange);
                break;
            }
            case CONVERT_TO_HTML: {
                this.processConvertToHTML(exchange);
                break;
            }
            case CONVERT_TO_JSON: {
                this.processConvertToJSON(exchange);
                break;
            }
            case EXTRACT_TEXT: {
                this.processExtractText(exchange);
                break;
            }
            case EXTRACT_STRUCTURED_DATA: {
                this.processExtractStructuredData(exchange);
                break;
            }
            case SUBMIT_ASYNC_CONVERSION: {
                this.processSubmitAsyncConversion(exchange);
                break;
            }
            case CHECK_CONVERSION_STATUS: {
                this.processCheckConversionStatus(exchange);
                break;
            }
            case BATCH_CONVERT_TO_MARKDOWN: {
                this.processBatchConversion(exchange, "markdown");
                break;
            }
            case BATCH_CONVERT_TO_HTML: {
                this.processBatchConversion(exchange, "html");
                break;
            }
            case BATCH_CONVERT_TO_JSON: {
                this.processBatchConversion(exchange, "json");
                break;
            }
            case BATCH_EXTRACT_TEXT: {
                this.processBatchConversion(exchange, "text");
                break;
            }
            case BATCH_EXTRACT_STRUCTURED_DATA: {
                this.processBatchConversion(exchange, "json");
                break;
            }
            case EXTRACT_METADATA: {
                this.processExtractMetadata(exchange);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported operation: " + String.valueOf((Object)operation));
            }
        }
    }

    private DoclingOperations getOperation(Exchange exchange) {
        DoclingOperations operation = (DoclingOperations)((Object)exchange.getIn().getHeader("CamelDoclingOperation", DoclingOperations.class));
        if (operation == null) {
            operation = this.configuration.getOperation();
        }
        return operation;
    }

    private void processConvertToMarkdown(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer converting to markdown");
        if (this.configuration.isUseDoclingServe()) {
            String inputPath = this.getInputPath(exchange);
            String result = this.convertUsingDoclingServe(inputPath, "markdown", exchange);
            exchange.getIn().setBody((Object)result);
        } else {
            String inputPath = this.getInputPath(exchange);
            exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "markdown", exchange));
        }
    }

    private void processConvertToHTML(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer converting to HTML");
        if (this.configuration.isUseDoclingServe()) {
            String inputPath = this.getInputPath(exchange);
            String result = this.convertUsingDoclingServe(inputPath, "html", exchange);
            exchange.getIn().setBody((Object)result);
        } else {
            String inputPath = this.getInputPath(exchange);
            exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "html", exchange));
        }
    }

    private void processConvertToJSON(Exchange exchange) throws Exception {
        if (this.configuration.isUseDoclingServe()) {
            String inputPath = this.getInputPath(exchange);
            String result = this.convertUsingDoclingServe(inputPath, "json", exchange);
            exchange.getIn().setBody((Object)result);
        } else {
            String inputPath = this.getInputPath(exchange);
            exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "json", exchange));
        }
    }

    private void processExtractText(Exchange exchange) throws Exception {
        if (this.configuration.isUseDoclingServe()) {
            String inputPath = this.getInputPath(exchange);
            String result = this.convertUsingDoclingServe(inputPath, "text", exchange);
            exchange.getIn().setBody((Object)result);
        } else {
            String inputPath = this.getInputPath(exchange);
            exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "text", exchange));
        }
    }

    private void processExtractStructuredData(Exchange exchange) throws Exception {
        if (this.configuration.isUseDoclingServe()) {
            String inputPath = this.getInputPath(exchange);
            String result = this.convertUsingDoclingServe(inputPath, "json", exchange);
            exchange.getIn().setBody((Object)result);
        } else {
            String inputPath = this.getInputPath(exchange);
            exchange.getIn().setBody((Object)this.executeDoclingCommand(inputPath, "json", exchange));
        }
    }

    private void processSubmitAsyncConversion(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer submitting async conversion");
        if (!this.configuration.isUseDoclingServe()) {
            throw new IllegalStateException("SUBMIT_ASYNC_CONVERSION operation requires docling-serve mode (useDoclingServe=true)");
        }
        String inputPath = this.getInputPath(exchange);
        String outputFormat = (String)exchange.getIn().getHeader("CamelDoclingOutputFormat", String.class);
        if (outputFormat == null) {
            outputFormat = this.configuration.getOutputFormat();
        }
        String taskId = this.doclingServeClient.convertDocumentAsync(inputPath, outputFormat);
        LOG.debug("Async conversion submitted with task ID: {}", (Object)taskId);
        exchange.getIn().setBody((Object)taskId);
        exchange.getIn().setHeader("CamelDoclingTaskId", (Object)taskId);
    }

    private void processCheckConversionStatus(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer checking conversion status");
        if (!this.configuration.isUseDoclingServe()) {
            throw new IllegalStateException("CHECK_CONVERSION_STATUS operation requires docling-serve mode (useDoclingServe=true)");
        }
        String taskId = (String)exchange.getIn().getHeader("CamelDoclingTaskId", String.class);
        if (taskId == null) {
            Object body = exchange.getIn().getBody();
            if (body instanceof String) {
                taskId = (String)body;
            } else {
                throw new IllegalArgumentException("Task ID must be provided in header CamelDoclingTaskId or in message body");
            }
        }
        LOG.debug("Checking status for task ID: {}", (Object)taskId);
        ConversionStatus status = this.doclingServeClient.checkConversionStatus(taskId);
        exchange.getIn().setBody((Object)status);
        exchange.getIn().setHeader("CamelDoclingTaskId", (Object)status.getTaskId());
        exchange.getIn().setHeader("CamelDoclingTaskStatus", (Object)status.getStatus().toString());
        if (status.getProgress() != null) {
            exchange.getIn().setHeader("CamelDoclingTaskProgress", (Object)status.getProgress());
        }
        if (status.isCompleted() && status.getResult() != null) {
            exchange.getIn().setHeader("CamelDoclingResult", (Object)status.getResult());
        }
        if (status.isFailed() && status.getErrorMessage() != null) {
            exchange.getIn().setHeader("CamelDoclingErrorMessage", (Object)status.getErrorMessage());
        }
    }

    private void processExtractMetadata(Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer extracting metadata");
        String inputPath = this.getInputPath(exchange);
        DocumentMetadata metadata = this.configuration.isUseDoclingServe() ? this.doclingServeClient.extractMetadata(inputPath, this.configuration.isExtractAllMetadata(), this.configuration.isIncludeRawMetadata()) : this.extractMetadataUsingCLI(inputPath, exchange);
        exchange.getIn().setBody((Object)metadata);
        if (this.configuration.isIncludeMetadataInHeaders()) {
            this.setMetadataHeaders(exchange, metadata);
        }
        LOG.debug("Metadata extraction completed for: {}", (Object)inputPath);
    }

    private DocumentMetadata extractMetadataUsingCLI(String inputPath, Exchange exchange) throws Exception {
        LOG.debug("Extracting metadata using Docling CLI for: {}", (Object)inputPath);
        String jsonOutput = this.executeDoclingCommand(inputPath, "json", exchange);
        DocumentMetadata metadata = this.parseMetadataFromJson(jsonOutput, inputPath);
        return metadata;
    }

    private DocumentMetadata parseMetadataFromJson(String jsonOutput, String inputPath) {
        DocumentMetadata metadata = new DocumentMetadata();
        metadata.setFilePath(inputPath);
        try {
            JsonNode docNode;
            ObjectMapper mapper = new ObjectMapper();
            JsonNode rootNode = mapper.readTree(jsonOutput);
            File file = new File(inputPath);
            if (file.exists()) {
                metadata.setFileName(file.getName());
                metadata.setFileSizeBytes(file.length());
            }
            if (rootNode.has("metadata")) {
                JsonNode metadataNode = rootNode.get("metadata");
                this.extractFieldsFromJsonNode(metadata, metadataNode);
            }
            if (rootNode.has("document") && (docNode = rootNode.get("document")).has("name")) {
                metadata.setTitle(docNode.get("name").asText());
            }
            if (rootNode.has("pages")) {
                metadata.setPageCount(rootNode.get("pages").size());
            } else if (rootNode.has("num_pages")) {
                metadata.setPageCount(rootNode.get("num_pages").asInt());
            }
            if (this.configuration.isIncludeRawMetadata()) {
                Map rawMap = (Map)mapper.convertValue((Object)rootNode, Map.class);
                metadata.setRawMetadata(rawMap);
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to parse metadata from JSON output: {}", (Object)e.getMessage(), (Object)e);
        }
        return metadata;
    }

    private void extractFieldsFromJsonNode(DocumentMetadata metadata, JsonNode metadataNode) {
        String dateStr;
        if (metadataNode.has("title")) {
            metadata.setTitle(metadataNode.get("title").asText());
        }
        if (metadataNode.has("author")) {
            metadata.setAuthor(metadataNode.get("author").asText());
        }
        if (metadataNode.has("creator")) {
            metadata.setCreator(metadataNode.get("creator").asText());
        }
        if (metadataNode.has("producer")) {
            metadata.setProducer(metadataNode.get("producer").asText());
        }
        if (metadataNode.has("subject")) {
            metadata.setSubject(metadataNode.get("subject").asText());
        }
        if (metadataNode.has("keywords")) {
            metadata.setKeywords(metadataNode.get("keywords").asText());
        }
        if (metadataNode.has("language")) {
            metadata.setLanguage(metadataNode.get("language").asText());
        }
        if (metadataNode.has("creation_date") || metadataNode.has("creationDate")) {
            dateStr = metadataNode.has("creation_date") ? metadataNode.get("creation_date").asText() : metadataNode.get("creationDate").asText();
            try {
                metadata.setCreationDate(Instant.parse(dateStr));
            }
            catch (Exception e) {
                LOG.debug("Failed to parse creation date: {}", (Object)dateStr);
            }
        }
        if (metadataNode.has("modification_date") || metadataNode.has("modificationDate")) {
            dateStr = metadataNode.has("modification_date") ? metadataNode.get("modification_date").asText() : metadataNode.get("modificationDate").asText();
            try {
                metadata.setModificationDate(Instant.parse(dateStr));
            }
            catch (Exception e) {
                LOG.debug("Failed to parse modification date: {}", (Object)dateStr);
            }
        }
        if (this.configuration.isExtractAllMetadata()) {
            metadataNode.fields().forEachRemaining(entry -> {
                String key = (String)entry.getKey();
                if (!DoclingMetadataFields.isStandardField(key)) {
                    JsonNode value = (JsonNode)entry.getValue();
                    if (value.isTextual()) {
                        metadata.addCustomMetadata(key, value.asText());
                    } else if (value.isNumber()) {
                        metadata.addCustomMetadata(key, value.asLong());
                    } else if (value.isBoolean()) {
                        metadata.addCustomMetadata(key, value.asBoolean());
                    } else {
                        metadata.addCustomMetadata(key, value.toString());
                    }
                }
            });
        }
    }

    private void setMetadataHeaders(Exchange exchange, DocumentMetadata metadata) {
        if (metadata.getTitle() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataTitle", (Object)metadata.getTitle());
        }
        if (metadata.getAuthor() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataAuthor", (Object)metadata.getAuthor());
        }
        if (metadata.getCreator() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataCreator", (Object)metadata.getCreator());
        }
        if (metadata.getProducer() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataProducer", (Object)metadata.getProducer());
        }
        if (metadata.getSubject() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataSubject", (Object)metadata.getSubject());
        }
        if (metadata.getKeywords() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataKeywords", (Object)metadata.getKeywords());
        }
        if (metadata.getCreationDate() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataCreationDate", (Object)metadata.getCreationDate());
        }
        if (metadata.getModificationDate() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataModificationDate", (Object)metadata.getModificationDate());
        }
        if (metadata.getPageCount() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataPageCount", (Object)metadata.getPageCount());
        }
        if (metadata.getLanguage() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataLanguage", (Object)metadata.getLanguage());
        }
        if (metadata.getDocumentType() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataDocumentType", (Object)metadata.getDocumentType());
        }
        if (metadata.getFormat() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataFormat", (Object)metadata.getFormat());
        }
        if (metadata.getFileSizeBytes() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataFileSize", (Object)metadata.getFileSizeBytes());
        }
        if (metadata.getFileName() != null) {
            exchange.getIn().setHeader("CamelDoclingMetadataFileName", (Object)metadata.getFileName());
        }
        if (metadata.getCustomMetadata() != null && !metadata.getCustomMetadata().isEmpty()) {
            exchange.getIn().setHeader("CamelDoclingMetadataCustom", metadata.getCustomMetadata());
        }
        if (this.configuration.isIncludeRawMetadata() && metadata.getRawMetadata() != null && !metadata.getRawMetadata().isEmpty()) {
            exchange.getIn().setHeader("CamelDoclingMetadataRaw", metadata.getRawMetadata());
        }
    }

    private void processBatchConversion(Exchange exchange, String outputFormat) throws Exception {
        LOG.debug("DoclingProducer processing batch conversion with format: {}", (Object)outputFormat);
        if (!this.configuration.isUseDoclingServe()) {
            throw new IllegalStateException("Batch operations require docling-serve mode (useDoclingServe=true)");
        }
        List<String> documentPaths = this.extractDocumentList(exchange);
        if (documentPaths.isEmpty()) {
            throw new IllegalArgumentException("No documents provided for batch processing");
        }
        LOG.debug("Processing batch of {} documents", (Object)documentPaths.size());
        int batchSize = (Integer)exchange.getIn().getHeader("CamelDoclingBatchSize", (Object)this.configuration.getBatchSize(), Integer.class);
        int parallelism = (Integer)exchange.getIn().getHeader("CamelDoclingBatchParallelism", (Object)this.configuration.getBatchParallelism(), Integer.class);
        boolean failOnFirstError = (Boolean)exchange.getIn().getHeader("CamelDoclingBatchFailOnFirstError", (Object)this.configuration.isBatchFailOnFirstError(), Boolean.class);
        long batchTimeout = (Long)exchange.getIn().getHeader("CamelDoclingBatchTimeout", (Object)this.configuration.getBatchTimeout(), Long.class);
        boolean useAsync = this.configuration.isUseAsyncMode();
        Boolean asyncModeHeader = (Boolean)exchange.getIn().getHeader("CamelDoclingUseAsyncMode", Boolean.class);
        if (asyncModeHeader != null) {
            useAsync = asyncModeHeader;
        }
        BatchProcessingResults results = this.doclingServeClient.convertDocumentsBatch(documentPaths, outputFormat, batchSize, parallelism, failOnFirstError, useAsync, batchTimeout);
        boolean splitResults = this.configuration.isSplitBatchResults();
        Boolean splitResultsHeader = (Boolean)exchange.getIn().getHeader("CamelDoclingBatchSplitResults", Boolean.class);
        if (splitResultsHeader != null) {
            splitResults = splitResultsHeader;
        }
        exchange.getIn().setHeader("CamelDoclingBatchTotalDocuments", (Object)results.getTotalDocuments());
        exchange.getIn().setHeader("CamelDoclingBatchSuccessCount", (Object)results.getSuccessCount());
        exchange.getIn().setHeader("CamelDoclingBatchFailureCount", (Object)results.getFailureCount());
        exchange.getIn().setHeader("CamelDoclingBatchProcessingTime", (Object)results.getTotalProcessingTimeMs());
        if (splitResults) {
            exchange.getIn().setBody(results.getResults());
            LOG.info("Batch conversion completed: {} documents, {} succeeded, {} failed - returning individual results for splitting", new Object[]{results.getTotalDocuments(), results.getSuccessCount(), results.getFailureCount()});
        } else {
            exchange.getIn().setBody((Object)results);
            LOG.info("Batch conversion completed: {} documents, {} succeeded, {} failed", new Object[]{results.getTotalDocuments(), results.getSuccessCount(), results.getFailureCount()});
        }
    }

    private List<String> extractDocumentList(Exchange exchange) {
        Object body = exchange.getIn().getBody();
        if (body instanceof List) {
            List list = (List)body;
            if (list.isEmpty()) {
                return new ArrayList<String>();
            }
            if (list.get(0) instanceof String) {
                return list;
            }
            if (list.get(0) instanceof File) {
                return list.stream().map(File::getAbsolutePath).collect(Collectors.toList());
            }
            return list.stream().map(Object::toString).collect(Collectors.toList());
        }
        if (body instanceof Collection) {
            Collection collection = (Collection)body;
            return collection.stream().map(obj -> {
                if (obj instanceof String) {
                    return (String)obj;
                }
                if (obj instanceof File) {
                    return ((File)obj).getAbsolutePath();
                }
                return obj.toString();
            }).collect(Collectors.toList());
        }
        if (body instanceof String[]) {
            return List.of((String[])body);
        }
        if (body instanceof File[]) {
            File[] files = (File[])body;
            ArrayList<String> paths = new ArrayList<String>();
            for (File file : files) {
                paths.add(file.getAbsolutePath());
            }
            return paths;
        }
        if (body instanceof String) {
            String path = (String)body;
            File dir = new File(path);
            if (dir.isDirectory()) {
                File[] files = dir.listFiles();
                if (files != null) {
                    ArrayList<String> paths = new ArrayList<String>();
                    for (File file : files) {
                        if (!file.isFile()) continue;
                        paths.add(file.getAbsolutePath());
                    }
                    return paths;
                }
            } else {
                return List.of(path);
            }
        }
        throw new IllegalArgumentException("Unsupported body type for batch processing: " + (body != null ? body.getClass().getName() : "null") + ". Expected List<String>, List<File>, String[], File[], or directory path");
    }

    private String convertUsingDoclingServe(String inputPath, String outputFormat) throws Exception {
        return this.convertUsingDoclingServe(inputPath, outputFormat, null);
    }

    private String convertUsingDoclingServe(String inputPath, String outputFormat, Exchange exchange) throws Exception {
        Boolean asyncModeHeader;
        boolean useAsync = this.configuration.isUseAsyncMode();
        if (exchange != null && (asyncModeHeader = (Boolean)exchange.getIn().getHeader("CamelDoclingUseAsyncMode", Boolean.class)) != null) {
            useAsync = asyncModeHeader;
        }
        if (useAsync) {
            LOG.debug("Using async mode for conversion");
            return this.doclingServeClient.convertDocumentAsyncAndWait(inputPath, outputFormat);
        }
        LOG.debug("Using sync mode for conversion");
        return this.doclingServeClient.convertDocument(inputPath, outputFormat);
    }

    private String getInputPath(Exchange exchange) throws InvalidPayloadException, IOException {
        String inputPath = (String)exchange.getIn().getHeader("CamelDoclingInputFilePath", String.class);
        if (inputPath != null) {
            this.validateFileSize(inputPath);
            return inputPath;
        }
        Object body = exchange.getIn().getBody();
        if (body instanceof String) {
            String content = (String)body;
            if (content.startsWith("http://") || content.startsWith("https://")) {
                return content;
            }
            if (content.startsWith("/") || content.contains("\\")) {
                this.validateFileSize(content);
                return content;
            }
            Path tempFile = Files.createTempFile("docling-", ".tmp", new FileAttribute[0]);
            Files.write(tempFile, content.getBytes(), new OpenOption[0]);
            this.validateFileSize(tempFile.toString());
            return tempFile.toString();
        }
        if (body instanceof byte[]) {
            byte[] content = (byte[])body;
            if ((long)content.length > this.configuration.getMaxFileSize()) {
                throw new IllegalArgumentException("File size exceeds maximum allowed size: " + this.configuration.getMaxFileSize());
            }
            Path tempFile = Files.createTempFile("docling-", ".tmp", new FileAttribute[0]);
            Files.write(tempFile, content, new OpenOption[0]);
            return tempFile.toString();
        }
        if (body instanceof File) {
            File file = (File)body;
            this.validateFileSize(file.getAbsolutePath());
            return file.getAbsolutePath();
        }
        throw new InvalidPayloadException(exchange, String.class);
    }

    private void validateFileSize(String filePath) throws IOException {
        long fileSize;
        Path path = Paths.get(filePath, new String[0]);
        if (Files.exists(path, new LinkOption[0]) && (fileSize = Files.size(path)) > this.configuration.getMaxFileSize()) {
            throw new IllegalArgumentException("File size (" + fileSize + " bytes) exceeds maximum allowed size: " + this.configuration.getMaxFileSize());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String executeDoclingCommand(String inputPath, String outputFormat, Exchange exchange) throws Exception {
        LOG.debug("DoclingProducer executing Docling command for input: {} with format: {}", (Object)inputPath, (Object)outputFormat);
        Path tempOutputDir = Files.createTempDirectory("docling-output", new FileAttribute[0]);
        try {
            List<String> command = this.buildDoclingCommand(inputPath, outputFormat, exchange, tempOutputDir.toString());
            LOG.debug("Executing Docling command: {}", command);
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            if (this.configuration.getWorkingDirectory() != null) {
                processBuilder.directory(new File(this.configuration.getWorkingDirectory()));
            }
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            StringBuilder error = new StringBuilder();
            try (BufferedReader outputReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                 BufferedReader errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));){
                String line;
                while ((line = outputReader.readLine()) != null) {
                    LOG.debug("Docling output: {}", (Object)line);
                    output.append(line).append("\n");
                }
                while ((line = errorReader.readLine()) != null) {
                    error.append(line).append("\n");
                }
            }
            boolean finished = process.waitFor(this.configuration.getProcessTimeout(), TimeUnit.MILLISECONDS);
            if (!finished) {
                process.destroyForcibly();
                throw new RuntimeException("Docling process timed out after " + this.configuration.getProcessTimeout() + " milliseconds");
            }
            int exitCode = process.exitValue();
            if (exitCode != 0) {
                throw new RuntimeException("Docling process failed with exit code " + exitCode + ". Error: " + error.toString());
            }
            String result = this.readGeneratedOutputFile(tempOutputDir, inputPath, outputFormat);
            if (!this.configuration.isContentInBody()) {
                result = this.moveOutputFileToFinalLocation(tempOutputDir, inputPath, outputFormat);
            }
            String string = result;
            return string;
        }
        finally {
            if (this.configuration.isContentInBody()) {
                this.deleteDirectory(tempOutputDir);
            }
        }
    }

    private String readGeneratedOutputFile(Path outputDir, String inputPath, String outputFormat) throws IOException {
        Path inputFilePath = Paths.get(inputPath, new String[0]);
        String baseName = inputFilePath.getFileName().toString();
        int lastDot = baseName.lastIndexOf(46);
        if (lastDot > 0) {
            baseName = baseName.substring(0, lastDot);
        }
        String extension = this.getOutputFileExtension(outputFormat);
        String expectedFileName = baseName + "." + extension;
        Path outputFile = outputDir.resolve(expectedFileName);
        Path actualOutputFile = null;
        if (Files.exists(outputFile, new LinkOption[0])) {
            actualOutputFile = outputFile;
        } else {
            try (Stream<Path> stream = Files.list(outputDir);){
                actualOutputFile = stream.findFirst().orElse(null);
                if (actualOutputFile == null || !Files.isRegularFile(actualOutputFile, new LinkOption[0])) {
                    throw new RuntimeException("No output file generated in: " + String.valueOf(outputDir));
                }
            }
        }
        if (this.configuration.isContentInBody()) {
            String content = Files.readString(actualOutputFile);
            try {
                Files.delete(actualOutputFile);
                LOG.debug("Deleted output file: {}", (Object)actualOutputFile);
            }
            catch (IOException e) {
                LOG.warn("Failed to delete output file: {}", (Object)actualOutputFile, (Object)e);
            }
            return content;
        }
        return actualOutputFile.toString();
    }

    private String moveOutputFileToFinalLocation(Path tempOutputDir, String inputPath, String outputFormat) throws IOException {
        String extension;
        String expectedFileName;
        Path tempOutputFile;
        Path inputFilePath = Paths.get(inputPath, new String[0]);
        String baseName = inputFilePath.getFileName().toString();
        int lastDot = baseName.lastIndexOf(46);
        if (lastDot > 0) {
            baseName = baseName.substring(0, lastDot);
        }
        if (!Files.exists(tempOutputFile = tempOutputDir.resolve(expectedFileName = baseName + "." + (extension = this.getOutputFileExtension(outputFormat))), new LinkOption[0])) {
            try (Stream<Path> stream = Files.list(tempOutputDir);){
                tempOutputFile = stream.findFirst().orElse(null);
                if (tempOutputFile == null || !Files.isRegularFile(tempOutputFile, new LinkOption[0])) {
                    throw new RuntimeException("No output file generated in: " + String.valueOf(tempOutputDir));
                }
            }
        }
        Path finalOutputFile = inputFilePath.getParent().resolve(tempOutputFile.getFileName());
        int counter = 1;
        while (Files.exists(finalOutputFile, new LinkOption[0])) {
            String nameWithoutExt = baseName;
            String ext = extension;
            finalOutputFile = inputFilePath.getParent().resolve(nameWithoutExt + "_" + counter + "." + ext);
            ++counter;
        }
        Files.move(tempOutputFile, finalOutputFile, new CopyOption[0]);
        LOG.debug("Moved output file from {} to {}", (Object)tempOutputFile, (Object)finalOutputFile);
        return finalOutputFile.toString();
    }

    private String getOutputFileExtension(String outputFormat) {
        switch (outputFormat.toLowerCase()) {
            case "markdown": 
            case "md": {
                return "md";
            }
            case "html": {
                return "html";
            }
            case "json": {
                return "json";
            }
            case "text": {
                return "txt";
            }
        }
        return "md";
    }

    private void deleteDirectory(Path directory) {
        try {
            if (Files.exists(directory, new LinkOption[0])) {
                Files.walk(directory, new FileVisitOption[0]).sorted((a, b) -> b.compareTo((Path)a)).forEach(path -> {
                    try {
                        Files.delete(path);
                    }
                    catch (IOException e) {
                        LOG.warn("Failed to delete temporary file: {}", path, (Object)e);
                    }
                });
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to clean up temporary directory: {}", (Object)directory, (Object)e);
        }
    }

    private List<String> buildDoclingCommand(String inputPath, String outputFormat, Exchange exchange, String outputDirectory) {
        ArrayList<String> command = new ArrayList<String>();
        command.add(this.configuration.getDoclingCommand());
        this.addCustomArguments(command, exchange);
        this.addOutputFormatArguments(command, outputFormat);
        this.addOcrArguments(command);
        this.addLayoutArguments(command);
        this.addOutputDirectoryArguments(command, exchange, outputDirectory);
        command.add(inputPath);
        return command;
    }

    private void addCustomArguments(List<String> command, Exchange exchange) {
        List customArgs = (List)exchange.getIn().getHeader("CamelDoclingCustomArguments", List.class);
        if (customArgs != null && !customArgs.isEmpty()) {
            LOG.debug("Adding custom Docling arguments: {}", (Object)customArgs);
            command.addAll(customArgs);
        }
    }

    private void addOutputFormatArguments(List<String> command, String outputFormat) {
        if (outputFormat != null && !outputFormat.isEmpty()) {
            command.add("--to");
            command.add(this.mapToDoclingFormat(outputFormat));
        }
    }

    private void addOcrArguments(List<String> command) {
        if (!this.configuration.isEnableOCR()) {
            command.add("--no-ocr");
        } else if (this.configuration.getOcrLanguage() != null) {
            command.add("--ocr-lang");
            command.add(this.configuration.getOcrLanguage());
        }
    }

    private void addLayoutArguments(List<String> command) {
        if (this.configuration.isIncludeLayoutInfo()) {
            command.add("--show-layout");
        }
    }

    private void addOutputDirectoryArguments(List<String> command, Exchange exchange, String outputDirectory) {
        String outputPath = (String)exchange.getIn().getHeader("CamelDoclingOutputFilePath", String.class);
        if (outputPath != null) {
            command.add("--output");
            command.add(outputPath);
        } else {
            command.add("--output");
            command.add(outputDirectory);
        }
    }

    private String mapToDoclingFormat(String outputFormat) {
        switch (outputFormat.toLowerCase()) {
            case "markdown": {
                return "md";
            }
            case "html": {
                return "html";
            }
            case "json": {
                return "json";
            }
            case "text": {
                return "text";
            }
        }
        return "md";
    }
}

