/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.gcp.bigquery;

import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.googleapis.services.AbstractGoogleClientRequest;
import com.google.api.client.util.BackOff;
import com.google.api.client.util.BackOffUtils;
import com.google.api.client.util.Sleeper;
import com.google.api.services.bigquery.Bigquery;
import com.google.api.services.bigquery.model.Dataset;
import com.google.api.services.bigquery.model.DatasetReference;
import com.google.api.services.bigquery.model.Job;
import com.google.api.services.bigquery.model.JobConfiguration;
import com.google.api.services.bigquery.model.JobConfigurationExtract;
import com.google.api.services.bigquery.model.JobConfigurationLoad;
import com.google.api.services.bigquery.model.JobConfigurationQuery;
import com.google.api.services.bigquery.model.JobReference;
import com.google.api.services.bigquery.model.JobStatistics;
import com.google.api.services.bigquery.model.JobStatus;
import com.google.api.services.bigquery.model.Table;
import com.google.api.services.bigquery.model.TableDataInsertAllRequest;
import com.google.api.services.bigquery.model.TableDataInsertAllResponse;
import com.google.api.services.bigquery.model.TableDataList;
import com.google.api.services.bigquery.model.TableReference;
import com.google.api.services.bigquery.model.TableRow;
import com.google.cloud.hadoop.util.ApiErrorExtractor;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.beam.sdk.io.gcp.bigquery.BigQueryServices;
import org.apache.beam.sdk.io.gcp.bigquery.BigQueryTableRowIterator;
import org.apache.beam.sdk.options.BigQueryOptions;
import org.apache.beam.sdk.options.GcsOptions;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.util.AttemptBoundedExponentialBackOff;
import org.apache.beam.sdk.util.IntervalBoundedExponentialBackOff;
import org.apache.beam.sdk.util.Transport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BigQueryServicesImpl
implements BigQueryServices {
    private static final Logger LOG = LoggerFactory.getLogger(BigQueryServicesImpl.class);
    private static final int MAX_RPC_ATTEMPTS = 10;
    private static final long INITIAL_RPC_BACKOFF_MILLIS = TimeUnit.SECONDS.toMillis(1L);
    private static final long INITIAL_JOB_STATUS_POLL_BACKOFF_MILLIS = TimeUnit.SECONDS.toMillis(1L);

    BigQueryServicesImpl() {
    }

    @Override
    public BigQueryServices.JobService getJobService(BigQueryOptions options) {
        return new JobServiceImpl(options);
    }

    @Override
    public BigQueryServices.DatasetService getDatasetService(BigQueryOptions options) {
        return new DatasetServiceImpl(options);
    }

    @Override
    public BigQueryServices.BigQueryJsonReader getReaderFromTable(BigQueryOptions bqOptions, TableReference tableRef) {
        return BigQueryJsonReaderImpl.fromTable(bqOptions, tableRef);
    }

    @Override
    public BigQueryServices.BigQueryJsonReader getReaderFromQuery(BigQueryOptions bqOptions, String query, String projectId, @Nullable Boolean flatten) {
        return BigQueryJsonReaderImpl.fromQuery(bqOptions, query, projectId, flatten);
    }

    @VisibleForTesting
    static <T> T executeWithRetries(AbstractGoogleClientRequest<T> request, String errorMessage, Sleeper sleeper, BackOff backoff) throws IOException, InterruptedException {
        IOException lastException = null;
        while (true) {
            try {
                return (T)request.execute();
            }
            catch (IOException e) {
                LOG.warn("Ignore the error and retry the request.", (Throwable)e);
                lastException = e;
                if (BigQueryServicesImpl.nextBackOff(sleeper, backoff)) continue;
                throw new IOException(errorMessage, lastException);
            }
            break;
        }
    }

    private static boolean nextBackOff(Sleeper sleeper, BackOff backoff) throws InterruptedException {
        try {
            return BackOffUtils.next((Sleeper)sleeper, (BackOff)backoff);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static class BigQueryJsonReaderImpl
    implements BigQueryServices.BigQueryJsonReader {
        BigQueryTableRowIterator iterator;

        private BigQueryJsonReaderImpl(BigQueryTableRowIterator iterator) {
            this.iterator = iterator;
        }

        private static BigQueryServices.BigQueryJsonReader fromQuery(BigQueryOptions bqOptions, String query, String projectId, @Nullable Boolean flattenResults) {
            return new BigQueryJsonReaderImpl(BigQueryTableRowIterator.fromQuery(query, projectId, Transport.newBigQueryClient((BigQueryOptions)bqOptions).build(), flattenResults));
        }

        private static BigQueryServices.BigQueryJsonReader fromTable(BigQueryOptions bqOptions, TableReference tableRef) {
            return new BigQueryJsonReaderImpl(BigQueryTableRowIterator.fromTable(tableRef, Transport.newBigQueryClient((BigQueryOptions)bqOptions).build()));
        }

        @Override
        public boolean start() throws IOException {
            try {
                this.iterator.open();
                return this.iterator.advance();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Interrupted during start() operation", e);
            }
        }

        @Override
        public boolean advance() throws IOException {
            try {
                return this.iterator.advance();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Interrupted during advance() operation", e);
            }
        }

        @Override
        public TableRow getCurrent() throws NoSuchElementException {
            return this.iterator.getCurrent();
        }

        @Override
        public void close() throws IOException {
            this.iterator.close();
        }
    }

    @VisibleForTesting
    static class DatasetServiceImpl
    implements BigQueryServices.DatasetService {
        private static final long UPLOAD_BATCH_SIZE_BYTES = 65536L;
        private static final long MAX_ROWS_PER_BATCH = 500L;
        private static final int MAX_INSERT_ATTEMPTS = 5;
        private static final long INITIAL_INSERT_BACKOFF_INTERVAL_MS = 200L;
        private static final long INITIAL_RATE_LIMIT_EXCEEDED_BACKOFF_MS = TimeUnit.SECONDS.toMillis(1L);
        private static final long MAX_RATE_LIMIT_EXCEEDED_BACKOFF_MS = TimeUnit.MINUTES.toMillis(2L);
        private final ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
        private final Bigquery client;
        private final PipelineOptions options;
        private final long maxRowsPerBatch;
        private ExecutorService executor;

        @VisibleForTesting
        DatasetServiceImpl(Bigquery client, PipelineOptions options) {
            this.client = client;
            this.options = options;
            this.maxRowsPerBatch = 500L;
            this.executor = null;
        }

        @VisibleForTesting
        DatasetServiceImpl(Bigquery client, PipelineOptions options, long maxRowsPerBatch) {
            this.client = client;
            this.options = options;
            this.maxRowsPerBatch = maxRowsPerBatch;
            this.executor = null;
        }

        private DatasetServiceImpl(BigQueryOptions bqOptions) {
            this.client = Transport.newBigQueryClient((BigQueryOptions)bqOptions).build();
            this.options = bqOptions;
            this.maxRowsPerBatch = 500L;
            this.executor = null;
        }

        @Override
        public Table getTable(String projectId, String datasetId, String tableId) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            return (Table)BigQueryServicesImpl.executeWithRetries(this.client.tables().get(projectId, datasetId, tableId), String.format("Unable to get table: %s, aborting after %d retries.", tableId, 10), Sleeper.DEFAULT, (BackOff)backoff);
        }

        @Override
        public void deleteTable(String projectId, String datasetId, String tableId) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            BigQueryServicesImpl.executeWithRetries(this.client.tables().delete(projectId, datasetId, tableId), String.format("Unable to delete table: %s, aborting after %d retries.", tableId, 10), Sleeper.DEFAULT, (BackOff)backoff);
        }

        @Override
        public boolean isTableEmpty(String projectId, String datasetId, String tableId) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            TableDataList dataList = (TableDataList)BigQueryServicesImpl.executeWithRetries(this.client.tabledata().list(projectId, datasetId, tableId), String.format("Unable to list table data: %s, aborting after %d retries.", tableId, 10), Sleeper.DEFAULT, (BackOff)backoff);
            return dataList.getRows() == null || dataList.getRows().isEmpty();
        }

        @Override
        public Dataset getDataset(String projectId, String datasetId) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            return (Dataset)BigQueryServicesImpl.executeWithRetries(this.client.datasets().get(projectId, datasetId), String.format("Unable to get dataset: %s, aborting after %d retries.", datasetId, 10), Sleeper.DEFAULT, (BackOff)backoff);
        }

        @Override
        public void createDataset(String projectId, String datasetId, String location, String description) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            this.createDataset(projectId, datasetId, location, description, Sleeper.DEFAULT, (BackOff)backoff);
        }

        @VisibleForTesting
        void createDataset(String projectId, String datasetId, String location, String description, Sleeper sleeper, BackOff backoff) throws IOException, InterruptedException {
            Throwable lastException;
            DatasetReference datasetRef = new DatasetReference().setProjectId(projectId).setDatasetId(datasetId);
            Dataset dataset = new Dataset().setDatasetReference(datasetRef).setLocation(location).setFriendlyName(location).setDescription(description);
            do {
                try {
                    this.client.datasets().insert(projectId, dataset).execute();
                    return;
                }
                catch (GoogleJsonResponseException e) {
                    if (this.errorExtractor.itemAlreadyExists((IOException)((Object)e))) {
                        return;
                    }
                    LOG.warn("Ignore the error and retry creating the dataset.", (Throwable)e);
                    lastException = e;
                }
                catch (IOException e) {
                    LOG.warn("Ignore the error and retry creating the dataset.", (Throwable)e);
                    lastException = e;
                }
            } while (BigQueryServicesImpl.nextBackOff(sleeper, backoff));
            throw new IOException(String.format("Unable to create dataset: %s, aborting after %d .", datasetId, 10), lastException);
        }

        @Override
        public void deleteDataset(String projectId, String datasetId) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            BigQueryServicesImpl.executeWithRetries(this.client.datasets().delete(projectId, datasetId), String.format("Unable to delete table: %s, aborting after %d retries.", datasetId, 10), Sleeper.DEFAULT, (BackOff)backoff);
        }

        @Override
        public long insertAll(TableReference ref, List<TableRow> rowList, @Nullable List<String> insertIdList) throws IOException, InterruptedException {
            Preconditions.checkNotNull((Object)ref, (Object)"ref");
            if (this.executor == null) {
                this.executor = ((GcsOptions)this.options.as(GcsOptions.class)).getExecutorService();
            }
            if (insertIdList != null && rowList.size() != insertIdList.size()) {
                throw new AssertionError((Object)"If insertIdList is not null it needs to have at least as many elements as rowList");
            }
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(5, 200L);
            long retTotalDataSize = 0L;
            ArrayList<TableDataInsertAllResponse.InsertErrors> allErrors = new ArrayList<TableDataInsertAllResponse.InsertErrors>();
            List<TableRow> rowsToPublish = rowList;
            List<String> idsToPublish = insertIdList;
            while (true) {
                int i;
                ArrayList<TableRow> retryRows = new ArrayList<TableRow>();
                ArrayList<String> retryIds = idsToPublish != null ? new ArrayList<String>() : null;
                int strideIndex = 0;
                LinkedList<TableDataInsertAllRequest.Rows> rows = new LinkedList<TableDataInsertAllRequest.Rows>();
                int dataSize = 0;
                ArrayList<Future<List<TableDataInsertAllResponse.InsertErrors>>> futures = new ArrayList<Future<List<TableDataInsertAllResponse.InsertErrors>>>();
                ArrayList<Integer> strideIndices = new ArrayList<Integer>();
                for (i = 0; i < rowsToPublish.size(); ++i) {
                    TableRow row = rowsToPublish.get(i);
                    TableDataInsertAllRequest.Rows out = new TableDataInsertAllRequest.Rows();
                    if (idsToPublish != null) {
                        out.setInsertId(idsToPublish.get(i));
                    }
                    out.setJson(row.getUnknownKeys());
                    rows.add(out);
                    if ((long)(dataSize += row.toString().length()) < 65536L && (long)rows.size() < this.maxRowsPerBatch && i != rowsToPublish.size() - 1) continue;
                    TableDataInsertAllRequest content = new TableDataInsertAllRequest();
                    content.setRows(rows);
                    final Bigquery.Tabledata.InsertAll insert = this.client.tabledata().insertAll(ref.getProjectId(), ref.getDatasetId(), ref.getTableId(), content);
                    futures.add(this.executor.submit(new Callable<List<TableDataInsertAllResponse.InsertErrors>>(){

                        @Override
                        public List<TableDataInsertAllResponse.InsertErrors> call() throws IOException {
                            IntervalBoundedExponentialBackOff backoff = new IntervalBoundedExponentialBackOff(MAX_RATE_LIMIT_EXCEEDED_BACKOFF_MS, INITIAL_RATE_LIMIT_EXCEEDED_BACKOFF_MS);
                            while (true) {
                                try {
                                    return ((TableDataInsertAllResponse)insert.execute()).getInsertErrors();
                                }
                                catch (IOException e) {
                                    if (new ApiErrorExtractor().rateLimited((Throwable)e)) {
                                        LOG.info("BigQuery insertAll exceeded rate limit, retrying");
                                        try {
                                            Thread.sleep(backoff.nextBackOffMillis());
                                        }
                                        catch (InterruptedException interrupted) {
                                            throw new IOException("Interrupted while waiting before retrying insertAll");
                                        }
                                        continue;
                                    }
                                    throw e;
                                }
                                break;
                            }
                        }
                    }));
                    strideIndices.add(strideIndex);
                    retTotalDataSize += (long)dataSize;
                    dataSize = 0;
                    strideIndex = i + 1;
                    rows = new LinkedList();
                }
                try {
                    for (i = 0; i < futures.size(); ++i) {
                        List errors = (List)((Future)futures.get(i)).get();
                        if (errors == null) continue;
                        for (TableDataInsertAllResponse.InsertErrors error : errors) {
                            allErrors.add(error);
                            if (error.getIndex() == null) {
                                throw new IOException("Insert failed: " + allErrors);
                            }
                            int errorIndex = error.getIndex().intValue() + (Integer)strideIndices.get(i);
                            retryRows.add(rowsToPublish.get(errorIndex));
                            if (retryIds == null) continue;
                            retryIds.add(idsToPublish.get(errorIndex));
                        }
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted while inserting " + rowsToPublish);
                }
                catch (ExecutionException e) {
                    throw new RuntimeException(e.getCause());
                }
                if (allErrors.isEmpty() || backoff.atMaxAttempts()) break;
                try {
                    Thread.sleep(backoff.nextBackOffMillis());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted while waiting before retrying insert of " + retryRows);
                }
                LOG.info("Retrying failed inserts to BigQuery");
                rowsToPublish = retryRows;
                idsToPublish = retryIds;
                allErrors.clear();
            }
            if (!allErrors.isEmpty()) {
                throw new IOException("Insert failed: " + allErrors);
            }
            return retTotalDataSize;
        }
    }

    @VisibleForTesting
    static class JobServiceImpl
    implements BigQueryServices.JobService {
        private final ApiErrorExtractor errorExtractor = new ApiErrorExtractor();
        private final Bigquery client;

        @VisibleForTesting
        JobServiceImpl(Bigquery client) {
            this.client = client;
        }

        private JobServiceImpl(BigQueryOptions options) {
            this.client = Transport.newBigQueryClient((BigQueryOptions)options).build();
        }

        @Override
        public void startLoadJob(JobReference jobRef, JobConfigurationLoad loadConfig) throws InterruptedException, IOException {
            Job job = new Job().setJobReference(jobRef).setConfiguration(new JobConfiguration().setLoad(loadConfig));
            JobServiceImpl.startJob(job, this.errorExtractor, this.client);
        }

        @Override
        public void startExtractJob(JobReference jobRef, JobConfigurationExtract extractConfig) throws InterruptedException, IOException {
            Job job = new Job().setJobReference(jobRef).setConfiguration(new JobConfiguration().setExtract(extractConfig));
            JobServiceImpl.startJob(job, this.errorExtractor, this.client);
        }

        @Override
        public void startQueryJob(JobReference jobRef, JobConfigurationQuery queryConfig) throws IOException, InterruptedException {
            Job job = new Job().setJobReference(jobRef).setConfiguration(new JobConfiguration().setQuery(queryConfig));
            JobServiceImpl.startJob(job, this.errorExtractor, this.client);
        }

        private static void startJob(Job job, ApiErrorExtractor errorExtractor, Bigquery client) throws IOException, InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            JobServiceImpl.startJob(job, errorExtractor, client, Sleeper.DEFAULT, (BackOff)backoff);
        }

        @VisibleForTesting
        static void startJob(Job job, ApiErrorExtractor errorExtractor, Bigquery client, Sleeper sleeper, BackOff backoff) throws IOException, InterruptedException {
            JobReference jobRef = job.getJobReference();
            Throwable lastException = null;
            do {
                try {
                    client.jobs().insert(jobRef.getProjectId(), job).execute();
                    return;
                }
                catch (GoogleJsonResponseException e) {
                    if (errorExtractor.itemAlreadyExists((IOException)((Object)e))) {
                        return;
                    }
                    LOG.warn("Ignore the error and retry inserting the job.", (Throwable)e);
                    lastException = e;
                }
                catch (IOException e) {
                    LOG.warn("Ignore the error and retry inserting the job.", (Throwable)e);
                    lastException = e;
                }
            } while (BigQueryServicesImpl.nextBackOff(sleeper, backoff));
            throw new IOException(String.format("Unable to insert job: %s, aborting after %d .", jobRef.getJobId(), 10), lastException);
        }

        @Override
        public Job pollJob(JobReference jobRef, int maxAttempts) throws InterruptedException {
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(maxAttempts, INITIAL_JOB_STATUS_POLL_BACKOFF_MILLIS);
            return this.pollJob(jobRef, Sleeper.DEFAULT, (BackOff)backoff);
        }

        @VisibleForTesting
        Job pollJob(JobReference jobRef, Sleeper sleeper, BackOff backoff) throws InterruptedException {
            do {
                try {
                    Job job = (Job)this.client.jobs().get(jobRef.getProjectId(), jobRef.getJobId()).execute();
                    JobStatus status = job.getStatus();
                    if (status != null && status.getState() != null && status.getState().equals("DONE")) {
                        return job;
                    }
                }
                catch (IOException e) {
                    LOG.warn("Ignore the error and retry polling job status.", (Throwable)e);
                }
            } while (BigQueryServicesImpl.nextBackOff(sleeper, backoff));
            LOG.warn("Unable to poll job status: {}, aborting after reached max .", (Object)jobRef.getJobId());
            return null;
        }

        @Override
        public JobStatistics dryRunQuery(String projectId, String query) throws InterruptedException, IOException {
            Job job = new Job().setConfiguration(new JobConfiguration().setQuery(new JobConfigurationQuery().setQuery(query)).setDryRun(Boolean.valueOf(true)));
            AttemptBoundedExponentialBackOff backoff = new AttemptBoundedExponentialBackOff(10, INITIAL_RPC_BACKOFF_MILLIS);
            return ((Job)BigQueryServicesImpl.executeWithRetries(this.client.jobs().insert(projectId, job), String.format("Unable to dry run query: %s, aborting after %d retries.", query, 10), Sleeper.DEFAULT, (BackOff)backoff)).getStatistics();
        }
    }
}

