/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.aries.proxy;

import java.util.Collection;
import java.util.concurrent.Callable;

import org.osgi.framework.Bundle;

/**
 * The proxy manager service allows clients to generate and manage proxies.
 */
public interface ProxyManager 
{
  /**
   * Create a proxy that delegates to an object instance which may change
   * over time
   * 
   * @param clientBundle The bundle providing the class to be proxied
   * @param classes  The interfaces and/or classes to be proxied
   * @param dispatcher A {@link Callable} that will called each time the proxy 
   *                   is invoked to locate the object to delegate to
   * @param template A template object for the proxy, may be null if only interfaces
   *                 need to be proxied. Supplying a templates typically offer a 
   *                 significant performance boost to the resulting proxy.
   * @return A proxy object that delegates to real objects under the covers
   * @throws UnableToProxyException
   */
  public Object createDelegatingProxy(Bundle clientBundle, Collection<Class<?>> classes, Callable<Object> dispatcher, Object template) throws UnableToProxyException;
  
  /**
   * Creates a proxy that invokes the supplied {@link InvocationListener} 
   * immediately before and after any non-private method is called.
   * 
   * @param clientBundle
   * @param classes
   * @param delegate
   * @param wrapper
   * @return
   * @throws UnableToProxyException
   */
  public Object createInterceptingProxy(Bundle clientBundle, Collection<Class<?>> classes, 
      Object delegate, InvocationListener wrapper) throws UnableToProxyException;
  
  /**
   * Creates a single proxy that both delegates and intercepts. See
   * ProxyManager.createDelegatingProxy(Bundle, Collection, Callable) 
   * and ProxyManager.createInterceptingProxy(Bundle, Collection, Object, InvocationListener)
   * 
   * @param clientBundle
   * @param classes
   * @param dispatcher
   * @param template A template object for the proxy, may be null if only interfaces
   *                 need to be proxied. Supplying a templates typically offer a 
   *                 significant performance boost to the resulting proxy.
   * @param wrapper
   * @return
   * @throws UnableToProxyException
   */
  public Object createDelegatingInterceptingProxy(Bundle clientBundle, Collection<Class<?>> classes, 
      Callable<Object> dispatcher, Object template, InvocationListener wrapper) throws UnableToProxyException;
  
  /**
   * This method unwraps the provided proxy returning the target object.
   * 
   * @param proxy the proxy to unwrap.
   * @return      the target object.
   */
  public Callable<Object> unwrap(Object proxy);
  /**
   * Returns true if and only if the specified object was generated by the ProxyManager. See
   * ProxyManager.createDelegatingProxy(Bundle, Collection, Callable) for details on how to create a proxy. 
   * @param proxy The proxy object to test
   * @return      true if it is a proxy, false otherwise.
   */
  public boolean isProxy(Object proxy);
}
