/*
 * Decompiled with CFR 0.152.
 */
package org.altbeacon.beacon;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import android.util.Log;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.logging.LogManager;
import org.altbeacon.bluetooth.BleAdvertisement;
import org.altbeacon.bluetooth.Pdu;

public class BeaconParser {
    private static final String TAG = "BeaconParser";
    public static final String ALTBEACON_LAYOUT = "m:2-3=beac,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25";
    public static final String EDDYSTONE_TLM_LAYOUT = "x,s:0-1=feaa,m:2-2=20,d:3-3,d:4-5,d:6-7,d:8-11,d:12-15";
    public static final String EDDYSTONE_UID_LAYOUT = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";
    public static final String EDDYSTONE_URL_LAYOUT = "s:0-1=feaa,m:2-2=10,p:3-3:-41,i:4-20v";
    private static final Pattern I_PATTERN = Pattern.compile("i\\:(\\d+)\\-(\\d+)([blv]*)?");
    private static final Pattern M_PATTERN = Pattern.compile("m\\:(\\d+)-(\\d+)\\=([0-9A-Fa-f]+)");
    private static final Pattern S_PATTERN = Pattern.compile("s\\:(\\d+)-(\\d+)\\=([0-9A-Fa-f]+)");
    private static final Pattern D_PATTERN = Pattern.compile("d\\:(\\d+)\\-(\\d+)([bl]*)?");
    private static final Pattern P_PATTERN = Pattern.compile("p\\:(\\d+)\\-(\\d+)\\:?([\\-\\d]+)?");
    private static final Pattern X_PATTERN = Pattern.compile("x");
    private static final char[] HEX_ARRAY = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
    private static final String LITTLE_ENDIAN_SUFFIX = "l";
    private static final String VARIABLE_LENGTH_SUFFIX = "v";
    private Long mMatchingBeaconTypeCode;
    protected final List<Integer> mIdentifierStartOffsets = new ArrayList<Integer>();
    protected final List<Integer> mIdentifierEndOffsets = new ArrayList<Integer>();
    protected final List<Boolean> mIdentifierLittleEndianFlags = new ArrayList<Boolean>();
    protected final List<Integer> mDataStartOffsets = new ArrayList<Integer>();
    protected final List<Integer> mDataEndOffsets = new ArrayList<Integer>();
    protected final List<Boolean> mDataLittleEndianFlags = new ArrayList<Boolean>();
    protected final List<Boolean> mIdentifierVariableLengthFlags = new ArrayList<Boolean>();
    protected Integer mMatchingBeaconTypeCodeStartOffset;
    protected Integer mMatchingBeaconTypeCodeEndOffset;
    protected Integer mServiceUuidStartOffset;
    protected Integer mServiceUuidEndOffset;
    protected Long mServiceUuid;
    protected Boolean mExtraFrame;
    protected Integer mPowerStartOffset;
    protected Integer mPowerEndOffset;
    protected Integer mDBmCorrection;
    protected Integer mLayoutSize;
    protected Boolean mAllowPduOverflow = true;
    protected String mIdentifier;
    protected int[] mHardwareAssistManufacturers = new int[]{76};

    public BeaconParser() {
    }

    public BeaconParser(String identifier) {
        this.mIdentifier = identifier;
    }

    public BeaconParser setBeaconLayout(String beaconLayout) {
        Log.d((String)TAG, (String)("Parsing beacon layout: " + beaconLayout));
        String[] terms = beaconLayout.split(",");
        this.mExtraFrame = false;
        for (String term : terms) {
            Boolean littleEndian;
            int endOffset;
            int startOffset;
            boolean found = false;
            Matcher matcher = I_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    littleEndian = matcher.group(3).contains(LITTLE_ENDIAN_SUFFIX);
                    this.mIdentifierLittleEndianFlags.add(littleEndian);
                    Boolean variableLength = matcher.group(3).contains(VARIABLE_LENGTH_SUFFIX);
                    this.mIdentifierVariableLengthFlags.add(variableLength);
                    this.mIdentifierStartOffsets.add(startOffset);
                    this.mIdentifierEndOffsets.add(endOffset);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
            }
            matcher = D_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    littleEndian = matcher.group(3).contains(LITTLE_ENDIAN_SUFFIX);
                    this.mDataLittleEndianFlags.add(littleEndian);
                    this.mDataStartOffsets.add(startOffset);
                    this.mDataEndOffsets.add(endOffset);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
            }
            matcher = P_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    int dBmCorrection = 0;
                    if (matcher.group(3) != null) {
                        dBmCorrection = Integer.parseInt(matcher.group(3));
                    }
                    this.mDBmCorrection = dBmCorrection;
                    this.mPowerStartOffset = startOffset;
                    this.mPowerEndOffset = endOffset;
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer power byte offset in term: " + term);
                }
            }
            matcher = M_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    int startOffset2 = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    this.mMatchingBeaconTypeCodeStartOffset = startOffset2;
                    this.mMatchingBeaconTypeCodeEndOffset = endOffset;
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
                String hexString = matcher.group(3);
                try {
                    this.mMatchingBeaconTypeCode = Long.decode("0x" + hexString);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse beacon type code: " + hexString + " in term: " + term);
                }
            }
            matcher = S_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    int startOffset3 = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    this.mServiceUuidStartOffset = startOffset3;
                    this.mServiceUuidEndOffset = endOffset;
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
                String hexString = matcher.group(3);
                try {
                    this.mServiceUuid = Long.decode("0x" + hexString);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse serviceUuid: " + hexString + " in term: " + term);
                }
            }
            matcher = X_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                this.mExtraFrame = true;
            }
            if (found) continue;
            LogManager.d(TAG, "cannot parse term %s", term);
            throw new BeaconLayoutException("Cannot parse beacon layout term: " + term);
        }
        if (!this.mExtraFrame.booleanValue()) {
            if (this.mIdentifierStartOffsets.size() == 0 || this.mIdentifierEndOffsets.size() == 0) {
                throw new BeaconLayoutException("You must supply at least one identifier offset with a prefix of 'i'");
            }
            if (this.mPowerStartOffset == null || this.mPowerEndOffset == null) {
                throw new BeaconLayoutException("You must supply a power byte offset with a prefix of 'p'");
            }
        }
        if (this.mMatchingBeaconTypeCodeStartOffset == null || this.mMatchingBeaconTypeCodeEndOffset == null) {
            throw new BeaconLayoutException("You must supply a matching beacon type expression with a prefix of 'm'");
        }
        this.mLayoutSize = this.calculateLayoutSize();
        return this;
    }

    public String getIdentifier() {
        return this.mIdentifier;
    }

    public int[] getHardwareAssistManufacturers() {
        return this.mHardwareAssistManufacturers;
    }

    public void setHardwareAssistManufacturerCodes(int[] manufacturers) {
        this.mHardwareAssistManufacturers = manufacturers;
    }

    public void setAllowPduOverflow(Boolean enabled) {
        this.mAllowPduOverflow = enabled;
    }

    public Long getMatchingBeaconTypeCode() {
        return this.mMatchingBeaconTypeCode;
    }

    public int getMatchingBeaconTypeCodeStartOffset() {
        return this.mMatchingBeaconTypeCodeStartOffset;
    }

    public int getMatchingBeaconTypeCodeEndOffset() {
        return this.mMatchingBeaconTypeCodeEndOffset;
    }

    public Long getServiceUuid() {
        return this.mServiceUuid;
    }

    public int getMServiceUuidStartOffset() {
        return this.mServiceUuidStartOffset;
    }

    public int getServiceUuidEndOffset() {
        return this.mServiceUuidEndOffset;
    }

    @TargetApi(value=5)
    public Beacon fromScanData(byte[] scanData, int rssi, BluetoothDevice device) {
        return this.fromScanData(scanData, rssi, device, new Beacon());
    }

    @TargetApi(value=5)
    protected Beacon fromScanData(byte[] bytesToProcess, int rssi, BluetoothDevice device, Beacon beacon) {
        BleAdvertisement advert = new BleAdvertisement(bytesToProcess);
        boolean parseFailed = false;
        Pdu pduToParse = null;
        int startByte = 0;
        ArrayList<Identifier> identifiers = new ArrayList<Identifier>();
        ArrayList<Long> dataFields = new ArrayList<Long>();
        for (Pdu pdu : advert.getPdus()) {
            if (pdu.getType() == 22 || pdu.getType() == -1) {
                pduToParse = pdu;
                if (!LogManager.isVerboseLoggingEnabled()) break;
                LogManager.d(TAG, "Processing pdu type %02X: %s with startIndex: %d, endIndex: %d", pdu.getType(), BeaconParser.bytesToHex(bytesToProcess), pdu.getStartIndex(), pdu.getEndIndex());
                break;
            }
            if (!LogManager.isVerboseLoggingEnabled()) continue;
            LogManager.d(TAG, "Ignoring pdu type %02X", pdu.getType());
        }
        if (pduToParse == null) {
            if (LogManager.isVerboseLoggingEnabled()) {
                LogManager.d(TAG, "No PDUs to process in this packet.", new Object[0]);
            }
            parseFailed = true;
        } else {
            byte[] serviceUuidBytes = null;
            byte[] typeCodeBytes = BeaconParser.longToByteArray(this.getMatchingBeaconTypeCode(), this.mMatchingBeaconTypeCodeEndOffset - this.mMatchingBeaconTypeCodeStartOffset + 1);
            if (this.getServiceUuid() != null) {
                serviceUuidBytes = BeaconParser.longToByteArray(this.getServiceUuid(), this.mServiceUuidEndOffset - this.mServiceUuidStartOffset + 1, false);
            }
            startByte = pduToParse.getStartIndex();
            boolean patternFound = false;
            if (this.getServiceUuid() == null) {
                if (this.byteArraysMatch(bytesToProcess, startByte + this.mMatchingBeaconTypeCodeStartOffset, typeCodeBytes, 0)) {
                    patternFound = true;
                }
            } else if (this.byteArraysMatch(bytesToProcess, startByte + this.mServiceUuidStartOffset, serviceUuidBytes, 0) && this.byteArraysMatch(bytesToProcess, startByte + this.mMatchingBeaconTypeCodeStartOffset, typeCodeBytes, 0)) {
                patternFound = true;
            }
            if (!patternFound) {
                if (this.getServiceUuid() == null) {
                    if (LogManager.isVerboseLoggingEnabled()) {
                        LogManager.d(TAG, "This is not a matching Beacon advertisement. (Was expecting %s.  The bytes I see are: %s", this.byteArrayToString(typeCodeBytes), BeaconParser.bytesToHex(bytesToProcess));
                    }
                } else if (LogManager.isVerboseLoggingEnabled()) {
                    LogManager.d(TAG, "This is not a matching Beacon advertisement. Was expecting %s at offset %d and %s at offset %d.  The bytes I see are: %s", this.byteArrayToString(serviceUuidBytes), startByte + this.mServiceUuidStartOffset, this.byteArrayToString(typeCodeBytes), startByte + this.mMatchingBeaconTypeCodeStartOffset, BeaconParser.bytesToHex(bytesToProcess));
                }
                parseFailed = true;
                beacon = null;
            } else if (LogManager.isVerboseLoggingEnabled()) {
                LogManager.d(TAG, "This is a recognized beacon advertisement -- %s seen", this.byteArrayToString(typeCodeBytes));
                LogManager.d(TAG, "Bytes are: %s", BeaconParser.bytesToHex(bytesToProcess));
            }
            if (patternFound) {
                int endIndex;
                int i;
                if (bytesToProcess.length <= startByte + this.mLayoutSize && this.mAllowPduOverflow.booleanValue()) {
                    if (LogManager.isVerboseLoggingEnabled()) {
                        LogManager.d(TAG, "Expanding buffer because it is too short to parse: " + bytesToProcess.length + ", needed: " + (startByte + this.mLayoutSize), new Object[0]);
                    }
                    bytesToProcess = this.ensureMaxSize(bytesToProcess, startByte + this.mLayoutSize);
                }
                for (i = 0; i < this.mIdentifierEndOffsets.size(); ++i) {
                    Identifier identifier;
                    endIndex = this.mIdentifierEndOffsets.get(i) + startByte;
                    if (endIndex > pduToParse.getEndIndex() && this.mIdentifierVariableLengthFlags.get(i).booleanValue()) {
                        if (LogManager.isVerboseLoggingEnabled()) {
                            LogManager.d(TAG, "Need to truncate identifier by " + (endIndex - pduToParse.getEndIndex()), new Object[0]);
                        }
                        identifier = Identifier.fromBytes(bytesToProcess, this.mIdentifierStartOffsets.get(i) + startByte, pduToParse.getEndIndex() + 1, this.mIdentifierLittleEndianFlags.get(i));
                        identifiers.add(identifier);
                        continue;
                    }
                    if (endIndex > pduToParse.getEndIndex() && !this.mAllowPduOverflow.booleanValue()) {
                        parseFailed = true;
                        if (!LogManager.isVerboseLoggingEnabled()) continue;
                        LogManager.d(TAG, "Cannot parse identifier " + i + " because PDU is too short.  endIndex: " + endIndex + " PDU endIndex: " + pduToParse.getEndIndex(), new Object[0]);
                        continue;
                    }
                    identifier = Identifier.fromBytes(bytesToProcess, this.mIdentifierStartOffsets.get(i) + startByte, endIndex + 1, this.mIdentifierLittleEndianFlags.get(i));
                    identifiers.add(identifier);
                }
                for (i = 0; i < this.mDataEndOffsets.size(); ++i) {
                    endIndex = this.mDataEndOffsets.get(i) + startByte;
                    if (endIndex > pduToParse.getEndIndex() && !this.mAllowPduOverflow.booleanValue()) {
                        if (LogManager.isVerboseLoggingEnabled()) {
                            LogManager.d(TAG, "Cannot parse data field " + i + " because PDU is too short.  endIndex: " + endIndex + " PDU endIndex: " + pduToParse.getEndIndex() + ".  Setting value to 0", new Object[0]);
                        }
                        dataFields.add(new Long(0L));
                        continue;
                    }
                    String dataString = this.byteArrayToFormattedString(bytesToProcess, this.mDataStartOffsets.get(i) + startByte, endIndex, this.mDataLittleEndianFlags.get(i));
                    dataFields.add(Long.parseLong(dataString));
                }
                if (this.mPowerStartOffset != null) {
                    int endIndex2 = this.mPowerEndOffset + startByte;
                    int txPower = 0;
                    try {
                        if (endIndex2 > pduToParse.getEndIndex() && !this.mAllowPduOverflow.booleanValue()) {
                            parseFailed = true;
                            if (LogManager.isVerboseLoggingEnabled()) {
                                LogManager.d(TAG, "Cannot parse power field because PDU is too short.  endIndex: " + endIndex2 + " PDU endIndex: " + pduToParse.getEndIndex(), new Object[0]);
                            }
                        } else {
                            String powerString = this.byteArrayToFormattedString(bytesToProcess, this.mPowerStartOffset + startByte, this.mPowerEndOffset + startByte, false);
                            txPower = Integer.parseInt(powerString) + this.mDBmCorrection;
                            if (txPower > 127) {
                                txPower -= 256;
                            }
                            beacon.mTxPower = txPower;
                        }
                    }
                    catch (NumberFormatException e1) {
                    }
                    catch (NullPointerException e2) {
                        // empty catch block
                    }
                }
            }
        }
        if (parseFailed) {
            beacon = null;
        } else {
            int beaconTypeCode = 0;
            String beaconTypeString = this.byteArrayToFormattedString(bytesToProcess, this.mMatchingBeaconTypeCodeStartOffset + startByte, this.mMatchingBeaconTypeCodeEndOffset + startByte, false);
            beaconTypeCode = Integer.parseInt(beaconTypeString);
            int manufacturer = 0;
            String manufacturerString = this.byteArrayToFormattedString(bytesToProcess, startByte, startByte + 1, true);
            manufacturer = Integer.parseInt(manufacturerString);
            String macAddress = null;
            String name = null;
            if (device != null) {
                macAddress = device.getAddress();
                name = device.getName();
            }
            beacon.mIdentifiers = identifiers;
            beacon.mDataFields = dataFields;
            beacon.mRssi = rssi;
            beacon.mBeaconTypeCode = beaconTypeCode;
            beacon.mServiceUuid = this.mServiceUuid != null ? (int)this.mServiceUuid.longValue() : -1;
            beacon.mBluetoothAddress = macAddress;
            beacon.mBluetoothName = name;
            beacon.mManufacturer = manufacturer;
            beacon.mParserIdentifier = this.mIdentifier;
        }
        return beacon;
    }

    @TargetApi(value=9)
    public byte[] getBeaconAdvertisementData(Beacon beacon) {
        int index;
        int identifierNum;
        if (beacon.getIdentifiers().size() != this.getIdentifierCount()) {
            throw new IllegalArgumentException("Beacon has " + beacon.getIdentifiers().size() + " identifiers but format requires " + this.getIdentifierCount());
        }
        int lastIndex = -1;
        if (this.mMatchingBeaconTypeCodeEndOffset != null && this.mMatchingBeaconTypeCodeEndOffset > lastIndex) {
            lastIndex = this.mMatchingBeaconTypeCodeEndOffset;
        }
        if (this.mPowerEndOffset != null && this.mPowerEndOffset > lastIndex) {
            lastIndex = this.mPowerEndOffset;
        }
        for (identifierNum = 0; identifierNum < this.mIdentifierEndOffsets.size(); ++identifierNum) {
            if (this.mIdentifierEndOffsets.get(identifierNum) == null || this.mIdentifierEndOffsets.get(identifierNum) <= lastIndex) continue;
            lastIndex = this.mIdentifierEndOffsets.get(identifierNum);
        }
        for (identifierNum = 0; identifierNum < this.mDataEndOffsets.size(); ++identifierNum) {
            if (this.mDataEndOffsets.get(identifierNum) == null || this.mDataEndOffsets.get(identifierNum) <= lastIndex) continue;
            lastIndex = this.mDataEndOffsets.get(identifierNum);
        }
        int adjustedIdentifiersLength = 0;
        for (int identifierNum2 = 0; identifierNum2 < this.mIdentifierStartOffsets.size(); ++identifierNum2) {
            if (!this.mIdentifierVariableLengthFlags.get(identifierNum2).booleanValue()) continue;
            int declaredIdentifierLength = this.mIdentifierEndOffsets.get(identifierNum2) - this.mIdentifierStartOffsets.get(identifierNum2) + 1;
            int actualIdentifierLength = beacon.getIdentifier(identifierNum2).getByteCount();
            adjustedIdentifiersLength += actualIdentifierLength;
            adjustedIdentifiersLength -= declaredIdentifierLength;
        }
        byte[] advertisingBytes = new byte[(lastIndex += adjustedIdentifiersLength) + 1 - 2];
        long beaconTypeCode = this.getMatchingBeaconTypeCode();
        for (index = this.mMatchingBeaconTypeCodeStartOffset.intValue(); index <= this.mMatchingBeaconTypeCodeEndOffset; ++index) {
            byte value;
            advertisingBytes[index - 2] = value = (byte)(this.getMatchingBeaconTypeCode() >> 8 * (this.mMatchingBeaconTypeCodeEndOffset - index) & 0xFFL);
        }
        for (int identifierNum3 = 0; identifierNum3 < this.mIdentifierStartOffsets.size(); ++identifierNum3) {
            byte[] identifierBytes = beacon.getIdentifier(identifierNum3).toByteArrayOfSpecifiedEndianness(this.mIdentifierLittleEndianFlags.get(identifierNum3) == false);
            if (identifierBytes.length < this.getIdentifierByteCount(identifierNum3)) {
                if (!this.mIdentifierVariableLengthFlags.get(identifierNum3).booleanValue()) {
                    if (this.mIdentifierLittleEndianFlags.get(identifierNum3).booleanValue()) {
                        identifierBytes = Arrays.copyOf(identifierBytes, this.getIdentifierByteCount(identifierNum3));
                    } else {
                        byte[] newIdentifierBytes = new byte[this.getIdentifierByteCount(identifierNum3)];
                        System.arraycopy(identifierBytes, 0, newIdentifierBytes, this.getIdentifierByteCount(identifierNum3) - identifierBytes.length, identifierBytes.length);
                        identifierBytes = newIdentifierBytes;
                    }
                }
                LogManager.d(TAG, "Expanded identifier because it is too short.  It is now: " + this.byteArrayToString(identifierBytes), new Object[0]);
            } else if (identifierBytes.length > this.getIdentifierByteCount(identifierNum3)) {
                identifierBytes = this.mIdentifierLittleEndianFlags.get(identifierNum3) != false ? Arrays.copyOfRange(identifierBytes, this.getIdentifierByteCount(identifierNum3) - identifierBytes.length, this.getIdentifierByteCount(identifierNum3)) : Arrays.copyOf(identifierBytes, this.getIdentifierByteCount(identifierNum3));
                LogManager.d(TAG, "Truncated identifier because it is too long.  It is now: " + this.byteArrayToString(identifierBytes), new Object[0]);
            } else {
                LogManager.d(TAG, "Identifier size is just right: " + this.byteArrayToString(identifierBytes), new Object[0]);
            }
            for (int index2 = this.mIdentifierStartOffsets.get(identifierNum3).intValue(); index2 <= this.mIdentifierStartOffsets.get(identifierNum3) + identifierBytes.length - 1; ++index2) {
                advertisingBytes[index2 - 2] = identifierBytes[index2 - this.mIdentifierStartOffsets.get(identifierNum3)];
            }
        }
        if (this.mPowerStartOffset != null && this.mPowerEndOffset != null) {
            for (index = this.mPowerStartOffset.intValue(); index <= this.mPowerEndOffset; ++index) {
                advertisingBytes[index - 2] = (byte)(beacon.getTxPower() >> 8 * (index - this.mPowerStartOffset) & 0xFF);
            }
        }
        for (int dataFieldNum = 0; dataFieldNum < this.mDataStartOffsets.size(); ++dataFieldNum) {
            long dataField = beacon.getDataFields().get(dataFieldNum);
            int dataFieldLength = this.mDataEndOffsets.get(dataFieldNum) - this.mDataStartOffsets.get(dataFieldNum);
            for (int index3 = 0; index3 <= dataFieldLength; ++index3) {
                int endianCorrectedIndex = index3;
                if (!this.mDataLittleEndianFlags.get(dataFieldNum).booleanValue()) {
                    endianCorrectedIndex = dataFieldLength - index3;
                }
                advertisingBytes[this.mDataStartOffsets.get((int)dataFieldNum).intValue() - 2 + endianCorrectedIndex] = (byte)(dataField >> 8 * index3 & 0xFFL);
            }
        }
        return advertisingBytes;
    }

    public BeaconParser setMatchingBeaconTypeCode(Long typeCode) {
        this.mMatchingBeaconTypeCode = typeCode;
        return this;
    }

    public int getIdentifierByteCount(int identifierNum) {
        return this.mIdentifierEndOffsets.get(identifierNum) - this.mIdentifierStartOffsets.get(identifierNum) + 1;
    }

    public int getIdentifierCount() {
        return this.mIdentifierStartOffsets.size();
    }

    public int getDataFieldCount() {
        return this.mDataStartOffsets.size();
    }

    public int getPowerCorrection() {
        return this.mDBmCorrection;
    }

    protected static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    public static byte[] longToByteArray(long longValue, int length) {
        return BeaconParser.longToByteArray(longValue, length, true);
    }

    public static byte[] longToByteArray(long longValue, int length, boolean bigEndian) {
        byte[] array = new byte[length];
        for (int i = 0; i < length; ++i) {
            int adjustedI = bigEndian ? i : length - i - 1;
            long mask = 255L << (length - adjustedI - 1) * 8;
            long shift = (length - adjustedI - 1) * 8;
            long value = (longValue & mask) >> (int)shift;
            array[i] = (byte)value;
        }
        return array;
    }

    private int calculateLayoutSize() {
        int lastEndOffset = 0;
        if (this.mIdentifierEndOffsets != null) {
            for (int endOffset : this.mIdentifierEndOffsets) {
                if (endOffset <= lastEndOffset) continue;
                lastEndOffset = endOffset;
            }
        }
        if (this.mDataEndOffsets != null) {
            for (int endOffset : this.mDataEndOffsets) {
                if (endOffset <= lastEndOffset) continue;
                lastEndOffset = endOffset;
            }
        }
        if (this.mPowerEndOffset != null && this.mPowerEndOffset > lastEndOffset) {
            lastEndOffset = this.mPowerEndOffset;
        }
        if (this.mServiceUuidEndOffset != null && this.mServiceUuidEndOffset > lastEndOffset) {
            lastEndOffset = this.mServiceUuidEndOffset;
        }
        return lastEndOffset + 1;
    }

    private boolean byteArraysMatch(byte[] array1, int offset1, byte[] array2, int offset2) {
        int minSize;
        int n = minSize = array1.length > array2.length ? array2.length : array1.length;
        if (offset1 + minSize > array1.length || offset2 + minSize > array2.length) {
            return false;
        }
        for (int i = 0; i < minSize; ++i) {
            if (array1[i + offset1] == array2[i + offset2]) continue;
            return false;
        }
        return true;
    }

    private String byteArrayToString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < bytes.length; ++i) {
            sb.append(String.format("%02x", bytes[i]));
            sb.append(" ");
        }
        return sb.toString().trim();
    }

    private String byteArrayToFormattedString(byte[] byteBuffer, int startIndex, int endIndex, boolean littleEndian) {
        int length;
        int i;
        byte[] bytes = new byte[endIndex - startIndex + 1];
        if (littleEndian) {
            for (i = 0; i <= endIndex - startIndex; ++i) {
                bytes[i] = byteBuffer[startIndex + bytes.length - 1 - i];
            }
        } else {
            for (i = 0; i <= endIndex - startIndex; ++i) {
                bytes[i] = byteBuffer[startIndex + i];
            }
        }
        if ((length = endIndex - startIndex + 1) < 5) {
            long number = 0L;
            for (int i2 = 0; i2 < bytes.length; ++i2) {
                long byteValue = bytes[bytes.length - i2 - 1] & 0xFF;
                long positionValue = (long)Math.pow(256.0, (double)i2 * 1.0);
                long calculatedValue = byteValue * positionValue;
                number += calculatedValue;
            }
            return Long.toString(number);
        }
        String hexString = BeaconParser.bytesToHex(bytes);
        if (bytes.length == 16) {
            StringBuilder sb = new StringBuilder();
            sb.append(hexString.substring(0, 8));
            sb.append("-");
            sb.append(hexString.substring(8, 12));
            sb.append("-");
            sb.append(hexString.substring(12, 16));
            sb.append("-");
            sb.append(hexString.substring(16, 20));
            sb.append("-");
            sb.append(hexString.substring(20, 32));
            return sb.toString();
        }
        return "0x" + hexString;
    }

    @TargetApi(value=9)
    private byte[] ensureMaxSize(byte[] array, int requiredLength) {
        if (array.length >= requiredLength) {
            return array;
        }
        return Arrays.copyOf(array, requiredLength);
    }

    public static class BeaconLayoutException
    extends RuntimeException {
        public BeaconLayoutException(String s) {
            super(s);
        }
    }
}

