/*
 * Decompiled with CFR 0.152.
 */
package org.altbeacon.beacon;

import android.annotation.TargetApi;
import android.bluetooth.BluetoothDevice;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.logging.LogManager;

public class BeaconParser {
    private static final String TAG = "BeaconParser";
    private static final Pattern I_PATTERN = Pattern.compile("i\\:(\\d+)\\-(\\d+)(l?)");
    private static final Pattern M_PATTERN = Pattern.compile("m\\:(\\d+)-(\\d+)\\=([0-9A-Fa-f]+)");
    private static final Pattern D_PATTERN = Pattern.compile("d\\:(\\d+)\\-(\\d+)([bl]?)");
    private static final Pattern P_PATTERN = Pattern.compile("p\\:(\\d+)\\-(\\d+)");
    private static final char[] HEX_ARRAY = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
    private Long mMatchingBeaconTypeCode;
    protected List<Integer> mIdentifierStartOffsets;
    protected List<Integer> mIdentifierEndOffsets;
    protected List<Boolean> mIdentifierLittleEndianFlags;
    protected List<Integer> mDataStartOffsets;
    protected List<Integer> mDataEndOffsets;
    protected List<Boolean> mDataLittleEndianFlags;
    protected Integer mMatchingBeaconTypeCodeStartOffset;
    protected Integer mMatchingBeaconTypeCodeEndOffset;
    protected Integer mPowerStartOffset;
    protected Integer mPowerEndOffset;
    protected int[] mHardwareAssistManufacturers = new int[]{76};

    public BeaconParser() {
        this.mIdentifierStartOffsets = new ArrayList<Integer>();
        this.mIdentifierEndOffsets = new ArrayList<Integer>();
        this.mDataStartOffsets = new ArrayList<Integer>();
        this.mDataEndOffsets = new ArrayList<Integer>();
        this.mDataLittleEndianFlags = new ArrayList<Boolean>();
        this.mIdentifierLittleEndianFlags = new ArrayList<Boolean>();
    }

    public BeaconParser setBeaconLayout(String beaconLayout) {
        String[] terms;
        for (String term : terms = beaconLayout.split(",")) {
            Boolean littleEndian;
            int endOffset;
            int startOffset;
            boolean found = false;
            Matcher matcher = I_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    littleEndian = matcher.group(3).equals("l");
                    this.mIdentifierLittleEndianFlags.add(littleEndian);
                    this.mIdentifierStartOffsets.add(startOffset);
                    this.mIdentifierEndOffsets.add(endOffset);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
            }
            matcher = D_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    littleEndian = matcher.group(3).equals("l");
                    this.mDataLittleEndianFlags.add(littleEndian);
                    this.mDataStartOffsets.add(startOffset);
                    this.mDataEndOffsets.add(endOffset);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
            }
            matcher = P_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    startOffset = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    this.mPowerStartOffset = startOffset;
                    this.mPowerEndOffset = endOffset;
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer power byte offset in term: " + term);
                }
            }
            matcher = M_PATTERN.matcher(term);
            while (matcher.find()) {
                found = true;
                try {
                    int startOffset2 = Integer.parseInt(matcher.group(1));
                    endOffset = Integer.parseInt(matcher.group(2));
                    this.mMatchingBeaconTypeCodeStartOffset = startOffset2;
                    this.mMatchingBeaconTypeCodeEndOffset = endOffset;
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse integer byte offset in term: " + term);
                }
                String hexString = matcher.group(3);
                try {
                    this.mMatchingBeaconTypeCode = Long.decode("0x" + hexString);
                }
                catch (NumberFormatException e) {
                    throw new BeaconLayoutException("Cannot parse beacon type code: " + hexString + " in term: " + term);
                }
            }
            if (found) continue;
            LogManager.d(TAG, "cannot parse term %s", term);
            throw new BeaconLayoutException("Cannot parse beacon layout term: " + term);
        }
        if (this.mPowerStartOffset == null || this.mPowerEndOffset == null) {
            throw new BeaconLayoutException("You must supply a power byte offset with a prefix of 'p'");
        }
        if (this.mMatchingBeaconTypeCodeStartOffset == null || this.mMatchingBeaconTypeCodeEndOffset == null) {
            throw new BeaconLayoutException("You must supply a matching beacon type expression with a prefix of 'm'");
        }
        if (this.mIdentifierStartOffsets.size() == 0 || this.mIdentifierEndOffsets.size() == 0) {
            throw new BeaconLayoutException("You must supply at least one identifier offset withh a prefix of 'i'");
        }
        return this;
    }

    public int[] getHardwareAssistManufacturers() {
        return this.mHardwareAssistManufacturers;
    }

    public void setHardwareAssistManufacturerCodes(int[] manufacturers) {
        this.mHardwareAssistManufacturers = manufacturers;
    }

    public Long getMatchingBeaconTypeCode() {
        return this.mMatchingBeaconTypeCode;
    }

    public int getMatchingBeaconTypeCodeStartOffset() {
        return this.mMatchingBeaconTypeCodeStartOffset;
    }

    public int getMatchingBeaconTypeCodeEndOffset() {
        return this.mMatchingBeaconTypeCodeEndOffset;
    }

    @TargetApi(value=5)
    public Beacon fromScanData(byte[] scanData, int rssi, BluetoothDevice device) {
        return this.fromScanData(scanData, rssi, device, new Beacon());
    }

    @TargetApi(value=5)
    protected Beacon fromScanData(byte[] scanData, int rssi, BluetoothDevice device, Beacon beacon) {
        int startByte;
        boolean patternFound = false;
        byte[] typeCodeBytes = BeaconParser.longToByteArray(this.getMatchingBeaconTypeCode(), this.mMatchingBeaconTypeCodeEndOffset - this.mMatchingBeaconTypeCodeStartOffset + 1);
        for (startByte = 2; startByte <= 5; ++startByte) {
            if (!this.byteArraysMatch(scanData, startByte + this.mMatchingBeaconTypeCodeStartOffset, typeCodeBytes, 0)) continue;
            patternFound = true;
            break;
        }
        if (!patternFound) {
            LogManager.d(TAG, "This is not a matching Beacon advertisement. (Was expecting %s.  The bytes I see are: %s", this.byteArrayToString(typeCodeBytes), BeaconParser.bytesToHex(scanData));
            return null;
        }
        LogManager.d(TAG, "This is a recognized beacon advertisement -- %04x seen", this.getMatchingBeaconTypeCode());
        ArrayList<Identifier> identifiers = new ArrayList<Identifier>();
        for (int i = 0; i < this.mIdentifierEndOffsets.size(); ++i) {
            Identifier identifier = Identifier.fromBytes(scanData, this.mIdentifierStartOffsets.get(i) + startByte, this.mIdentifierEndOffsets.get(i) + startByte + 1, this.mIdentifierLittleEndianFlags.get(i));
            identifiers.add(identifier);
        }
        ArrayList<Long> dataFields = new ArrayList<Long>();
        for (int i = 0; i < this.mDataEndOffsets.size(); ++i) {
            String dataString = this.byteArrayToFormattedString(scanData, this.mDataStartOffsets.get(i) + startByte, this.mDataEndOffsets.get(i) + startByte, this.mDataLittleEndianFlags.get(i));
            dataFields.add(Long.parseLong(dataString));
            LogManager.d(TAG, "parsing found data field %s", i);
        }
        int txPower = 0;
        String powerString = this.byteArrayToFormattedString(scanData, this.mPowerStartOffset + startByte, this.mPowerEndOffset + startByte, false);
        txPower = Integer.parseInt(powerString);
        if (txPower > 127) {
            txPower -= 256;
        }
        int beaconTypeCode = 0;
        String beaconTypeString = this.byteArrayToFormattedString(scanData, this.mMatchingBeaconTypeCodeStartOffset + startByte, this.mMatchingBeaconTypeCodeEndOffset + startByte, false);
        beaconTypeCode = Integer.parseInt(beaconTypeString);
        int manufacturer = 0;
        String manufacturerString = this.byteArrayToFormattedString(scanData, startByte, startByte + 1, true);
        manufacturer = Integer.parseInt(manufacturerString);
        String macAddress = null;
        String name = null;
        if (device != null) {
            macAddress = device.getAddress();
            name = device.getName();
        }
        beacon.mIdentifiers = identifiers;
        beacon.mDataFields = dataFields;
        beacon.mTxPower = txPower;
        beacon.mRssi = rssi;
        beacon.mBeaconTypeCode = beaconTypeCode;
        beacon.mBluetoothAddress = macAddress;
        beacon.mBluetoothName = name;
        beacon.mManufacturer = manufacturer;
        return beacon;
    }

    public byte[] getBeaconAdvertisementData(Beacon beacon) {
        int index;
        int identifierNum;
        int lastIndex = -1;
        if (this.mMatchingBeaconTypeCodeEndOffset != null && this.mMatchingBeaconTypeCodeEndOffset > lastIndex) {
            lastIndex = this.mMatchingBeaconTypeCodeEndOffset;
        }
        if (this.mPowerEndOffset != null && this.mPowerEndOffset > lastIndex) {
            lastIndex = this.mPowerEndOffset;
        }
        for (identifierNum = 0; identifierNum < this.mIdentifierStartOffsets.size(); ++identifierNum) {
            if (this.mIdentifierEndOffsets.get(identifierNum) == null || this.mIdentifierEndOffsets.get(identifierNum) <= lastIndex) continue;
            lastIndex = this.mIdentifierEndOffsets.get(identifierNum);
        }
        for (identifierNum = 0; identifierNum < this.mDataEndOffsets.size(); ++identifierNum) {
            if (this.mDataEndOffsets.get(identifierNum) == null || this.mDataEndOffsets.get(identifierNum) <= lastIndex) continue;
            lastIndex = this.mDataEndOffsets.get(identifierNum);
        }
        byte[] advertisingBytes = new byte[lastIndex + 1 - 2];
        long beaconTypeCode = this.getMatchingBeaconTypeCode();
        for (index = this.mMatchingBeaconTypeCodeStartOffset.intValue(); index <= this.mMatchingBeaconTypeCodeEndOffset; ++index) {
            byte value;
            advertisingBytes[index - 2] = value = (byte)(this.getMatchingBeaconTypeCode() >> 8 * (this.mMatchingBeaconTypeCodeEndOffset - index) & 0xFFL);
        }
        for (int identifierNum2 = 0; identifierNum2 < this.mIdentifierStartOffsets.size(); ++identifierNum2) {
            byte[] identifierBytes = beacon.getIdentifier(identifierNum2).toByteArrayOfSpecifiedEndianness(this.mIdentifierLittleEndianFlags.get(identifierNum2));
            for (int index2 = this.mIdentifierStartOffsets.get(identifierNum2).intValue(); index2 <= this.mIdentifierEndOffsets.get(identifierNum2); ++index2) {
                int identifierByteIndex = this.mIdentifierEndOffsets.get(identifierNum2) - index2;
                advertisingBytes[index2 - 2] = identifierByteIndex < identifierBytes.length ? identifierBytes[this.mIdentifierEndOffsets.get(identifierNum2) - index2] : (byte)0;
            }
        }
        for (index = this.mPowerStartOffset.intValue(); index <= this.mPowerEndOffset; ++index) {
            advertisingBytes[index - 2] = (byte)(beacon.getTxPower() >> 8 * (index - this.mPowerStartOffset) & 0xFF);
        }
        for (int dataFieldNum = 0; dataFieldNum < this.mDataStartOffsets.size(); ++dataFieldNum) {
            long dataField = beacon.getDataFields().get(dataFieldNum);
            for (int index3 = this.mDataStartOffsets.get(dataFieldNum).intValue(); index3 <= this.mDataEndOffsets.get(dataFieldNum); ++index3) {
                int endianCorrectedIndex = index3;
                if (this.mDataLittleEndianFlags.get(dataFieldNum).booleanValue()) {
                    endianCorrectedIndex = this.mDataEndOffsets.get(dataFieldNum) - index3;
                }
                advertisingBytes[endianCorrectedIndex - 2] = (byte)(dataField >> 8 * (index3 - this.mDataStartOffsets.get(dataFieldNum)) & 0xFFL);
            }
        }
        return advertisingBytes;
    }

    public BeaconParser setMatchingBeaconTypeCode(Long typeCode) {
        this.mMatchingBeaconTypeCode = typeCode;
        return this;
    }

    public int getIdentifierByteCount(int identifierNum) {
        return this.mIdentifierEndOffsets.get(identifierNum) - this.mIdentifierStartOffsets.get(identifierNum) + 1;
    }

    protected static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    public static byte[] longToByteArray(long longValue, int length) {
        byte[] array = new byte[length];
        for (int i = 0; i < length; ++i) {
            long mask = 255L << (length - i - 1) * 8;
            long shift = (length - i - 1) * 8;
            long value = (longValue & mask) >> (int)shift;
            array[i] = (byte)value;
        }
        return array;
    }

    private boolean byteArraysMatch(byte[] array1, int offset1, byte[] array2, int offset2) {
        int minSize = array1.length > array2.length ? array2.length : array1.length;
        for (int i = 0; i < minSize; ++i) {
            if (array1[i + offset1] == array2[i + offset2]) continue;
            return false;
        }
        return true;
    }

    private String byteArrayToString(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < bytes.length; ++i) {
            sb.append(String.format("%02x", bytes[i]));
            sb.append(" ");
        }
        return sb.toString().trim();
    }

    private String byteArrayToFormattedString(byte[] byteBuffer, int startIndex, int endIndex, boolean littleEndian) {
        int length;
        int i;
        byte[] bytes = new byte[endIndex - startIndex + 1];
        if (littleEndian) {
            for (i = 0; i <= endIndex - startIndex; ++i) {
                bytes[i] = byteBuffer[startIndex + bytes.length - 1 - i];
            }
        } else {
            for (i = 0; i <= endIndex - startIndex; ++i) {
                bytes[i] = byteBuffer[startIndex + i];
            }
        }
        if ((length = endIndex - startIndex + 1) < 5) {
            long number = 0L;
            LogManager.d(TAG, "Byte array is size %s", bytes.length);
            for (int i2 = 0; i2 < bytes.length; ++i2) {
                LogManager.d(TAG, "index is %s", i2);
                long byteValue = bytes[bytes.length - i2 - 1] & 0xFF;
                long positionValue = (long)Math.pow(256.0, (double)i2 * 1.0);
                long calculatedValue = byteValue * positionValue;
                LogManager.d(TAG, "calculatedValue for position %s with positionValue %s and byteValue %s is %s", i2, positionValue, byteValue, calculatedValue);
                number += calculatedValue;
            }
            return Long.toString(number);
        }
        String hexString = BeaconParser.bytesToHex(bytes);
        if (bytes.length == 16) {
            StringBuilder sb = new StringBuilder();
            sb.append(hexString.substring(0, 8));
            sb.append("-");
            sb.append(hexString.substring(8, 12));
            sb.append("-");
            sb.append(hexString.substring(12, 16));
            sb.append("-");
            sb.append(hexString.substring(16, 20));
            sb.append("-");
            sb.append(hexString.substring(20, 32));
            return sb.toString();
        }
        return "0x" + hexString;
    }

    public static class BeaconLayoutException
    extends RuntimeException {
        public BeaconLayoutException(String s) {
            super(s);
        }
    }
}

