/*
 * Decompiled with CFR 0.152.
 */
package alluxio.client.file.cache.store;

import alluxio.client.file.cache.PageId;
import alluxio.client.file.cache.PageInfo;
import alluxio.client.file.cache.PageStore;
import alluxio.client.file.cache.store.LocalPageStoreOptions;
import alluxio.exception.PageNotFoundException;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NotThreadSafe
public class LocalPageStore
implements PageStore {
    private static final Logger LOG = LoggerFactory.getLogger(LocalPageStore.class);
    private final String mRoot;
    private final long mPageSize;
    private final long mCacheSize;
    private final int mFileBuckets;
    private final Pattern mPagePattern;

    public LocalPageStore(LocalPageStoreOptions options) {
        this.mRoot = options.getRootDir();
        this.mPageSize = options.getPageSize();
        this.mCacheSize = options.getCacheSize();
        this.mFileBuckets = options.getFileBuckets();
        Path rootDir = Paths.get(this.mRoot, new String[0]);
        this.mPagePattern = Pattern.compile(String.format("%s/%d/(\\d+)/([^/]+)/(\\d+)", Pattern.quote(rootDir.toString()), this.mPageSize));
    }

    @Override
    public void put(PageId pageId, byte[] page) throws IOException {
        Path p = this.getFilePath(pageId);
        if (!Files.exists(p, new LinkOption[0])) {
            Path parent = (Path)Preconditions.checkNotNull((Object)p.getParent(), (Object)"parent of cache file should not be null");
            Files.createDirectories(parent, new FileAttribute[0]);
            Files.createFile(p, new FileAttribute[0]);
        }
        try (FileOutputStream fos = new FileOutputStream(p.toFile(), false);){
            fos.write(page);
        }
    }

    @Override
    public ReadableByteChannel get(PageId pageId, int pageOffset) throws IOException, PageNotFoundException {
        Preconditions.checkArgument((pageOffset >= 0 ? 1 : 0) != 0, (Object)"page offset should be non-negative");
        Path p = this.getFilePath(pageId);
        if (!Files.exists(p, new LinkOption[0])) {
            throw new PageNotFoundException(p.toString());
        }
        File f = p.toFile();
        Preconditions.checkArgument(((long)pageOffset <= f.length() ? 1 : 0) != 0, (String)"page offset %s exceeded page size %s", (int)pageOffset, (long)f.length());
        FileInputStream fis = new FileInputStream(p.toFile());
        try {
            fis.skip(pageOffset);
            return fis.getChannel();
        }
        catch (Throwable t) {
            fis.close();
            throw t;
        }
    }

    @Override
    public void delete(PageId pageId, long pageSize) throws IOException, PageNotFoundException {
        Path p = this.getFilePath(pageId);
        if (!Files.exists(p, new LinkOption[0])) {
            throw new PageNotFoundException(p.toString());
        }
        Files.delete(p);
        Path parent = (Path)Preconditions.checkNotNull((Object)p.getParent(), (Object)"parent of cache file should not be null");
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(parent);){
            if (!stream.iterator().hasNext()) {
                Files.delete(parent);
            }
        }
    }

    private Path getFilePath(PageId pageId) {
        return Paths.get(this.mRoot, Long.toString(this.mPageSize), this.getFileBucket(pageId.getFileId()), pageId.getFileId(), Long.toString(pageId.getPageIndex()));
    }

    private String getFileBucket(String fileId) {
        return Integer.toString(Math.floorMod(fileId.hashCode(), this.mFileBuckets));
    }

    @Nullable
    private PageId getPageId(Path path) {
        Matcher matcher = this.mPagePattern.matcher(path.toString());
        if (!matcher.matches()) {
            return null;
        }
        try {
            String fileBucket = (String)Preconditions.checkNotNull((Object)matcher.group(1));
            String fileId = (String)Preconditions.checkNotNull((Object)matcher.group(2));
            if (!fileBucket.equals(this.getFileBucket(fileId))) {
                return null;
            }
            String fileName = (String)Preconditions.checkNotNull((Object)matcher.group(3));
            long pageIndex = Long.parseLong(fileName);
            return new PageId(fileId, pageIndex);
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    @Nullable
    private PageInfo getPageInfo(Path path) {
        long pageSize;
        PageId pageId = this.getPageId(path);
        if (pageId == null) {
            LOG.error("Unrecognized page file" + path);
            return null;
        }
        try {
            pageSize = Files.size(path);
        }
        catch (IOException e) {
            LOG.error("Failed to get file size for " + path, (Throwable)e);
            return null;
        }
        return new PageInfo(pageId, pageSize);
    }

    @Override
    public void close() {
    }

    @Override
    public Stream<PageInfo> getPages() throws IOException {
        Path rootDir = Paths.get(this.mRoot, new String[0]);
        return Files.walk(rootDir, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(this::getPageInfo);
    }

    @Override
    public long getCacheSize() {
        return this.mCacheSize;
    }
}

