package org.airbloc.ui.intro;

import android.arch.lifecycle.ViewModelProviders;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.text.Editable;
import android.text.Html;
import android.text.TextWatcher;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.ScrollView;
import android.widget.TextView;

import org.airbloc.R;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.ui.ConsentViewModel;
import org.airbloc.ui.utils.KeyboardVisibilityUtils;
import org.airbloc.ui.utils.LiveDataUtils;

import static org.airbloc.ui.AirblocUiConstants.AIRBLOC_TRACKER_URL;

public class IntroFragment extends Fragment {
    private ConsentViewModel viewModel;

    private TextView emailHintText;
    private EditText emailInput;

    private KeyboardVisibilityUtils keyboardUtils;

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_intro, container, false);
        viewModel = ViewModelProviders.of(getActivity()).get(ConsentViewModel.class);

        TextView title = view.findViewById(R.id.org_airbloc__text_intro_title);
        title.setText(Html.fromHtml(String.format(title.getText().toString(),
                getString(R.string.org_airbloc_sdk_consent_app_title))));

        View learnMoreButton = view.findViewById(R.id.container_learn_airbloc_tracker);
        learnMoreButton.setOnClickListener(v -> viewModel.openUrl(AIRBLOC_TRACKER_URL));

        View beginButton = view.findViewById(R.id.button_begin);
        beginButton.setEnabled(false);
        beginButton.setBackgroundResource(R.color.grey300);
        beginButton.setOnClickListener(v -> onBeginClick());

        emailHintText = view.findViewById(R.id.text_consent_hint);
        emailInput = view.findViewById(R.id.edit_consent_email);
        LiveDataUtils.bind(this, viewModel.getEmail(), emailInput::setText);

        emailInput.addTextChangedListener(new TextWatcher() {
            @Override
            public void afterTextChanged(Editable s) {
                if (s.length() == 0) {
                    beginButton.setEnabled(false);
                    beginButton.setBackgroundResource(R.color.grey300);
                } else {
                    beginButton.setEnabled(true);
                    beginButton.setBackgroundResource(R.drawable.btn_primary_bottom);
                }
                emailHintText.setText(R.string.input_email_desc);
                emailHintText.setTextColor(getResources().getColor(R.color.grey600));
                emailInput.setBackgroundResource(R.drawable.bg_edit);
            }

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {}

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {}
        });

        emailInput.setOnEditorActionListener((v, action, event) -> {
            if (action == EditorInfo.IME_ACTION_NEXT) {
                onBeginClick();
                return true;
            }
            return false;
        });

        // scroll to bottom when user click the EditText
        ScrollView scroll = view.findViewById(R.id.scroll_intro);
        keyboardUtils = new KeyboardVisibilityUtils(getActivity());
        keyboardUtils.adjustScrollOnKeyboardInput(scroll);

        ViewGroup scrollContent = view.findViewById(R.id.container_intro);
        view.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                // let them scroll to top on keyboard shown state
                AirblocLogger.d("scroll.getMeasuredHeight(): %d", scroll.getMeasuredHeight());
                scrollContent.setMinimumHeight(scroll.getMeasuredHeight());

                view.getViewTreeObserver().removeOnGlobalLayoutListener(this);
            }
        });
        return view;
    }

    private void onBeginClick() {
        String email = emailInput.getText().toString();

        boolean wasEmailValid = viewModel.finishIntro(email);
        if (!wasEmailValid) {
            emailHintText.setText(R.string.input_email_error);
            emailHintText.setTextColor(getResources().getColor(R.color.error));
            emailInput.setBackgroundResource(R.drawable.bg_edit_error);
        } else {
            keyboardUtils.hideKeyboardFrom(emailInput);
        }
    }
}
