package org.airbloc.ui;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.MediatorLiveData;
import android.arch.lifecycle.MutableLiveData;
import android.arch.lifecycle.ViewModel;
import android.util.SparseArray;

import org.airbloc.sdk.Airbloc;
import org.airbloc.sdk.appinfo.AppInfo;
import org.airbloc.sdk.appinfo.DataTypeCategory;
import org.airbloc.sdk.consent.Consents;
import org.airbloc.sdk.internal.AirblocExecutors;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.sdk.user.UserManager;
import org.airbloc.ui.utils.SingleLiveEvent;

public class ConsentViewModel extends ViewModel {

    private Airbloc airbloc = Airbloc.getInstance();

    // email format regex from RFC 5322
    private static final String EMAIL_FORMAT = "(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|\"(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21\\x23-\\x5b\\x5d-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])*\")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x21-\\x5a\\x53-\\x7f]|\\\\[\\x01-\\x09\\x0b\\x0c\\x0e-\\x7f])+)\\])";

    // loaded from repository
    private AppInfo appInfo;

    private MutableLiveData<String> email = new MutableLiveData<>();
    private MutableLiveData<Boolean> thirdPartyAgreement = new MutableLiveData<>();
    private MutableLiveData<Boolean> serviceAgreement = new MutableLiveData<>();

    // calculated from consents and third party agreement
    private MediatorLiveData<Integer> ablAmount = new MediatorLiveData<>();

    private Consents consents = new Consents();
    private SparseArray<MutableLiveData<Boolean>> liveConsents = new SparseArray<>();

    // screen transition events
    private SingleLiveEvent<Throwable> loadFinishEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<Void> introStartEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<Void> introFinishEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<Boolean> consentFinishEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<Void> finishEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<Void> earnMoreAblEvent = new SingleLiveEvent<>();
    private SingleLiveEvent<String> openUrlEvent = new SingleLiveEvent<>();

    public ConsentViewModel() {
        thirdPartyAgreement.setValue(false);
        serviceAgreement.setValue(false);

        ablAmount.addSource(thirdPartyAgreement, v -> calculateAblAmount());

        // fill default email value for user convenience if set
        UserManager user = airbloc.getCurrentUser();
        email.setValue(user.hasSignedIn() ? user.getId() : "");

        // load app info and consent info
        airbloc.loadAppInfo()
                .thenAccept(loadedAppInfo -> appInfo = loadedAppInfo)
                .thenComposeAsync(v -> airbloc.getConsents().load())
                .thenAcceptAsync(loadedConsents -> {
                    loadFinishEvent.postValue(null);

                    if (loadedConsents != null && loadedConsents.asList().size() > 0) {
                        loadConsents(loadedConsents);
                        consentFinishEvent.call();

                    } else {
                        initialize();
                        introStartEvent.call();
                    }
                }, AirblocExecutors.getInstance().getUiExecutor())
                .exceptionally(err -> {
                    loadFinishEvent.postValue(err);
                    return null;
                });
    }

    private void loadConsents(Consents loadedConsents) {
        for (Consents.Consent consent : loadedConsents.asList()) {
            consents.consent(consent.categoryId, consent.consented);
            getConsentOf(consent.categoryId).setValue(consent.consented);
        }
        serviceAgreement.setValue(loadedConsents.isAgreed("service_policy"));
        thirdPartyAgreement.setValue(loadedConsents.isAgreed("third_party"));
        AirblocLogger.i("Loaded existing consents");
    }

    private void initialize() {
        calculateAblAmount();
    }

    private void calculateAblAmount() {
        int amount = 0;
        for (Consents.Consent consent : consents.asList()) {
            if (consent.consented) {
                DataTypeCategory category = appInfo.getCategoryById(consent.categoryId);
                amount += category.rewardAmount;
            }
        }

        if (thirdPartyAgreement.getValue() == Boolean.TRUE) {
            amount = amount * (int)(appInfo.getIncentiveRate(AppInfo.THIRD_PARTY_INCENTIVE));
        }
        ablAmount.setValue(amount);
    }

    public boolean finishIntro(String email) {
        if (!email.matches(EMAIL_FORMAT)) {
            return false;
        }
        this.email.setValue(email);
        airbloc.signIn(email);

        // load existing consents if exists
        airbloc.getConsents().load().thenAccept(this::loadConsents);

        startModifyConsent();
        return true;
    }

    public void startModifyConsent() {
        introFinishEvent.call();
    }

    public boolean toggleConsent(DataTypeCategory category) {
        boolean oldConsentValue = consents.isConsented(category.id);

        consents.consent(category.id, !oldConsentValue);
        getConsentOf(category).postValue(!oldConsentValue);

        return !oldConsentValue;
    }

    public void toggleServiceAgreement() {
        serviceAgreement.postValue(serviceAgreement.getValue() != Boolean.TRUE);
    }

    public void toggleThirdPartyAgreement() {
        thirdPartyAgreement.postValue(thirdPartyAgreement.getValue() != Boolean.TRUE);
    }

    public void finishConsent() {
        consentFinishEvent.call();
        saveConsents();
    }

    public void saveConsents() {
        airbloc.getConsents()
                .setAgreement("third_party", thirdPartyAgreement.getValue())
                .setAgreement("service_policy", serviceAgreement.getValue())
                .consentAll(consents);
    }

    public void goEarnMoreAbl() {
        earnMoreAblEvent.call();
    }

    public void openUrl(String url) {
        openUrlEvent.setValue(url);
    }

    public void finish() {
        finishEvent.call();
    }

    public MutableLiveData<Boolean> getConsentOf(DataTypeCategory category) {
        return getConsentOf(category.id);
    }

    public MutableLiveData<Boolean> getConsentOf(int categoryId) {
        MutableLiveData<Boolean> consent = liveConsents.get(categoryId);
        if (consent == null) {
            consent = new MutableLiveData<>();
            liveConsents.put(categoryId, consent);

            // should track new consents for ABL calculation
            ablAmount.addSource(consent, v -> calculateAblAmount());
        }
        return consent;
    }

    public LiveData<String> getEmail() {
        return email;
    }

    public LiveData<Integer> getAblAmount() {
        return ablAmount;
    }

    public LiveData<Boolean> getThirdPartyAgreement() {
        return thirdPartyAgreement;
    }

    public LiveData<Boolean> getServiceAgreement() {
        return serviceAgreement;
    }

    public LiveData<Throwable> getLoadFinishEvent() {
        return loadFinishEvent;
    }

    public LiveData<Void> getIntroStartEvent() {
        return introStartEvent;
    }

    public LiveData<Void> getIntroFinishEvent() {
        return introFinishEvent;
    }

    public LiveData<Boolean> getConsentFinishEvent() {
        return consentFinishEvent;
    }

    public LiveData<Void> getFinishEvent() {
        return finishEvent;
    }

    public LiveData<Void> getEarnMoreAblEvent() {
        return earnMoreAblEvent;
    }

    public LiveData<String> getOpenUrlEvent() {
        return openUrlEvent;
    }

    public AppInfo getAppInfo() {
        return appInfo;
    }
}
