package org.airbloc.sdk.internal;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.concurrent.TimeUnit;

public class AirblocLocalStorage {

    private static final String PREFS_NAME = "airbloc_data";

    private SharedPreferences prefs;

    public AirblocLocalStorage(Context context) {
        this(context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE));
    }

    AirblocLocalStorage(SharedPreferences prefs) {
        this.prefs = prefs;
    }

    /**
     * @return {@code true} if the key exists in the local storage.
     */
    public boolean contains(String key) {
        // reason why not using SharedPreferences#contains(): we need to consider TTL
        return get(key) != null;
    }

    /**
     * Store data permanently.
     *
     * @param key Key of the value. Value will be overwritten when the key collides.
     * @param value Any entity class implementing {@link JsonMappable#toJson()}
     */
    public void put(String key, JsonMappable value) {
        put(key, value, 0, null);
    }

    /**
     * Store data for a given TTL duration.
     *
     * @param key Key of the value. Value will be overwritten when the key collides.
     * @param value Any entity class implementing {@link JsonMappable#toJson()}
     * @param timeToLive How long the data should be stored. Data will be removed after TTL.
     * @param timeUnit {@link TimeUnit}
     */
    public void put(String key, JsonMappable value, long timeToLive, TimeUnit timeUnit) {
        JsonData data = new JsonData()
                .put("createdAt", System.currentTimeMillis())
                .put("payload", value);

        if (timeUnit != null) {
            data.put("timeToLive", timeUnit.toMillis(timeToLive));
        }
        prefs.edit()
                .putString(key, data.toJson().toString())
                .apply();
    }

    public JsonData get(String key) {
        String rawData = prefs.getString(key, null);
        if (rawData == null) {
            return null;
        }

        JsonData data;
        try {
            data = JsonData.fromJSON(new JSONObject(rawData));
        } catch (JSONException ignored) {
            return null;
        }

        if (data.has("timeToLive")) {
            long createdAt = data.getLong("createdAt");
            long ttl = data.getLong("timeToLive");

            // check expire
            if (createdAt + ttl > System.currentTimeMillis()) {
                remove(key);
                return null;
            }
        }
        return data.get("payload");
    }

    public void remove(String key) {
        prefs.edit()
                .remove(key)
                .apply();
    }

    @SuppressLint("ApplySharedPref")
    public boolean clear() {
        return prefs.edit().clear().commit();
    }
}

