package org.airbloc.sdk.internal;

import org.airbloc.sdk.internal.tasks.UiThreadExecutor;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class AirblocExecutors {

    // How many tasks can be run at once?
    private static final int MAX_BACKGROUND_THREADS = 8;

    private static ExecutorService concurrentExecutor;
    public static ExecutorService serialExecutor;
    public static UiThreadExecutor uiExecutor;

    private static ScheduledExecutorService debounceScheduler;
    private static ConcurrentHashMap<Object, Future<?>> delayedMap = new ConcurrentHashMap<>();

    private static AirblocExecutors instance;

    private AirblocExecutors() {
        concurrentExecutor = Executors.newFixedThreadPool(MAX_BACKGROUND_THREADS);
        serialExecutor = Executors.newSingleThreadExecutor();
        debounceScheduler = Executors.newSingleThreadScheduledExecutor();
    }

    public static AirblocExecutors getInstance() {
        if (instance == null) {
            instance = new AirblocExecutors();
        }
        return instance;
    }

    public Executor getConcurrentExecutor() {
        return concurrentExecutor;
    }

    public Executor getSerialExecutor() {
        return serialExecutor;
    }

    public Executor getUiExecutor() {
        if (uiExecutor == null) {
            uiExecutor = UiThreadExecutor.getInstance();
        }
        return uiExecutor;
    }

    /**
     * Debounces {@code callable} by {@code delay}, i.e., schedules it to be executed after {@code delay},
     * or cancels its execution if the method is called with the same key within the {@code delay} again.
     */
    public void debounce(Object key, Runnable runnable, long delay, TimeUnit unit) {
        Future<?> prev = delayedMap.put(key, debounceScheduler.schedule(() -> {
            try {
                runnable.run();
            } finally {
                delayedMap.remove(key);
            }
        }, delay, unit));
        if (prev != null) {
            prev.cancel(true);
        }
    }

    public void shutdownDebounces() {
        debounceScheduler.shutdownNow();
    }
}