package org.airbloc.sdk.appinfo;

import android.content.Context;

import org.airbloc.sdk.AirblocConfig;
import org.airbloc.sdk.internal.AirblocLocalStorage;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.sdk.internal.networking.AirblocHttpClient;
import org.airbloc.sdk.internal.networking.AirblocHttpRequest;
import org.airbloc.sdk.internal.networking.AirblocHttpResponse;

import java.util.concurrent.TimeUnit;

import java8.util.concurrent.CompletableFuture;

public class AppInfoRepositoryImpl implements AppInfoRepository {

    // caches app info into the local
    private static final String APP_INFO_CACHE_KEY = "app_info";

    // 5 hours TTL is given
    private static final int APP_INFO_CACHE_TTL_HOURS = 120;

    private AirblocHttpClient httpClient;
    private AirblocLocalStorage storage;

    private String appName;

    public AppInfoRepositoryImpl(Context context, AirblocHttpClient httpClient, AirblocConfig config) {
        this.httpClient = httpClient;
        this.appName = config.getAppName();
        this.storage = new AirblocLocalStorage(context);
    }

    @Override
    public CompletableFuture<AppInfo> get() {
        if (storage.contains(APP_INFO_CACHE_KEY)) {
            AirblocLogger.i("App info already fetched. Using cache...");
            return fetchFromCache();
        }
        return fetchFromRemote().thenApply(appInfo -> {
            storage.put(APP_INFO_CACHE_KEY, appInfo, APP_INFO_CACHE_TTL_HOURS, TimeUnit.HOURS);
            return appInfo;
        });
    }

    private CompletableFuture<AppInfo> fetchFromRemote() {
        return httpClient.call(new AirblocHttpRequest("GET", "/apps/" + appName + "/consent"))
                .thenApply(AirblocHttpResponse::getBody)
                .thenApply(AppInfo::fromJson);
    }

    private CompletableFuture<AppInfo> fetchFromCache() {
        return CompletableFuture
                .supplyAsync(() -> storage.get(APP_INFO_CACHE_KEY))
                .thenApply(AppInfo::fromJson);
    }
}
