package org.airbloc.ui.utils;

import android.app.Activity;
import android.arch.lifecycle.Lifecycle;
import android.arch.lifecycle.LifecycleObserver;
import android.arch.lifecycle.LifecycleOwner;
import android.arch.lifecycle.OnLifecycleEvent;
import android.graphics.Rect;
import android.support.v4.app.FragmentActivity;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.inputmethod.InputMethodManager;
import android.widget.ScrollView;

import java.util.LinkedList;
import java.util.List;

public class KeyboardVisibilityUtils {

    // Threshold for minimal keyboard height.
    private static final int MIN_KEYBOARD_HEIGHT_PX = 150;

    // Top-level window decor view.
    private View decorView;

    // for calculating keyboard size
    private final Rect windowVisibleDisplayFrame = new Rect();
    private int lastVisibleDecorViewHeight;

    // listeners
    private List<ScrollView> scrollViews = new LinkedList<>();

    public KeyboardVisibilityUtils(FragmentActivity activity) {
        decorView = activity.getWindow().getDecorView();

        final ViewTreeObserver.OnGlobalLayoutListener layoutListener = this::recalculateKeyboardSize;
        decorView.getViewTreeObserver().addOnGlobalLayoutListener(layoutListener);

        activity.getLifecycle().addObserver(new LifecycleObserver() {
            @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
            void onDestroy(LifecycleOwner owner) {
                decorView.getViewTreeObserver().removeOnGlobalLayoutListener(layoutListener);
            }
        });
    }

    private void recalculateKeyboardSize() {
        // Retrieve visible rectangle inside window.
        decorView.getWindowVisibleDisplayFrame(windowVisibleDisplayFrame);
        int visibleDecorViewHeight = windowVisibleDisplayFrame.height();

        // Decide whether keyboard is visible from changing decor view height.
        if (lastVisibleDecorViewHeight != 0) {

            if (lastVisibleDecorViewHeight > visibleDecorViewHeight + MIN_KEYBOARD_HEIGHT_PX) {
                // Calculate current keyboard height
                // (this includes also navigation bar height when in fullscreen mode).
                int currentKeyboardHeight = decorView.getHeight() - windowVisibleDisplayFrame.bottom;

                for (ScrollView scroll : scrollViews) {
                    scroll.smoothScrollTo(
                            scroll.getScrollX(),
                            scroll.getScrollY() + currentKeyboardHeight);
                }
            }
        }
        // Save current decor view height for the next call.
        lastVisibleDecorViewHeight = visibleDecorViewHeight;
    }

    public void adjustScrollOnKeyboardInput(ScrollView scroll) {
        scrollViews.add(scroll);
    }

    public void hideKeyboardFrom(View view) {
        InputMethodManager imm = (InputMethodManager) view
                .getContext().getSystemService(Activity.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
    }
}
