package org.airbloc.ui.consent;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.MediatorLiveData;
import android.arch.lifecycle.MutableLiveData;
import android.arch.lifecycle.Observer;
import android.arch.lifecycle.Transformations;
import android.arch.lifecycle.ViewModelProviders;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.app.FragmentStatePagerAdapter;
import android.support.v4.view.ViewPager;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;

import com.robinhood.ticker.TickerView;

import org.airbloc.R;
import org.airbloc.sdk.appinfo.DataTypeCategory;
import org.airbloc.sdk.internal.Functional;
import org.airbloc.ui.ConsentViewModel;
import org.airbloc.ui.utils.LiveDataUtils;

import java.util.Collection;
import java.util.List;

public class ConsentFragment extends Fragment {
    private ConsentViewModel viewModel;
    private MutableLiveData<Integer> currentPageIndex = new MutableLiveData<>();

    private ViewPager pager;

    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        View root = inflater.inflate(R.layout.fragment_consent, container, false);
        viewModel = ViewModelProviders.of(getActivity()).get(ConsentViewModel.class);

        Collection<DataTypeCategory> dataTypeCategories = viewModel.getAppInfo().categories.values();

        pager = root.findViewById(R.id.pager_data_types);
        setupPager(dataTypeCategories);

        LinearLayout indicator = root.findViewById(R.id.container_page_indicators);
        setupIndicator(indicator, dataTypeCategories);

        TickerView ablAmountTicker = root.findViewById(R.id.ticker_abl_amount);
        LiveDataUtils.bind(this,
                Transformations.map(viewModel.getAblAmount(), String::valueOf),
                ablAmountTicker::setText);

        return root;
    }

    private void setupPager(Collection<DataTypeCategory> dataTypeCategories) {
        pager.setAdapter(new DataTypePagerAdapter(getChildFragmentManager(), dataTypeCategories));

        // update page index state
        currentPageIndex.setValue(0);
        pager.addOnPageChangeListener(new ViewPager.OnPageChangeListener() {
            @Override
            public void onPageSelected(int newIndex) {
                currentPageIndex.setValue(newIndex);
            }

            @Override
            public void onPageScrollStateChanged(int i) {
            }

            @Override
            public void onPageScrolled(int i, float v, int i1) {
            }
        });

        // show a peek preview of next card
        int padding = getResources().getDimensionPixelSize(R.dimen.consent_card_page_padding);
        pager.setClipToPadding(false);
        pager.setPadding(padding, 0, padding, 0);
        pager.setPageMargin(getResources().getDimensionPixelSize(R.dimen.consent_card_page_margin));
        pager.setOffscreenPageLimit(dataTypeCategories.size() + 1);
    }

    private void setupIndicator(LinearLayout container, Collection<DataTypeCategory> dataTypeCategories) {
        int index = 0;
        int lastIndex = dataTypeCategories.size();

        for (DataTypeCategory category : dataTypeCategories) {
            LiveData<Boolean> consentStatus = viewModel.getConsentOf(category);

            container.addView(createIndicatorFor(consentStatus, index));
            index += 1;
        }

        // make it checked when user agrees to service agreement
        container.addView(
                createIndicatorFor(viewModel.getServiceAgreement(), lastIndex));
    }

    private View createIndicatorFor(LiveData<Boolean> source, int index) {
        View indicator = View.inflate(getContext(), R.layout.indicator_consent, null);
        indicator.setOnClickListener(v -> pager.setCurrentItem(index, true));

        // image should be changed according to the consent check state
        MediatorLiveData<Integer> checkImageState = new MediatorLiveData<>();
        checkImageState.setValue(R.drawable.indicator_normal);

        Observer updateIndicatorImage = (v) -> {
            if (currentPageIndex.getValue() == index) {
                checkImageState.setValue(source.getValue() == Boolean.TRUE
                        ? R.drawable.indicator_selected_checked
                        : R.drawable.indicator_selected);
            } else {
                checkImageState.setValue(source.getValue() == Boolean.TRUE
                        ? R.drawable.indicator_checked
                        : R.drawable.indicator_normal);
            }
        };
        checkImageState.addSource(source, updateIndicatorImage);
        checkImageState.addSource(currentPageIndex, updateIndicatorImage);

        ImageView indicatorImage = indicator.findViewById(R.id.indicator);
        LiveDataUtils.bind(this, checkImageState, indicatorImage::setImageResource);
        return indicator;
    }

    class DataTypePagerAdapter extends FragmentStatePagerAdapter {
        private List<DataTypeCardFragment> fragments;
        private AgreementCardFragment agreementCardFragment;

        DataTypePagerAdapter(FragmentManager fm, Collection<DataTypeCategory> dataTypeCategories) {
            super(fm);
            agreementCardFragment = AgreementCardFragment.newInstance();
            fragments = Functional.map(dataTypeCategories, DataTypeCardFragment::newInstance);
            notifyDataSetChanged();
        }

        @Override
        public Fragment getItem(int i) {
            if (i == fragments.size()) {
                // last item
                return agreementCardFragment;
            }
            return fragments.get(i);
        }

        @Override
        public int getCount() {
            // with counting AgreementCardFragment
            return fragments.size() + 1;
        }
    }
}
