package org.airbloc.ui;

import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentManager;
import android.support.v7.app.AppCompatActivity;

import org.airbloc.R;
import org.airbloc.sdk.internal.logger.AirblocLogger;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import java8.util.StringJoiner;
import java8.util.function.Supplier;
import java8.util.stream.StreamSupport;

/**
 * A parent activity of navigation {@link Fragment}.
 * Delivers similar functionality with Jetpack Navigation.
 */
public class AirblocNavigationActivity extends AppCompatActivity {

    private FragmentManager fm;
    private FragmentInfo currentFragment;
    private Map<String, FragmentInfo> fragmentInfos = new HashMap<>();

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        fm = getSupportFragmentManager();
    }

    @Override
    public void onBackPressed() {
        if (isRootFragment()) {
            finish();
            return;
        }
        super.onBackPressed();

        Fragment currentFragment = fm.findFragmentById(R.id.content_main);
        if (currentFragment == null) {
            finish();
            return;
        }
        FragmentInfo fragmentInfo = fragmentInfos.get(currentFragment.getClass().getName());
        setCurrentFragment(fragmentInfo);
    }

    public void navigateTo(FragmentInfo fragmentInfo, boolean recreate) {
        Fragment fragment = fm.findFragmentByTag(fragmentInfo.name);
        List<String> backStack = getBackStackEntries();

        if (recreate && backStack.contains(fragmentInfo.name)) {
            // inclusive pop
            fm.popBackStackImmediate(fragmentInfo.name, 1);

            fragment = fragmentInfo.newInstance();
            fm.beginTransaction()
                    .add(R.id.content_main, fragment, fragmentInfo.name)
                    .addToBackStack(fragmentInfo.name)
                    .commit();

        } else if (backStack.contains(fragmentInfo.name)) {
            fm.popBackStackImmediate(fragmentInfo.name, 0);

        } else if (fragment != null) {
            AirblocLogger.i("Existing fragment %s", fragmentInfo.name);
            fm.beginTransaction()
                    .replace(R.id.content_main, fragment, fragmentInfo.name)
                    .commit();

        } else {
            AirblocLogger.i("New fragment %s", fragmentInfo.name);
            fragment = fragmentInfo.newInstance();
            fm.beginTransaction()
                    .add(R.id.content_main, fragment, fragmentInfo.name)
                    .addToBackStack(fragmentInfo.name)
                    .commit();
        }
        setCurrentFragment(fragmentInfo);
    }

    public void navigateTo(FragmentInfo fragmentInfo) {
        navigateTo(fragmentInfo, false);
    }

    protected void setCurrentFragment(FragmentInfo fragment) {
        currentFragment = fragment;
        if (!fragmentInfos.containsKey(fragment.name)) {
            fragmentInfos.put(fragment.name, fragment);
        }

        AirblocLogger.d("Current Fragment: %s (backstack = %s)",
                fragment.name,
                join(getBackStackEntries()));
    }

    public boolean isRootFragment() {
        List<String> backStack = getBackStackEntries();
        return backStack.isEmpty() ||
                (backStack.size() == 1 && backStack.get(0).equals(currentFragment.name));
    }

    private List<String> getBackStackEntries() {
        List<String> fragmentNames = new ArrayList<>(fm.getBackStackEntryCount());
        for (int i = 0; i < fm.getBackStackEntryCount(); i++) {
            String name = fm.getBackStackEntryAt(i).getName();
            fragmentNames.add(String.valueOf(name));
        }
        return fragmentNames;
    }

    private String join(List<String> strings) {
        StringJoiner joiner = new StringJoiner(", ");
        StreamSupport.stream(strings).forEach(joiner::add);
        return joiner.toString();
    }

    public FragmentInfo getCurrentFragment() {
        return currentFragment;
    }

    public static class FragmentInfo {
        String name;
        Supplier<Fragment> constructor;

        public FragmentInfo(Class<? extends Fragment> clazz, Supplier<Fragment> constructor) {
            this.name = clazz.getName();
            this.constructor = constructor;
        }

        Fragment newInstance() {
            return constructor.get();
        }
    }
}
