package org.airbloc.sdk.user;

import android.content.Context;
import android.support.annotation.NonNull;

import org.airbloc.sdk.consent.ConsentManager;
import org.airbloc.sdk.internal.AirblocExecutors;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.sdk.internal.networking.AirblocHttpClient;

import java.util.concurrent.TimeUnit;

import java8.util.concurrent.CompletableFuture;

public class UserManager {

    // user is anonymous before calling Airbloc.changeUser
    private static final String ANONYMOUS_ID = "<anonymous>";

    // user data can be collected after 5s
    private static final long USER_DATA_COLLECTION_PERIOD = 5000;

    // default properties
    private static final String KEY_EMAIL = "email";
    private static final String KEY_TELEPHONE = "phoneNumber";

    private AirblocExecutors executors = AirblocExecutors.getInstance();

    private UserRepository userRepository;
    private User currentUser;

    public UserManager(Context context, AirblocHttpClient httpClient) {
        this(ANONYMOUS_ID, new UserRepositoryImpl(context, httpClient));
    }

    UserManager(String externalId, UserRepository repository) {
        currentUser = new User(externalId);
        userRepository = repository;
    }

    public CompletableFuture<Void> load() {
        return userRepository.get()
                .thenAccept(user -> {
                    if (user != null) {
                        currentUser = user;
                        AirblocLogger.d("Existing user session: %s (%s)", user.getExternalId(), user.getAccountId());
                    } else {
                        AirblocLogger.d("No user found.");
                    }
                });
    }

    public void signIn(String id) {
        if (!hasSignedIn()) {
            AirblocLogger.d("Signing in with %s", id);
            signUp(id);
        }
    }

    public void signUp(String id) {
        userRepository.signUp(id)
                .exceptionally(error -> {
                    AirblocLogger.e("Failed to sign up", error);
                    return null;
                })
                .thenAccept(user -> {
                    if (user == null) {
                        AirblocLogger.d("signUp: No user available", id);
                        return;
                    }
                    currentUser = user;
                });
    }

    public UserManager setEmail(String value) {
        currentUser.setAttribute(KEY_EMAIL, value);
        requestUpdate();
        return this;
    }

    public UserManager setPhoneNumber(String value) {
        currentUser.setAttribute(KEY_TELEPHONE, value);
        requestUpdate();
        return this;
    }

    public <T> UserManager setAttribute(@NonNull String key, T value) {
        currentUser.setAttribute(key, value);
        requestUpdate();
        return this;
    }

    public String getId() {
        return currentUser.getExternalId();
    }

    public String getAccountId() {
        return currentUser.getAccountId();
    }

    public User getUser() {
        return currentUser;
    }

    public boolean hasSignedIn() {
        return currentUser.getAccountId() != null;
    }

    private void requestUpdate() {
        executors.debounce(ConsentManager.class, () -> {
            // TODO: integrate attribute update API

        }, USER_DATA_COLLECTION_PERIOD, TimeUnit.MILLISECONDS);
    }
}
