package org.airbloc.sdk.internal.networking;

import android.support.annotation.WorkerThread;
import android.util.Log;

import org.airbloc.sdk.internal.AirblocExecutors;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.sdk.internal.networking.AirblocHttpError.Reason;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;

import java8.util.concurrent.CompletableFuture;

/**
 * Sends HTTP requests and get response.
 * @author Hyojun Kim
 */
public class AirblocHttpClient {
    private AirblocExecutors executor = AirblocExecutors.getInstance();

    private String endpoint;
    private String appToken;

    public AirblocHttpClient(String endpoint, String appToken) {
        this.endpoint = endpoint;
        this.appToken = appToken;
    }

    /**
     * Sends HTTP request asynchronously. The results will be given as {@link CompletableFuture}.
     * Note that the task will be executed on background; if you need to handle the result in UI,
     * you should pass the result to main looper using
     *
     * {@code resultFuture.thenApplyAsync(Function.identity(), executor.getUiExecutor())}
     */
    public CompletableFuture<AirblocHttpResponse> call(AirblocHttpRequest request) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                AirblocHttpResponse response = callSync(request);

                if (!response.isSuccessful()) {
                    AirblocHttpError error = AirblocHttpError.parse(response);
                    AirblocLogger.v(" └ Error => HTTP %d, %s", response.getStatus(), error.getMessage());
                    throw error;
                }
                AirblocLogger.v("└ Response => HTTP %d, %s", response.getStatus(), response.getRawBody());
                return response;

            } catch (UnknownHostException | ConnectException e) {
                AirblocLogger.w("Request failed - Internet not connected.");
                throw new AirblocHttpError(Reason.NoNetwork, e.getMessage());

            } catch (Exception e) {
                AirblocLogger.e("Request failed - " + request.body, e);
                throw new AirblocHttpError(Reason.RequestFailed, e.getMessage());
            }
        }, executor.getConcurrentExecutor());
    }

    @WorkerThread
    private AirblocHttpResponse callSync(AirblocHttpRequest request) throws Exception {
        HttpURLConnection conn = null;

        try {
            URL url = urlOnEndpoint(request.url);
            conn = (HttpURLConnection) url.openConnection();
            AirblocLogger.v("Request => %s %s %s", request.method, url.toString(), request.body);

            conn.setRequestMethod(request.method);
            conn.setRequestProperty("Authorization", "AIRBLOC-SDK-TOKEN " + appToken);
            conn.setRequestProperty("Accept", "application/json");
            conn.setRequestProperty("Accept-Encoding", "utf-8");
            conn.setRequestProperty("Content-Type", "application/json");

            if (request.body != null) {
                conn.setDoOutput(true);
                conn.setChunkedStreamingMode(0);

                try (OutputStream out = new BufferedOutputStream(conn.getOutputStream())) {
                    out.write(request.body.toString().getBytes("utf-8"));
                    out.flush();
                }
            }

            InputStream stream;
            int code = conn.getResponseCode();
            if (code >= 400 && conn.getErrorStream() != null) stream = conn.getErrorStream();
            else stream = conn.getInputStream();

            StringBuilder body = new StringBuilder();
            try (BufferedReader in = new BufferedReader(new InputStreamReader(stream))) {
                for (String line; (line = in.readLine()) != null; ) {
                    body.append(line).append('\n');
                }
            }
            return new AirblocHttpResponse(code, body.toString());

        } catch (OutOfMemoryError err) {
            // calling OutputStream#flush() often occurs OOM
            Log.wtf("Airbloc", "Out of memory error occurred.", err);
            throw err;

        } finally {
            if (conn != null) conn.disconnect();
        }
    }


    private URL urlOnEndpoint(String url) throws MalformedURLException {
        String trimmedEndpoint = endpoint.endsWith("/")
                ? endpoint.substring(0, endpoint.length() - 1) : endpoint;
        String trimmedUrl = url.startsWith("/") ? url.substring(1) : url;
        return new URL(trimmedEndpoint + "/" + trimmedUrl);
    }
}
