package org.airbloc.sdk.consent;

import android.content.Context;

import org.airbloc.sdk.AirblocConfig;
import org.airbloc.sdk.internal.AirblocLocalStorage;
import org.airbloc.sdk.internal.JsonData;
import org.airbloc.sdk.internal.networking.AirblocHttpClient;
import org.airbloc.sdk.internal.networking.AirblocHttpRequest;
import org.airbloc.sdk.internal.networking.AirblocHttpResponse;
import org.airbloc.sdk.user.User;

import java8.util.concurrent.CompletableFuture;

public class ConsentRepositoryImpl implements ConsentRepository {

    private static final String CONSENTS_KEY_PREFIX = "consents";

    private AirblocHttpClient httpClient;
    private AirblocLocalStorage storage;
    private AirblocConfig config;

    private Consents consents;

    public ConsentRepositoryImpl(Context context, AirblocHttpClient httpClient, AirblocConfig config) {
        this.httpClient = httpClient;
        this.storage = new AirblocLocalStorage(context);
        this.config = config;
    }

    @Override
    public CompletableFuture<Consents> get(User user) {
        if (consents == null) {
            Consents consentsFromLocal = retrieveFromLocal();
            if (consentsFromLocal == null) {
                String url = String.format(
                        "/consent/%s?account_id=%s", config.getAppName(), user.getAccountId());

                return httpClient.call(new AirblocHttpRequest("GET", url))
                        .thenApply(AirblocHttpResponse::getBody)
                        .thenApply(body -> (JsonData) body.get("consents"))
                        .thenApply(Consents::fromJson)
                        .exceptionally(err -> null);
            }
            consents = consentsFromLocal;
        }
        return CompletableFuture.supplyAsync(() -> consents);
    }

    @Override
    public void consent(String accountId, Consents consents) {
        JsonData body = new JsonData()
                .put("account_id", accountId)
                .put("consents", consents.toJson());

        String url = "/consent/" + config.getAppName() + "/";
        httpClient.call(new AirblocHttpRequest("PUT", url)
                .setBody(body));

        storeToLocal(consents);
        this.consents = consents;
    }

    private Consents retrieveFromLocal() {
        JsonData localConsentData = storage.get(CONSENTS_KEY_PREFIX);
        if (localConsentData == null) {
            return null;
        }
        return Consents.fromJson(localConsentData);
    }

    private void storeToLocal(Consents consents) {
        storage.put(CONSENTS_KEY_PREFIX, consents.toJson());
    }
}
