package org.airbloc.sdk;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.airbloc.sdk.appinfo.AppInfo;
import org.airbloc.sdk.appinfo.AppInfoRepository;
import org.airbloc.sdk.appinfo.AppInfoRepositoryImpl;
import org.airbloc.sdk.consent.ConsentManager;
import org.airbloc.sdk.datacollector.DataCollector;
import org.airbloc.sdk.event.Event;
import org.airbloc.sdk.internal.AirblocExecutors;
import org.airbloc.sdk.internal.logger.AirblocLogger;
import org.airbloc.sdk.internal.networking.AirblocHttpClient;
import org.airbloc.sdk.user.UserManager;

import java8.util.concurrent.CompletableFuture;

public class Airbloc {
    // Airbloc singleton instance
    private static Airbloc airbloc;

    private AirblocConfig config;
    private AirblocHttpClient httpClient;
    private AirblocExecutors executors;

    // a future of asynchronously initialized resources
    private CompletableFuture<Void> initializerTask;

    private UserManager userManager;
    private ConsentManager consentManager;
    private AppInfoRepository appInfoRepository;

    public static void configure(Context context, @Nullable AirblocConfig config) {
        if (config == null) {
            config = new AirblocConfig.Builder().build();
        }
        airbloc = new Airbloc(context, config);
    }

    public static Airbloc getInstance() {
        return airbloc;
    }

    Airbloc(Context context, @NonNull AirblocConfig config) {
        context = context.getApplicationContext();
        this.config = overrideConfig(context, config);

        this.executors = AirblocExecutors.getInstance();
        this.httpClient = new AirblocHttpClient(this.config.endpoint, this.config.appToken);
        this.userManager = new UserManager(context, httpClient);
        this.consentManager = new ConsentManager(context, userManager, httpClient, this.config);
        this.appInfoRepository = new AppInfoRepositoryImpl(context, httpClient, this.config);

        this.initializerTask = initialize(context);
    }

    private AirblocConfig overrideConfig(Context context, @NonNull AirblocConfig customConfig) {
        config = AirblocConfig.fromResources(context);
        customConfig.override(config);

        if (config.appName == null) {
            AirblocLogger.e("Error: You haven't registered App Name to your airbloc.xml. \n"
                    + "For details, please check our documentation.");
        }
        if (config.appToken == null) {
            AirblocLogger.e("Error: You haven't registered App Token to your airbloc.xml. \n"
                    + "For details, please check our documentation.");
        }
        return config;
    }

    /**
     * Initialize resources asynchronously.
     */
    private CompletableFuture<Void> initialize(Context context) {
        AirblocLogger.i("Airbloc Consent SDK %s (debug=%s)",
                SdkConstants.VERSION, String.valueOf(SdkConstants.IS_DEBUG_BUILD));

        // a. load saved user session if exists
        CompletableFuture fetchUser = userManager.load();

        // simultaneously run'em all
        CompletableFuture<Void> initializationFuture = CompletableFuture.allOf(
                fetchUser
                // TODO: add more tasks here
        );

        return initializationFuture
                .thenAcceptAsync(v -> AirblocLogger.d("Initialization complete."), executors.getUiExecutor())
                .exceptionally(AirblocLogger.logError("Failed to initialize"));
    }

    public boolean isInitialized() {
        return initializerTask.isDone();
    }

    public UserManager getCurrentUser() {
        return userManager;
    }

    public void signIn(String userId) {
        userManager.signIn(userId);
    }

    public ConsentManager getConsents() {
        return consentManager;
    }

    /**
     * Sends an event.
     * @throws RuntimeException if you didn't set up any data collection extensions
     */
    public void sendEvent(Event event) {
        if (!userManager.hasSignedIn()) {
            AirblocLogger.i("Skip sending event because no user has been signed in.");
            return;
        }
        DataCollector dataCollector = Extensions.getInstance().getDataCollector();
        dataCollector.sendEvent(userManager.getUser(), event);
    }

    @NonNull
    public CompletableFuture<AppInfo> loadAppInfo() {
        return appInfoRepository.get();
    }
}
