package org.airbloc.sdk.internal.logger;

import android.util.Log;

import org.airbloc.sdk.SdkConstants;

import java8.util.function.Function;

public class AirblocLogger {
    // default log level is INFO (VERBOSE on debug SDK build)
    private static int currentLevel = Log.INFO;

    // usage: AirblocLogger.setLogLevel(AirblocLogger.SUPRESS);
    public static final int SUPRESS = 99;

    private static CrashReporter crashReporter = new CrashReporter();

    public static void setLogLevel(int level) {
        currentLevel = level;
    }

    public static void v(String format, Object ...args) {
        String message = safeFormat(format, args);

        crashReporter.record(new LogEntry("verbose", message));
        if (currentLevel <= Log.VERBOSE) {
            Log.v(SdkConstants.LOG_TAG, message);
        }
    }

    public static void d(String format, Object ...args) {
        String message = safeFormat(format, args);

        crashReporter.record(new LogEntry("debug", message));
        if (currentLevel <= Log.DEBUG) {
            Log.d(SdkConstants.LOG_TAG, message);
        }
    }

    public static void i(String format, Object ...args) {
        String message = safeFormat(format, args);

        crashReporter.record(new LogEntry("info", message));
        if (currentLevel <= Log.INFO) {
            Log.i(SdkConstants.LOG_TAG, message);
        }
    }

    public static void w(String message) {
        crashReporter.record(new LogEntry("warning", message));
        if (currentLevel <= Log.WARN) {
            Log.w(SdkConstants.LOG_TAG, message);
        }
    }

    public static void w(String message, Throwable error) {
        crashReporter.record(new LogEntry("warning", message)
                .setErrorInfo(error));

        if (currentLevel <= Log.WARN) {
            Log.w(SdkConstants.LOG_TAG, "Warning: " + message, error);
        }
    }

    public static void e(String message, Throwable error) {
        crashReporter.record(new LogEntry("error", message)
                .setErrorInfo(error));

        if (currentLevel <= Log.ERROR) {
            Log.e(SdkConstants.LOG_TAG, message, error);
        }
    }

    public static void e(String message) {
        crashReporter.record(new LogEntry("error", message));
        if (currentLevel <= Log.ERROR) {
            Log.e(SdkConstants.LOG_TAG, message);
        }
    }

    public static Function<Throwable, Void> logError(String message) {
        return error -> {
            e(message, error);
            return null;
        };
    }

    /**
     * 이슈가 될 수 있는 중대한 에러를 로깅하고, 서버에 리포트한다.
     * 중대한 에러가 아니라면 {@link AirblocLogger#e} 혹은 {@link AirblocLogger#w}의 사용을 권고한다.
     * @param message 에러 메세지
     * @param error 에러
     */
    public static void wtf(String message, Throwable error) {
        crashReporter.record(new LogEntry("critical", message).setErrorInfo(error));
        if (currentLevel <= Log.ERROR) {
            Log.e(SdkConstants.LOG_TAG, message, error);
        }
        try {
            crashReporter.reportToServer();
        } catch (Throwable e) {
            AirblocLogger.e("Error occurred during reporting logs to server. ", e);
        }
    }

    public static String getFullLogs(){
        return crashReporter.getFullLogs();
    }

    private static String safeFormat(String format, Object ...args) {
        try {
            return String.format(format, args);

        } catch (Throwable e) {
            return String.format(format, args) + " [" + e.getMessage() + "]";
        }
    }
}
