package org.airbloc.sdk.internal;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class JsonData implements JsonMappable {
    private Map<String, Object> data = new HashMap<>();

    public static JsonData fromJSON(JSONObject object) {
        return (JsonData) parse(object);
    }

    public JsonData() {}

    public JsonData(Map<String, ?> data) {
        this.data = new HashMap<>(data);
    }

    private static Object parse(Object object) {
        if (object instanceof JSONObject) {
            JSONObject json = (JSONObject) object;
            JsonData result = new JsonData();

            Iterator<String> keys = json.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                result.put(key, parse(json.opt(key)));
            }
            return result;

        } else if (object instanceof JSONArray) {
            JSONArray array = (JSONArray) object;
            List<Object> result = new ArrayList<>(array.length());

            for (int i = 0; i < array.length(); i++) {
                result.add(parse(array.opt(i)));
            }
            return result;
        }
        return object;
    }

    public JsonData put(String key, Object value) {
        data.put(key, value);
        return this;
    }

    public JsonData put(String key, JsonMappable value) {
        data.put(key, value.toJson());
        return this;
    }

    /**
     * 주어진 값이 {@code null}이라면 해당 필드를 넣지 않는다.
     */
    public JsonMappable maybePut(String key, @Nullable Object value) {
        if (value != null) data.put(key, value);
        return this;
    }

    /**
     * 주어진 값이 {@code null}이라면 해당 필드를 넣지 않는다.
     */
    public JsonData maybePut(String key, @Nullable JsonMappable value) {
        if (value != null) data.put(key, value.toJson());
        return this;
    }

    @Nullable
    @SuppressWarnings("unchecked")
    public <T> T get(String key) {
        return (T) data.get(key);
    }

    @Nullable
    public String getString(String key) {
        Object value = data.get(key);
        return value != null ? value.toString() : null;
    }

    @NonNull
    public Long getLong(String key) {
        Object value = data.get(key);
        if (value instanceof Long) {
            return (Long) value;

        } else if (value instanceof Number) {
            return ((Number) value).longValue();

        } else if (value instanceof String) {
            try {
                return (long) Double.parseDouble((String) value);
            } catch (NumberFormatException ignored) {}
        }
        throw new ClassCastException(key + " is not long.");
    }

    @NonNull
    public Integer getInt(String key) {
        Object value = data.get(key);
        if (value instanceof Integer) {
            return (Integer) value;

        } else if (value instanceof Number) {
            return ((Number) value).intValue();

        } else if (value instanceof String) {
            try {
                return (int) Double.parseDouble((String) value);
            } catch (NumberFormatException ignored) {}
        }
        throw new ClassCastException(key + " is not int.");
    }

    public boolean has(String key) {
        return data.containsKey(key);
    }

    public boolean isEmpty() {
        return data.isEmpty();
    }

    public <T> void copyToMap(Map<String, T> to) {
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            to.put(entry.getKey(), (T) entry.getValue());
        }
    }

    /**
     * 파라미터들을 JSON으로 변환한다.
     * @return {@link JSONObject}
     */
    public JSONObject toJsonObject() {
        JSONObject json = new JSONObject();
        for (String key : data.keySet()) {
            Object value = get(key);
            try {
                json.put(key, doJsonConvert(value));
            } catch (JSONException ignored) {}
        }
        return json;
    }

    private Object doJsonConvert(Object value) {
        if (value instanceof JsonMappable) {
            JsonMappable mappable = (JsonMappable) value;
            return mappable.toJson().toJsonObject();
        }
        if (value instanceof Collection) {
            Collection values = (Collection) value;
            JSONArray array = new JSONArray();
            for (Object item : values) {
                array.put(doJsonConvert(item));
            }
            return array;
        }
        if (value instanceof Map) {
            return new JSONObject((Map) value);
        }
        // primitives
        return value;
    }

    @Override
    public String toString() {
        return this.toJsonObject().toString();
    }

    @Override
    public JsonData toJson() {
        return this;
    }
}
