/*
 * Copyright (c) 2002-2003 by OpenSymphony
 * All rights reserved.
 */
package com.opensymphony.xwork.interceptor;

import com.opensymphony.xwork.ActionContext;
import com.opensymphony.xwork.ActionInvocation;
import com.opensymphony.xwork.ValidationAware;
import com.opensymphony.xwork.util.XWorkConverter;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;


/**
 * A simple {@link ConversionErrorInterceptor} that sets the value of the field directly back in the context rather than
 * in the stack, which get evaluated as an OGNL expression. Workaround for:
 * https://www.sec-consult.com/files/20120104-0_Apache_Struts2_Multiple_Critical_Vulnerabilities.txt
 *
 */
public class ConversionErrorInterceptor implements Interceptor {
    //~ Methods ////////////////////////////////////////////////////////////////

    public void destroy() {
    }

    public void init() {
    }

    public String intercept(final ActionInvocation invocation) throws Exception {
        ActionContext invocationContext = invocation.getInvocationContext();
        final Map conversionErrors = invocationContext.getConversionErrors();

        ValidationAware action = null;

        if (invocation.getAction() instanceof ValidationAware) {
            action = (ValidationAware) invocation.getAction();
        }

        final Map filteredConversionErrors = new HashMap();

        for (Iterator it = conversionErrors.entrySet().iterator();
                it.hasNext();) {
            Map.Entry entry = (Map.Entry) it.next();
            String propertyName = (String) entry.getKey();
            String value = getStringValue(entry.getValue());

            if ((value != null) && !value.isEmpty()) {
                filteredConversionErrors.put(propertyName, value);

                if (action != null) {
                    String message = XWorkConverter.getConversionErrorMessage(propertyName, invocationContext.getValueStack());
                    action.addFieldError(propertyName, message);
                }
            }
        }

        // Add the value back to the page context, doesn't go through all the OGNL malarky. Form fields will see the value in the action first though
        invocation.addPreResultListener(new PreResultListener() {
                public void beforeResult(ActionInvocation invocation, String resultCode) {
                    if ((invocation.getStack() != null) && (invocation.getStack().getContext() != null)) {
                        Map map = invocation.getStack().getContext();

                        map.putAll(filteredConversionErrors);
                    }
                }
            });

        return invocation.invoke();
    }

    private static String getStringValue(final Object o) {
        if (o instanceof String[]) {
            String[] strings = (String[]) o;

            if (strings.length > 0) {
                return strings[0];
            } else {
                return null;
            }
        } else {
            return (o != null) ? o.toString() : "";
        }
    }
}
