package com.opensymphony.webwork.views.util;

import com.google.common.collect.ImmutableSet;
import com.opensymphony.webwork.components.Component;
import com.opensymphony.webwork.components.Form;
import com.opensymphony.xwork.util.OgnlUtil;
import org.apache.commons.logging.Log;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

public class SecurityUtil
{
    public static final String OGNL_UNSAFE = "OgnlUnsafe";

    private SecurityUtil()
    {
    }

    public static Map<String, Object> getUnescapedParameters(final Log log, final Component bean, final Map<String, Object> parameters)
    {
        if (parameters==null)
        {
            return null;
        }

        //Unit Tests rely on order, no point in fixing them...
        final Map<String, Object> unescapedParameters = (parameters instanceof HashMap)? new HashMap<String, Object>() : new TreeMap<String, Object>();

        //now, we bring back the original, unescaped values, unless the tag bean has already
        //modified the escaped value
        for (Map.Entry<String, Object> keyValue : parameters.entrySet())
        {
            final String key = keyValue.getKey();
            if (key.endsWith(OGNL_UNSAFE))
            {
                //just drop it.
                continue;
            }

            final Object escapedOrOveriddenValue = keyValue.getValue();
            final Object originalValueBeforeEscaping = getUnescapedValue(parameters, key); //this is the original, not escaped value

            if (originalValueBeforeEscaping instanceof String)
            {
                final String reescapedValue = wrapAsOgnlExpression(OgnlUtil.escapeLiteral((String) originalValueBeforeEscaping).toString());
                if (reescapedValue.equals(escapedOrOveriddenValue))
                {
                    unescapedParameters.put(key, originalValueBeforeEscaping);
                }
                else
                {
                    unescapedParameters.put(key, escapedOrOveriddenValue);
                    boolean isPlainOgnlEvaluation = originalValueBeforeEscaping.equals(escapedOrOveriddenValue);
                    if (log!=null && !isPlainOgnlEvaluation)
                    {
                        boolean knownOverridingTag = bean instanceof Form; //Form can just do whatever it likes
                        if (!knownOverridingTag)
                        {
                            String beanName = bean!=null ? bean.getClass().getCanonicalName() : "unknown";
                            log.debug(key + " was overridden from " + reescapedValue + " to " + escapedOrOveriddenValue + " by " + beanName);
                        }
                    }
                }
            }
            else
            {
                unescapedParameters.put(key, escapedOrOveriddenValue);
            }
        }
        return unescapedParameters;
    }

    public static String wrapAsOgnlExpression(final String expr)
    {
        return "%{" + expr + "}";
    }

    public static boolean hasUnescapedValue(final Map parameters, final String name)
    {
        return parameters.containsKey(getNameOfUnEscapedParameter(name));
    }

    private static Object getUnescapedValue(final Map<String, Object> parameters, final String name)
    {
        return parameters.get(getNameOfUnEscapedParameter(name));
    }

    private static String getNameOfUnEscapedParameter(final String name)
    {
        return name + OGNL_UNSAFE;
    }

}
