/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.multipart;

import org.apache.commons.logging.LogFactory;
import webwork.config.Configuration;
import http.utils.multipartrequest.ServletMultipartRequest;
import http.utils.multipartrequest.MultipartRequest;

import javax.servlet.http.HttpServletRequest;
import java.io.*;
import java.util.Enumeration;
import java.util.ArrayList;
import java.util.List;

/* ------------------------------------------------------------ */

/**
 * Multipart Form Data request adapter for Jason Pell's multipart utils
 *
 * @author Matt Baldree (matt@smallleap.com) modified for WW's use
 * @author Scott Farquhar (scott@atlassian.com) added i18n handling (WW-109)
 * @version $Id: PellMultiPartRequest.java,v 1.6 2005/03/19 19:20:21 fate Exp $
 */
public class PellMultiPartRequest extends MultiPartRequest
{

    /* ------------------------------------------------------------ */
    private ServletMultipartRequest multi;

    /**
     * Set the encoding for the uploaded parameters.  This needs to be set if you are using character sets other than
     * ASCII.
     * <p/>
     * The encoding is looked up from the configuration setting 'webwork.i18n.encoding'.  This is usually set in
     * default.properties & webwork.properties.
     */
    private String getEncoding()
    {
        try
        {
            String encoding = Configuration.getString("webwork.i18n.encoding");
            if (encoding != null)
            {
                return encoding;
            }
        }
        catch (IllegalArgumentException e)
        {
            LogFactory.getLog(PellMultiPartRequest.class).info("Could not get encoding property 'webwork.i18n.encoding' for file upload.  Using system default");
        }
        return null;
    }

    /**
     * @param maxSize        maximum size post allowed
     * @param saveDir        the directory to save off the file
     * @param servletRequest the request containing the multipart
     */
    public PellMultiPartRequest(HttpServletRequest servletRequest, String saveDir, int maxSize) throws IOException
    {
        boolean memoryUpload = false;
        try
        {
            memoryUpload = "true".equalsIgnoreCase(Configuration.getString("webwork.multipart.uploadToMemory"));
        }
        catch (IllegalArgumentException e)
        {
            LogFactory.getLog(PellMultiPartRequest.class).info("Could not read webwork.multipart.uploadToMemory property for file upload. Defaulting to false");
        }
        if (memoryUpload)
        {
            multi = new ServletMultipartRequest(servletRequest, maxSize, MultipartRequest.IGNORE_FILES_IF_MAX_BYES_EXCEEDED, getEncoding());
        }
        else
        {
            multi = new ServletMultipartRequest(servletRequest, saveDir, maxSize, MultipartRequest.IGNORE_FILES_IF_MAX_BYES_EXCEEDED, getEncoding());
        }
    }

    public Enumeration getParameterNames()
    {
        return multi.getParameterNames();
    }

    public String getParameter(String name)
    {
        return multi.getURLParameter(name);
    }

    public String[] getParameterValues(String name)
    {
        Enumeration e = multi.getURLParameters(name);
        if (!e.hasMoreElements())
        {
            return null;
        }

        List values = new ArrayList();
        while (e.hasMoreElements())
        {
            values.add(e.nextElement());
        }
        return (String[]) values.toArray(new String[values.size()]);
    }

    // Methods only in MultipartRequest
    public Enumeration getFileNames()
    {
        return multi.getFileParameterNames();
    }

    public String getFilesystemName(String name)
    {
        return multi.getBaseFilename(name);
    }

    public String getContentType(String name)
    {
        return multi.getContentType(name);
    }

    public File getFile(String name)
    {
        return multi.getFile(name);
    }
    public InputStream getMemoryFileContents(String name)
    {
        return multi.getFileContents(name);
    }
}
