/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.multipart;

import org.apache.commons.logging.*;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Enumeration;

/* ------------------------------------------------------------ */

/**
 * Multipart Form Data request.
 * <p/>
 * This class decodes the multipart/form-data stream sent by a HTML form that uses a file input item.
 *
 * @author Matt Baldree (matt@smallleap.com) modified for WW's use
 * @author Hani Suleiman - Added javadocs
 * @version $Id: MultiPartRequest.java,v 1.4 2003/05/08 00:28:12 fate Exp $
 */
public abstract class MultiPartRequest {
    protected static Log log = LogFactory.getLog(MultiPartRequest.class);

    /**
     * Is request a multipart request
     */
    public static boolean isMultiPart(HttpServletRequest request) {
        String content_type = request.getHeader("Content-Type");
        return content_type != null && content_type.startsWith("multipart/form-data");
    }

    public abstract Enumeration getParameterNames();

    public abstract String getParameter(String name);

    public abstract String[] getParameterValues(String name);

    // Methods only in MultipartRequest
    public abstract Enumeration getFileNames();

    /**
     * Get the client file name of the specified uploaded file.
     *
     * @param name the parameter name of the uploaded file
     *
     * @return The client filename
     */
    public abstract String getFilesystemName(String name);

    public abstract String getContentType(String name);

    /**
     * Get the File as saved by the multipart parser on the server filesystem.
     *
     * @param name The parameter name of the uploaded file
     *
     * @return the File on the local filesystem, or null if the multipart parser has not stored the file on disk (for
     *         example, if it is a memory based file upload)
     */
    public abstract File getFile(String name);

    /**
     * Deletes all the temporary files; Intentionally not abstract to preserve backwards compatibility
     * Not abstract to preserve backwards compatibility. Please override in subclass if you want to
     * support the cleanup after request.
     *
     */
    public void cleanup() {
        throw new UnsupportedOperationException("The cleanup was not implemented in this class.");
    }
}
