package webwork.config.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * An implementation of {@link webwork.config.util.ActionInfo}
 *
 * @since v4.0
 */
public class ActionInfoImpl implements ActionInfo
{
    private final String actionName;
    private final String actionAlias;
    private final Map<String, String> attributes;
    private final List<ViewInfo> viewInfos;
    private final String source;

    ActionInfoImpl(final String actionName, final String actionAlias, Map<String, String> attributeMap, final String source, final List<ViewInfo> viewInfos)
    {
        this.actionName = actionName;
        this.actionAlias = actionAlias;
        this.attributes = attributeMap;
        this.source = source;
        this.viewInfos = viewInfos;
    }

    public String getActionName()
    {
        return actionName;
    }

    public String getActionAlias()
    {
        return actionAlias;
    }

    public String getSource()
    {
        return source;
    }

    public Set<String> getAttributeNames()
    {
        return attributes.keySet();
    }

    public String getAttributeValue(final String attributeName)
    {
        return attributes.get(attributeName);
    }

    public Map<String, String> getAttributes()
    {
        return new HashMap<String, String>(attributes);
    }

    public Iterable<ViewInfo> getViews()
    {
        return viewInfos;
    }

    static class ViewInfoImpl implements ViewInfo
    {
        private final String viewName;
        private final String viewValue;
        private final Map<String, String> attributes;
        private ActionInfo actionInfo;

        ViewInfoImpl(final String viewName, final String viewValue, final Map<String, String> attributes)
        {
            this.viewName = viewName;
            this.viewValue = viewValue;
            this.attributes = attributes;
        }

        public ActionInfo getActionInfo()
        {
            return actionInfo;
        }

        public String getViewName()
        {
            return viewName;
        }

        public String getViewValue()
        {
            return viewValue;
        }

        public Set<String> getAttributeNames()
        {
            return attributes.keySet();
        }

        public String getAttributeValue(final String attributeName)
        {
            return attributes.get(attributeName);
        }

        public Map<String, String> getAttributes()
        {
            return Collections.unmodifiableMap(attributes);
        }

        void setActionInfo(ActionInfo actionInfo)
        {
            this.actionInfo = actionInfo;
        }

    }

    public static Builder builder(String actionName, String actionAlias)
    {
        return new Builder(actionName, actionAlias);
    }

    public static class Builder
    {

        public class View
        {
            private String viewName;
            private String viewValue;
            private Map<String, String> attributes = new HashMap<String, String>();

            View(String viewName, String viewValue)
            {
                this.viewName = viewName;
                this.viewValue = viewValue;
            }

            public View addAttributes(Map<String, String> attributes)
            {
                this.attributes.putAll(attributes);
                return this;
            }

            public View addAttribute(String attrName, String attrValue)
            {
                this.attributes.put(attrName, attrValue);
                return this;
            }

            public Builder endView()
            {
                Builder.this.viewInfos.add(new ViewInfoImpl(viewName, viewValue, attributes));
                return Builder.this;
            }
        }

        private String actionName;
        private String actionAlias;
        private String source;
        private Map<String, String> attributes = new HashMap<String, String>();
        private List<ViewInfo> viewInfos = new ArrayList<ViewInfo>();

        Builder(final String actionName, final String actionAlias)
        {
            this.actionName = actionName;
            this.actionAlias = actionAlias;
        }

        public Builder setSource(final String sourceOfConfig)
        {
            this.source = sourceOfConfig;
            return this;
        }

        public Builder addAttributes(Map<String, String> attributes)
        {
            this.attributes.putAll(attributes);
            return this;
        }

        public Builder addAttribute(String attrName, String attrValue)
        {
            this.attributes.put(attrName, attrValue);
            return this;
        }

        public View startView(String viewName, String viewValue)
        {
            return new View(viewName, viewValue);
        }

        /**
         * This can be called to merge a previous mergeSource object into the new 'to be built' action info object
         *
         * @param mergeSource the action info to merge from.  This can be null in which case nothing happens
         * @return the builder
         */
        public Builder merge(ActionInfo mergeSource)
        {
            if (mergeSource != null)
            {
                // only add attributes we don't already have values for
                {
                    for (Map.Entry<String, String> entry : mergeSource.getAttributes().entrySet())
                    {
                        if (!attributes.containsKey(entry.getKey()))
                        {
                            attributes.put(entry.getKey(), entry.getValue());
                        }
                    }
                }
                for (ViewInfo viewInfo : mergeSource.getViews())
                {
                    viewInfos.add(viewInfo);
                }
            }

            return this;
        }

        public ActionInfo build()
        {
            ActionInfoImpl actionInfo = new ActionInfoImpl(actionName, actionAlias, attributes, source, viewInfos);
            //
            // fix up back pointer to parent - I know I know but its immutable from the outsiders point of view
            for (ViewInfo viewInfo : viewInfos)
            {
                ((ViewInfoImpl) viewInfo).setActionInfo(actionInfo);
            }
            return actionInfo;
        }
    }
}
