/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action.factory;

import webwork.action.*;
import webwork.util.BeanUtil;

import java.util.Map;

/**
 * Sets properties on an action from parameters after locating it through the action factory proxy chain.
 *
 * @author Rickard \u00D6berg (rickard@middleware-company.com)
 * @version $Revision: 1.10 $
 */
public class ParametersActionFactoryProxy extends ActionFactoryProxy
{
    // Attributes ----------------------------------------------------

    // Constructors --------------------------------------------------

    public ParametersActionFactoryProxy(ActionFactory aFactory)
    {
        super(aFactory);
    }

    // ActionFactory implementation -------------------------------
    /**
     * Sets properties on the obtained action from the parameters. If an IllegalArgumentException is thrown, a
     * ResultException will be thrown with "error" as the result. This means that if a view is mapped to "error" then
     * the user will be returned and the field causing the error will be highlighted.
     *
     * @see BeanUtil#setProperties(Map, Object)
     */
    public Action getActionImpl(String aName)
            throws Exception
    {
        // Get action from factory
        Action action = getNextFactory().getActionImpl(aName);
        // Check if the NoParameters interface is implemented. In that
        // case no parameters should be set so we just return the action
        if (action instanceof NoParameters)
        {
            return action;
        }
        // Set parameters
        try
        {
            BeanUtil.setProperties(ActionContext.getParameters(), action);
        }
        catch (IllegalArgumentException e)
        {
            throw new ResultException(Action.ERROR);
        }

        // Return action
        return action;
    }

}
