/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.action.factory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import webwork.action.Action;
import webwork.action.ActionContext;
import webwork.config.Configuration;
import webwork.util.BeanUtil;

import java.util.Iterator;
import java.util.List;


/**
 * Action Factory proxy to handle action chaining. Adds actions performed to the action chain and copies properties from
 * each action to the next.
 *
 * @author Maurice C. Parker (maurice@vineyardenterprise.com)
 * @version $Revision: 2 $
 */
public class ChainingActionFactoryProxy extends ActionFactoryProxy
{
    // Attributes ----------------------------------------------------
    protected static Log log = LogFactory.getLog(ChainingActionFactoryProxy.class);
    // Default max linked actions is 10
    protected int maxLinks = 10;

    // Constructors --------------------------------------------------
    public ChainingActionFactoryProxy(ActionFactory aFactory)
    {
        super(aFactory);
        try
        {
            maxLinks = Integer.parseInt(Configuration.getString("webwork.action.chain.maxlinks"));
        }
        catch (IllegalArgumentException e)
        {
            log.warn("Unable to retrieve maximum chain action property. Defaulting to 10.");
        }
    }

    // ActionFactory implementation -------------------------------
    public Action getActionImpl(String aName)
            throws Exception
    {
        // Get action from factory
        Action action = getNextFactory().getActionImpl(aName);
        ActionContext context = ActionContext.getContext();

        List actions = (List) context.get("action.chain");

        if (actions != null)
        {
            if ((actions.size() + 1) > maxLinks)
            {
                throw new IllegalStateException("Exceeded maximum number of chained actions = " + maxLinks);
            }
            for (Iterator iter = actions.iterator(); iter.hasNext();)
            {
                BeanUtil.copy(iter.next(), action);
            }
            actions.add(action);
        }

        // Return action
        return action;

    }
}
